package com.netcore.android.db

import android.content.ContentValues
import android.database.Cursor
import androidx.annotation.VisibleForTesting
import com.netcore.android.SMTEventParamKeys
import com.netcore.android.inapp.SMTInAppRuleParser
import com.netcore.android.inapp.SMTInAppUtility
import com.netcore.android.inapp.model.SMTInAppRule
import com.netcore.android.logger.SMTLogger
import java.util.*

@Suppress("PrivatePropertyName")
/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * InApp Rules DB Table, All inapp  operations has handled here
 *
 * @see SMTDataBaseWrapper for method definitions
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */
internal class SMTInAppRulesTable(private val wrapper: SMTDataBaseWrapper) : SMTDBTable(wrapper) {

    // InApp Table Column names
    private val KEY_RULE_ID = "rule_id"   // Column I (Primary Key)
    private val KEY_EVENT_NAME = "event_name" //Column II (Event name for which the rule is defined)
    private val KEY_RULE_PAYLOAD = "payload" //Column III (JSON payload)
    private val KEY_MODIFIED_DATE = "modified_date" //Column IV (On which date rule is modified)
    private val KEY_ALREADY_VIEWED_COUNT = "already_viewed_count" //Column V (to keep track of frequency)
    private val KEY_EVENT_ID = "event_id"  // Column VI (ID of the event for which rule is defined)
    private val KEY_FROM_DATE = "form_date"  // Column VII (from date of the event )
    private val KEY_TO_DATE = "to_date"  // Column VIII (to date of the event)
    private val KEY_FREQUENCY_TYPE = "frequency_type"  // Column IX (Frequency type (day, campaign or session))
    private val KEY_FREQUENCY_TYPE_VALUE = "frequency_type_value"  // Column X (Frequency type value will timestamp/date)
    private val KEY_MAX_FREQUENCY = "max_frequency"  // Column XI (max number times to show)
    private val KEY_RANDOM_NUMBER = "random_number" //A number between 0-100 to validate the controlling group

    companion object {
        internal val IN_APP_TABLE = "InAppRule"
    }

    private val TAG = SMTInAppRulesTable::class.java.simpleName

    override fun createTable() {
        executeQuery(getInAppRuleTableCreateStatement())
    }

    /**
     * prepares the In App rules table creation query statement
     */
    private fun getInAppRuleTableCreateStatement(): String {
        return ("CREATE TABLE IF NOT EXISTS " + IN_APP_TABLE
                + " ( "
                + KEY_RULE_ID + " TEXT PRIMARY KEY, " // RULE ID
                + KEY_EVENT_NAME + " TEXT NOT NULL, " // Name of the event
                + KEY_RULE_PAYLOAD + " TEXT NOT NULL, " // Rule payload
                + KEY_MODIFIED_DATE + " TEXT , " // Time of the rule on which it is modified
                + KEY_ALREADY_VIEWED_COUNT + " INTEGER NOT NULL DEFAULT 0, " // View Frequency
                + KEY_EVENT_ID + " TEXT, " // ID of event for which rule is defined
                + KEY_FROM_DATE + " LONG,"
                + KEY_TO_DATE + " LONG,"
                + KEY_FREQUENCY_TYPE + " TEXT, "
                + KEY_FREQUENCY_TYPE_VALUE + " TEXT, "
                + KEY_RANDOM_NUMBER + " INTEGER, "
                + KEY_MAX_FREQUENCY + " INTEGER"
                + " ) ")
    }

    override fun upgradeTable(oldVersion: Int, newVersion: Int) {
        // if needs to do any action for updating the table

    }

    private fun insert(inAppRule: SMTInAppRule) {
        val contentValues = ContentValues()
        contentValues.put(KEY_RULE_ID, inAppRule.id)
        contentValues.put(KEY_TO_DATE, inAppRule.toDate)
        contentValues.put(KEY_FROM_DATE, inAppRule.fromDate)
        contentValues.put(KEY_EVENT_ID, inAppRule.eventId)
        contentValues.put(KEY_EVENT_NAME, inAppRule.eventName.toLowerCase(Locale.getDefault()))
        contentValues.put(KEY_FREQUENCY_TYPE, inAppRule.frequencyType)
        contentValues.put(KEY_MAX_FREQUENCY, inAppRule.frequency)
        contentValues.put(KEY_MODIFIED_DATE, inAppRule.modifiedDate)
        contentValues.put(KEY_RULE_PAYLOAD, inAppRule.payload)
        val cgRandomNumber = Random().nextInt(100)
        contentValues.put(KEY_RANDOM_NUMBER, cgRandomNumber)


        if (!isRuleExist(inAppRule)) {

            val id = wrapper.insert(IN_APP_TABLE, null, contentValues)
            if (id == ((-1).toLong())) {
                SMTLogger.e(TAG, "Rule insertion failed")
            }
        } else {

            if (isRuleModified(inAppRule)) {
                SMTLogger.d(TAG, "Rule modified")
                contentValues.put(KEY_ALREADY_VIEWED_COUNT, 0)
            }
            val id = wrapper.update(IN_APP_TABLE, contentValues, "$KEY_RULE_ID = ?", arrayOf(inAppRule.id))
            if (id == 0) {
                SMTLogger.e(TAG, "None of the rules got updated")
            }
        }


    }

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    fun isRuleModified(inAppRule: SMTInAppRule): Boolean {
        if (inAppRule.modifiedDate != null) {
            executeRawQuery("Select * from " + IN_APP_TABLE + " where " + KEY_RULE_ID + " = " + inAppRule.id + " AND " + KEY_MODIFIED_DATE + " != " + inAppRule.modifiedDate)?.let { return it.moveToFirst() }
        }
        return false

    }

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    fun isRuleExist(inAppRule: SMTInAppRule): Boolean {
        executeRawQuery("Select * from " + IN_APP_TABLE + " where " + KEY_RULE_ID + " = " + inAppRule.id)?.let { return it.moveToFirst() }
        return false
    }

    private fun getAllInAppRule(): MutableList<SMTInAppRule>? {

        val applicableRules: MutableList<SMTInAppRule> = mutableListOf()
        val currentTime = System.currentTimeMillis()
        val query = "select * from $IN_APP_TABLE where  $KEY_FROM_DATE <= $currentTime AND $KEY_TO_DATE >= $currentTime ORDER BY CAST($KEY_RULE_ID AS INT) ASC"

        val cursor = executeRawQuery(query)
        cursor?.let {
            if (it.moveToLast()) {
                do {
                    val inAppRule = SMTInAppRule()
                    inAppRule.id = cursor.getString(cursor.getColumnIndex(KEY_RULE_ID))
                    inAppRule.alreadyViewedCount = cursor.getInt(cursor.getColumnIndex(KEY_ALREADY_VIEWED_COUNT))
                    inAppRule.randomNumber = cursor.getInt(cursor.getColumnIndex(KEY_RANDOM_NUMBER))
                    inAppRule.eventId = cursor.getString(cursor.getColumnIndex(KEY_EVENT_ID))
                    inAppRule.eventName = cursor.getString(cursor.getColumnIndex(KEY_EVENT_NAME))
                    inAppRule.frequencyTypeValue = cursor.getLong(cursor.getColumnIndex(KEY_FREQUENCY_TYPE_VALUE))
                    inAppRule.toDate = cursor.getLong(cursor.getColumnIndex(KEY_TO_DATE)).toString()
                    inAppRule.fromDate = cursor.getLong(cursor.getColumnIndex(KEY_FROM_DATE)).toString()

                    setThePayloadData(cursor, inAppRule)

                    applicableRules.add(inAppRule)
                } while (cursor.moveToPrevious())
            }
        }
        //println("inapp : $query ${applicableRules.size}")
        cursor?.close()
        return applicableRules
    }

    internal fun getMatchingRules(payloadMap: HashMap<String, Any>): MutableList<SMTInAppRule>? {
        val applicableRules: MutableList<SMTInAppRule> = mutableListOf()
        val eventId: String? = payloadMap[SMTEventParamKeys.SMT_EVENT_ID]?.toString() ?: ""
        val eventName: String? = payloadMap[SMTEventParamKeys.SMT_EVENT_NAME]?.toString()?.toLowerCase()
                ?: ""
        val currentTime = System.currentTimeMillis()

        val query = if (eventId != null && eventId.trim().isNotEmpty() && eventId.toInt() > 0) {
            "select * from $IN_APP_TABLE where $KEY_EVENT_ID = $eventId AND $KEY_FROM_DATE <= $currentTime AND $KEY_TO_DATE >= $currentTime ORDER BY CAST($KEY_RULE_ID AS INT) ASC"
        } else if (!eventName.isNullOrEmpty()) {
            "select * from $IN_APP_TABLE where $KEY_EVENT_NAME = '$eventName' AND $KEY_FROM_DATE <= $currentTime AND $KEY_TO_DATE >= $currentTime ORDER BY CAST($KEY_RULE_ID AS INT) ASC"

        } else {
            return applicableRules
        }
        val cursor = executeRawQuery(query)
        cursor?.let {
            if (it.moveToLast()) {
                do {
                    val inAppRule = SMTInAppRule()
                    inAppRule.id = cursor.getString(cursor.getColumnIndex(KEY_RULE_ID))
                    inAppRule.alreadyViewedCount = cursor.getInt(cursor.getColumnIndex(KEY_ALREADY_VIEWED_COUNT))
                    inAppRule.randomNumber = cursor.getInt(cursor.getColumnIndex(KEY_RANDOM_NUMBER))
                    inAppRule.eventId = cursor.getString(cursor.getColumnIndex(KEY_EVENT_ID))
                    inAppRule.eventName = cursor.getString(cursor.getColumnIndex(KEY_EVENT_NAME))
                    inAppRule.frequencyTypeValue = cursor.getLong(cursor.getColumnIndex(KEY_FREQUENCY_TYPE_VALUE))
                    inAppRule.toDate = cursor.getLong(cursor.getColumnIndex(KEY_TO_DATE)).toString()
                    inAppRule.fromDate = cursor.getLong(cursor.getColumnIndex(KEY_FROM_DATE)).toString()

                    setThePayloadData(cursor, inAppRule)

                    applicableRules.add(inAppRule)
                } while (cursor.moveToPrevious())
            }
        }
        //println("inapp : $query ${applicableRules.size}")
        cursor?.close()
        return applicableRules
    }

    /**
     * Method to set the stored rule json string to the InAppRule model class
     */
    private fun setThePayloadData(cursor: Cursor, inAppRule: SMTInAppRule) {

        val payload = cursor.getString(cursor.getColumnIndex(KEY_RULE_PAYLOAD))
        SMTInAppRuleParser().parseStoredRule(payload, inAppRule)

    }

    /**
     * Method updates the table with the viewed count. If the requency type is day then resetting the frequency type value
     * // Frequency type value -  it is a timestamp to identify the number of viewed count for that particular day
     */
    internal fun updateInAppUsage(inAppRule: SMTInAppRule, currentDateTimeInMilli: Long) {
        val contentValues = ContentValues()
        contentValues.put(KEY_ALREADY_VIEWED_COUNT, inAppRule.alreadyViewedCount + 1)
        if (inAppRule.frequencyType == SMTInAppUtility.FREQUENCY_TYPE_DAY) {
            if (inAppRule.frequencyTypeValue != currentDateTimeInMilli) {
                //stored date and the current date are different so reset the view count for the new date
                contentValues.put(KEY_ALREADY_VIEWED_COUNT, 1)
            }
            contentValues.put(KEY_FREQUENCY_TYPE_VALUE, currentDateTimeInMilli)
        }
        wrapper.update(IN_APP_TABLE, contentValues, "$KEY_RULE_ID    = ?", arrayOf(inAppRule.id))
    }

    internal fun resetUsageForSessionTypeRule() {

        val list = getAllInAppRule()
        for (item in list.orEmpty()) {

            when (item.frequencyType) {
                "day" -> {
                    // Intentionally blank
                }
                "campaign" -> {
                    // Intentionally blank
                }
                else -> {
                    val contentValues = ContentValues()
                    contentValues.put(KEY_ALREADY_VIEWED_COUNT, 0)
                    wrapper.update(IN_APP_TABLE, contentValues, "$KEY_EVENT_ID == ${item.eventId}", null)
                }
            }
        }
    }

    /*private fun getAllRuleIds(): ArrayList<String> {
        val ids = arrayListOf<String>()
        val cursor = executeRawQuery("select $KEY_RULE_ID from $IN_APP_TABLE")
        if (cursor != null) {
            if (cursor.moveToFirst()) {
                do {
                    ids.add(cursor.getString(cursor.getColumnIndex(KEY_RULE_ID)))

                } while (cursor.moveToNext())
            }
        }
        return ids
    }*/


    /**
     * Method to delete the inapp rules which are not in the ruleIds. If ruleIds is null then delete all
     */
    internal fun deleteOtherInAppRules(ruleIds: String?) {
        if (ruleIds != null) {
            wrapper.delete(IN_APP_TABLE, "$KEY_RULE_ID NOT IN $ruleIds", null)
        } else {
            wrapper.delete(IN_APP_TABLE, null, null)
        }
    }

    internal fun insertInAppRules(inAppRules: ArrayList<SMTInAppRule>) {
        inAppRules.forEach {
            insert(it)
        }
    }

    /**
     * Update notification status in notification table
     * @param ruleId - Rule ID, which to be updated
     * @param columnName - columname which to be updated
     * @param columnValue - value to be updated
     */
    internal fun updateInApp(ruleId: String, columnName: String, columnValue: Int) {

        val values = ContentValues()


        values.put(columnName, columnValue)
        val selection = "$KEY_RULE_ID = ?"

        wrapper.update(IN_APP_TABLE, values, selection, arrayOf(ruleId))
    }
}