package com.netcore.android.mediadownloader

import android.content.Context
import com.netcore.android.logger.SMTLogger
import com.netcore.android.network.SMTThreadPoolManager
import com.netcore.android.notification.SMTNotificationType
import com.netcore.android.notification.models.SMTCarouselItemData
import com.netcore.android.notification.models.SMTNotificationData
import java.lang.ref.WeakReference


/**
 * @author Netcore
 * created on 13/02/2019
 * This is a singleton interfacing class for handling all media downloads of notification.
 */

internal class SMTMediaDownloadManager {

    val TAG: String = SMTMediaDownloadManager::class.java.simpleName

    /**
     * Internal Download statuses
     */
    enum class DownloadStatus(val value: Int) {
        PENDING(1),
        IN_PROGRESS(2),
        FAILED(3),
        COMPLETED(4)
    }

    /**
     * Notifcation Payload Holder
     */
    lateinit var notification: SMTNotificationData
    /**
     * Media Download status Listener
     */
    lateinit var mMediaDownloadListener: MediaDownloadListener
    /**
     * App Context
     */
    lateinit var mContext: Context

    /**
     * its the interfacing function which handles downloads
     * @param context - App context
     * @param notification - Notification payload
     * @param mediaDownloadListener - download status Call back
     */
    fun downloadMedia(context: Context, notification: SMTNotificationData, mediaDownloadListener: MediaDownloadListener) {
        this.notification = notification
        mMediaDownloadListener = mediaDownloadListener
        mContext = context

        handleDownloads()
    }


    /**
     * handle downloads of images / files
     * If it is carousel then CarouselDownloadAsyncTask
     * else SingleMediaDownloadAsyncTask get called
     */
    private fun handleDownloads() {
        SMTLogger.v(TAG, "handleDownloads get called")
        if (notification.mNotificationType == SMTNotificationType.CAROUSEL_PORTRAIT.type
                || notification.mNotificationType == SMTNotificationType.CAROUSEL_LANDSCAPE.type) {
            SMTLogger.v(TAG, "handleDownloads for carousel images")
            handleCarouselNotifications()
        } else {
            SMTLogger.v(TAG, "handleDownloads for singel file")
            handleSingleImageNotifications()
        }

    }

    /**
     * handles the download of carousel notification media files
     */
    private fun handleCarouselNotifications() {
        SMTLogger.d(TAG, "Handling carousel images download")
        if (!isAllImagesDownloaded()) {
            /**
             * If not all images downloaded from list then download the remaining
             * not considering retry of failed downloads
             */
            notification.mCarouselList?.let {
                for (carouselItem in it) {
                    if (carouselItem.mDownloadStatus == 0 || carouselItem.mDownloadStatus == DownloadStatus.PENDING.value) {
                        SMTLogger.d(TAG, "calling carousel download ${carouselItem.imgUrl}")
                        carouselItem.mDownloadStatus = DownloadStatus.IN_PROGRESS.value
                        /**
                         * start carousel item download
                         */
                        SMTCarouselDownloadAsyncTask(WeakReference(mContext), carouselItem, AsyncDownloadListener()).executeOnExecutor(SMTThreadPoolManager.getIntance())
                    }
                }
            }

        } else {
            /**
             * If at least one image is downloaded then send success else failed
             */
            if (isAllorAnySuccess()) {
                notification.mDownloadStatus = SMTMediaDownloadManager.DownloadStatus.COMPLETED.value

                removeUnFinishDownloadImage(notification.mCarouselList)
                SMTMediaPathUpdaterAsyncTask(WeakReference(mContext), notification).executeOnExecutor(SMTThreadPoolManager.getIntance())
                mMediaDownloadListener.onDownloadSuccess(notification)

            } else {
                removeUnFinishDownloadImage(notification.mCarouselList)
                mMediaDownloadListener.onDownloadFailed(notification)
            }
        }

    }

    /*
    * This method check if download status is not complete then remove from mCarouselList
    */
    private fun removeUnFinishDownloadImage(mCarouselList: ArrayList<SMTCarouselItemData>?) {
        mCarouselList?.let {
            val iterator: MutableIterator<SMTCarouselItemData> = it.iterator()
            while (iterator.hasNext()) {
                val carouselData: SMTCarouselItemData = iterator.next()
                if (carouselData.mDownloadStatus != DownloadStatus.COMPLETED.value) {
                    iterator.remove()
                }
            }
        }
    }


    /**
     * If all or any one image is downloaded in the carousel list then we consider it as success
     */
    private fun isAllorAnySuccess(): Boolean {
        SMTLogger.d(TAG, "Checking if at least one image download succeeded")
        var isSuccess = false
        notification.mCarouselList?.let {
            for (item in it) {
                if (item.mDownloadStatus == DownloadStatus.COMPLETED.value) {
                    isSuccess = true;
                    break
                }

            }
        }
        SMTLogger.d(TAG, "Is At least one image downloaded : $isSuccess")

        return isSuccess
    }

    /**
     * Method to check if all the images are downloaded or attempted to download
     * method returns false even if anything is in progress
     */
    private fun isAllImagesDownloaded(): Boolean {
        SMTLogger.d(TAG, "Checking all images downloaded")
        var isAllDone = true
        if (notification.mNotificationType != SMTNotificationType.CAROUSEL_LANDSCAPE.type
                && notification.mNotificationType != SMTNotificationType.CAROUSEL_PORTRAIT.type) {
            isAllDone = (notification.mDownloadStatus == DownloadStatus.COMPLETED.value || notification.mDownloadStatus == DownloadStatus.FAILED.value)
        } else {

            notification.mCarouselList?.let {
                for (item in it) {
                    // checks  if any of the request is in progress then return false
                    if (item.mDownloadStatus == 0 || item.mDownloadStatus == DownloadStatus.PENDING.value || item.mDownloadStatus == DownloadStatus.IN_PROGRESS.value) {
                        isAllDone = false
                        break
                    }
                }
            }
        }
        return isAllDone;
    }

    /**
     * Method to handle the notification with single image.
     * Assuming except Carousel all other notifications will have single media file
     */
    private fun handleSingleImageNotifications() {
        SMTLogger.d(TAG, "Handling notification for single media")
        if (!notification.mMediaUrl.isNullOrEmpty()) {
            notification.mDownloadStatus = DownloadStatus.IN_PROGRESS.value
            SMTSingleMediaDownloadAsyncTask(WeakReference(mContext), notification, AsyncDownloadListener()).executeOnExecutor(SMTThreadPoolManager.getIntance())
        } else {
            SMTLogger.d(TAG, "mediaurl is empty")
            notification.mDownloadStatus = DownloadStatus.FAILED.value
            mMediaDownloadListener.onDownloadFailed(notification)
        }
    }

    /**
     * Interface for  media download status callback
     */
    interface MediaDownloadListener {
        fun onDownloadSuccess(notification: SMTNotificationData)
        fun onDownloadFailed(notification: SMTNotificationData)
    }

    /**
     * Inteface for listening download status internally from the Async Tasks
     */
    inner class AsyncDownloadListener {
        fun onDownloadSuccess() {
            notification.mDownloadStatus = DownloadStatus.COMPLETED.value
            SMTMediaPathUpdaterAsyncTask(WeakReference(mContext), notification).executeOnExecutor(SMTThreadPoolManager.getIntance())
            mMediaDownloadListener.onDownloadSuccess(notification)

        }

        fun onCarouselDownloadSuccess(carouselItem: SMTCarouselItemData) {
            carouselItem.mDownloadStatus = DownloadStatus.COMPLETED.value
            handleCarouselNotifications()
        }

        fun onCarouselDownloadError(carouselItem: SMTCarouselItemData) {
            carouselItem.mDownloadStatus = DownloadStatus.FAILED.value
            handleCarouselNotifications()
        }

        fun onDownloadError() {
            notification.mDownloadStatus = DownloadStatus.FAILED.value
            mMediaDownloadListener.onDownloadFailed(notification)
        }
    }
}
