package com.netcore.android.workmgr

import android.content.Context
import androidx.work.WorkManager
import androidx.work.Worker
import androidx.work.WorkerParameters
import com.netcore.android.SMTConfigConstants
import com.netcore.android.SMTWorkManagerConst
import com.netcore.android.Smartech
import com.netcore.android.event.SMTEventCommonDataDump
import com.netcore.android.event.SMTNotificationSourceType
import com.netcore.android.logger.SMTLogger
import com.netcore.android.network.SMTApiService
import com.netcore.android.network.SMTEnumHttpMethodType
import com.netcore.android.network.SMTThreadPoolManager
import com.netcore.android.network.models.SMTPushAmpResponse
import com.netcore.android.network.models.SMTRequest
import com.netcore.android.network.models.SMTResponse
import com.netcore.android.notification.SMTNotificationGeneratorProvider
import com.netcore.android.notification.SMTPNHandler
import com.netcore.android.preference.SMTGUIDPreferenceHelper
import com.netcore.android.preference.SMTPreferenceConstants
import com.netcore.android.preference.SMTPreferenceHelper
import com.netcore.android.utility.SMTCommonUtility
import java.lang.ref.WeakReference
import java.util.concurrent.TimeUnit

internal class SMTPushAmpWorker(context: Context, params: WorkerParameters) : Worker(context, params) {

    private val TAG: String = SMTPushAmpWorker::class.java.simpleName

    override fun doWork(): Result {
        val context = applicationContext
        // if SDK is not active cancel PushAmp worker
        val arePushampConditionsSatisfied = SMTWorkerScheduler.getInstance().arePushampConditionsSatisfied(context)
        if (!arePushampConditionsSatisfied) {
            SMTLogger.i(TAG, "Pushamp is disabled : $arePushampConditionsSatisfied")
            SMTWorkerScheduler.getInstance().cancelPushAmp(context)
            return Result.failure()
        }

        SMTLogger.v(TAG, "PushAmp is running")
        val smtResponse = makePushAmpApiCall(context)

        if (!isStopped) {
            when (smtResponse != null && smtResponse.isSuccess) {
                true -> {
                    handlePushAmpApiSuccess(context, smtResponse)
                    Result.success()
                }
                false -> {
                    Result.failure()
                }
            }
        }
        return Result.success()
    }

    override fun onStopped() {
        super.onStopped()
        SMTLogger.v(TAG, "PushAmp worker is stopped.")
    }

    /**
     * Makes push amp api call
     */
    private fun makePushAmpApiCall(context: Context): SMTResponse {
        // if push amp is not enabled then return
        val requestBuilder = SMTRequest.Builder()
                .setHttpMethod(SMTEnumHttpMethodType.GET)
                .setBaseUrl(SMTPreferenceHelper.getAppPreferenceInstance(context, null).getString(SMTPreferenceConstants.SMT_BASE_URL_PUSHAMP))
                .setEndPoint(getPushAmpEndPoint(context))
                .setApiId(SMTRequest.SMTApiTypeID.PUSH_AMPLIFICATION)
        return SMTApiService(requestBuilder.build()).makeApiCall()
    }

    /**
     * Provides Push Amp end point
     * @return String - API end point
     */
    private fun getPushAmpEndPoint(context: Context): String {
        val requestTime = SMTPreferenceHelper.getAppPreferenceInstance(context, null).getString(SMTPreferenceConstants.SMT_PA_REQUEST_TIME, TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis()).toString())
        val guId = SMTGUIDPreferenceHelper.getAppPreferenceInstance(context, null).getString(SMTPreferenceConstants.SMT_GUID)
        val urlParams = SMTEventCommonDataDump(context).getURLParameters(true)
        return "${SMTConfigConstants.PUSHAMP_API_PATH}?$urlParams&guid=$guId&requestTime=$requestTime"
    }


    /**
     * handles push amp api success
     * schedules next push amp api call
     * @param response - API response
     */
    private fun handlePushAmpApiSuccess(context: Context, response: SMTResponse) {
        // Iterate through Push Amp notification to check if already shown to user or not
        // if not then show it to user, record into notification and event table
        val smtPreference = SMTPreferenceHelper.getAppPreferenceInstance(context, null)
        val pushAmpResponse = response as SMTPushAmpResponse
        SMTLogger.i(TAG, "PushAmp [:] ${pushAmpResponse.pushAmpData?.toString()}")
        pushAmpResponse.pushAmpData?.let { value ->
            if (!value.paEnabled) {
                SMTWorkerScheduler.getInstance().cancelPushAmp(context)
            }
        }

        // store push amp settings
        smtPreference.setBoolean(SMTPreferenceConstants.IS_PUSH_AMP_ENABLED, pushAmpResponse.pushAmpData?.paEnabled
                ?: true)
        smtPreference.setInt(SMTPreferenceConstants.PUSH_AMP_INTERVAL, pushAmpResponse.pushAmpData?.paInterval
                ?: SMTConfigConstants.DEFAULT_PUSH_AMP_INTERVAL)

        val areNotificationEnabled = SMTCommonUtility.areNotificationsEnabled(context)
        val isUserPNOptedIn = smtPreference.getBoolean(SMTPreferenceConstants.OPT_IN_OUT_PUSH_NOTIFICATION, true)
        val isNotificationListenerEnabled = smtPreference.getInt(SMTPreferenceConstants.SMT_MF_IS_NOTIFICATION_LISTENER_ENABLED, 0)
        if (areNotificationEnabled && isUserPNOptedIn) {
            SMTPreferenceHelper.getAppPreferenceInstance(context, null).setString(SMTPreferenceConstants.SMT_PA_REQUEST_TIME, TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis()).toString())
            if (pushAmpResponse.pushAmpData?.notificationString?.isNotEmpty() == true) {
                for (notif in pushAmpResponse.pushAmpData?.notificationString!!) {
                    Thread.sleep(100)
                    if (isNotificationListenerEnabled == 1) {
                        Smartech.getInstance(WeakReference(context)).getSMTNotificationListener()?.getSmartechNotifications(notif, SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP)
                    } else {
                        SMTPNHandler(SMTNotificationGeneratorProvider()).handleNotification(context, notif, SMTNotificationSourceType.NOTIFICATION_SOURCE_PAMP)
                    }
                }
            }
        }
    }
}