package com.netcore.android.workmgr

import android.content.Context
import androidx.work.Worker
import androidx.work.WorkerParameters
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.db.SMTEventTable
import com.netcore.android.event.SMTEventSyncStatus
import com.netcore.android.logger.SMTLogger
import com.netcore.android.utility.SMTCommonUtility
import org.json.JSONObject
import java.lang.ref.WeakReference


/**
 *  This class will be manage InProgress events.
 *
 *  @author Netcore
 *
 *  @Description This class will change the status of InProgress events which are present in database
 *  and whose timestamp value is greater then 2 days so it means this events are not processed because of some
 *  error and we send these events back to server with retry set to 1.
 *
 */
internal class InProgressEventWorker(context: Context, param: WorkerParameters) : Worker(context, param) {

    val TAG: String = InProgressEventWorker::class.java.name

    override fun doWork(): Result {
        try {

            val context = applicationContext

            val progressEventHashMap = SMTDataBaseService.getInstance(WeakReference(context)).getInProgressEvents(30)

            /**
             *  Fetch all InProgress events fro DB
             *  and check events timestamp to make sure
             *  that events timestamp is less then
             *  2 days ( default day count ) so
             *  change the status of events to failed and
             *  try it to send back to server otherwise
             *  skip this flow
             *
             */
            if (progressEventHashMap.size > 0) {
                val eventIdArrayList = ArrayList<Int>()

                for ((key, value) in progressEventHashMap) {
                    val objChild = JSONObject(value)
                    val eventTimestamp = objChild.optString("eventTime")
                    if (eventTimestamp.isNotEmpty()) {
                        val isEventStale = SMTCommonUtility.checkDateDifferenceProgressEvent(System.currentTimeMillis(), eventTimestamp.toLong())

                        /**
                         *  if event timestamp is old then current timestamp by 2 days then
                         *  change the status to Failed and retry flag to 1
                         */
                        if (isEventStale) {
                            eventIdArrayList.add(key.toInt())
                            SMTLogger.internal(TAG, "Event in progress $key status updated to failed")
                        } else {
                            SMTLogger.internal(TAG, "Timestamp is not greater then interval")
                        }
                    }
                }
                updateStatusOfProgressEvent(context, eventIdArrayList.toTypedArray())
            }
            return Result.success()
        } catch (e: Exception) {
            e.printStackTrace()
            return Result.failure()
        }

    }

    private fun updateStatusOfProgressEvent(context: Context, eventIdArray: Array<Int>) {
        SMTDataBaseService.getInstance(WeakReference(context)).updateEventProcessingStatus(eventIdArray, SMTEventTable.KEY_EVENT_SYNC_STATUS, SMTEventSyncStatus.EVENT_SYNC_STATUS_FAILED)
        SMTDataBaseService.getInstance(WeakReference(context)).updateFailedBatchPayload()
    }
}