package com.netcore.android.utility

import android.content.Context
import android.content.IntentFilter
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.os.Build
import android.telephony.TelephonyManager
import com.netcore.android.SMTActivityLifecycleCallback
import com.netcore.android.listeners.SMTAppLifeCycleListener
import com.netcore.android.listeners.SMTNetworkStateListener
import com.netcore.android.logger.SMTLogger


/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * Singleton class to provides device network related Info
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */
internal class SMTNetworkInfo(private val context: Context) : SMTAppLifeCycleListener, SMTNetworkStateListener {

    private val TAG = SMTNetworkInfo::class.java.simpleName

    var networkMode = getNetworkType(context).toLowerCase()
    var carrier = getCarrier(context).toLowerCase()
    var countryCode = getCountryCode(context).toLowerCase()
    var radio = getRadioType(context).toLowerCase()
    private lateinit var networkStateReceiver: SMTNetowrkStateReceiver

    /**
     * Provides device radio type
     * like GPRS, Edge, WCDMA, HSDPA, HSUPA, CDMA1x, CDMAEVDORev0, CDMAEVDORevA, CDMAEVDORevB, eHRPD, LTE
     * @param context App context
     */
    private fun getRadioType(context: Context): String {
        try {
            // Fall back to network type
            val teleMan = context.getSystemService(Context.TELEPHONY_SERVICE) as TelephonyManager
            val networkType = teleMan.networkType
            return when (networkType) {
                TelephonyManager.NETWORK_TYPE_GPRS -> "GPRS"
                TelephonyManager.NETWORK_TYPE_EDGE -> "EDGE"
                TelephonyManager.NETWORK_TYPE_UMTS -> "UMTS"
                TelephonyManager.NETWORK_TYPE_HSDPA -> "HSDPA"
                TelephonyManager.NETWORK_TYPE_HSUPA -> "HSUPA"
                TelephonyManager.NETWORK_TYPE_HSPA -> "HSPA"
                TelephonyManager.NETWORK_TYPE_CDMA -> "CDMA"
                TelephonyManager.NETWORK_TYPE_EVDO_0 -> "CDMA - EvDo rev. 0"
                TelephonyManager.NETWORK_TYPE_EVDO_A -> "CDMA - EvDo rev. A"
                TelephonyManager.NETWORK_TYPE_EVDO_B -> "CDMA - EvDo rev. B"
                TelephonyManager.NETWORK_TYPE_1xRTT -> "CDMA - 1xRTT"
                TelephonyManager.NETWORK_TYPE_LTE -> "LTE"
                TelephonyManager.NETWORK_TYPE_EHRPD -> "CDMA - eHRPD"
                TelephonyManager.NETWORK_TYPE_IDEN -> "iDEN"
                TelephonyManager.NETWORK_TYPE_HSPAP -> "HSPA+"
                TelephonyManager.NETWORK_TYPE_GSM -> "GSM"
                TelephonyManager.NETWORK_TYPE_TD_SCDMA -> "TD_SCDMA"
                TelephonyManager.NETWORK_TYPE_IWLAN -> "IWLAN"
                else -> "UNKNOWN"
            }
        } catch (e: Exception) {
            SMTLogger.e(TAG, "Error while fetching Radio type error :-$e")
            return "UNKNOWN"
        }

    }

    /**
     * Provides device Netowrk type
     * like “no-network”, “wifi”, “2g”, “3g”, “4g”
     * @return String - Network type
     */
    private fun getNetworkType(context: Context): String {

        if (!SMTCommonUtility.isNetworkAvailable(context)) return "offline"

        val connManager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager

        // Check for network type WIFI
        if (Build.VERSION.SDK_INT < 23) {
            val ni = connManager.activeNetworkInfo


            @Suppress("DEPRECATION")
            if (ni?.isConnected == true && ni.type == ConnectivityManager.TYPE_WIFI) {
                return "wifi"
            }

        } else {
            val nw = connManager.activeNetwork
            val nc = connManager.getNetworkCapabilities(nw)
            if (nc?.hasTransport(NetworkCapabilities.TRANSPORT_WIFI) == true) {
                return "wifi"
            }
        }
        // Fall back to network type
        val mTelephonyManager = context.getSystemService(Context.TELEPHONY_SERVICE) as TelephonyManager

        val networkType = mTelephonyManager.networkType
        when (networkType) {
            TelephonyManager.NETWORK_TYPE_GPRS,
            TelephonyManager.NETWORK_TYPE_EDGE,
            TelephonyManager.NETWORK_TYPE_CDMA,
            TelephonyManager.NETWORK_TYPE_1xRTT,
            TelephonyManager.NETWORK_TYPE_IDEN -> return "2g"
            TelephonyManager.NETWORK_TYPE_UMTS,
            TelephonyManager.NETWORK_TYPE_EVDO_0,
            TelephonyManager.NETWORK_TYPE_EVDO_A,
            TelephonyManager.NETWORK_TYPE_HSDPA,
            TelephonyManager.NETWORK_TYPE_HSUPA,
            TelephonyManager.NETWORK_TYPE_HSPA,
            TelephonyManager.NETWORK_TYPE_EVDO_B,
            TelephonyManager.NETWORK_TYPE_EHRPD,
            TelephonyManager.NETWORK_TYPE_HSPAP -> return "3g"
            TelephonyManager.NETWORK_TYPE_LTE -> return "4g"
            else -> return "unknown"
        }
    }

    /**
     * Provides Carrier info
     * @param context - App context
     * @return String - Carrier name
     */
    private fun getCarrier(context: Context): String {
        try {
            val manager = context
                    .getSystemService(Context.TELEPHONY_SERVICE) as TelephonyManager
            return manager.networkOperatorName
        } catch (e: Exception) {
            SMTLogger.e(TAG, "Error while fetching Carrier error :-$e")
        }
        return ""
    }

    /**
     * Provided country code
     * @param context - App Context
     * @return Stirng - Country Code
     */
    private fun getCountryCode(context: Context): String {
        return try {
            val tm = context.getSystemService(Context.TELEPHONY_SERVICE) as TelephonyManager
            tm.simCountryIso
        } catch (ignore: Exception) {
            SMTLogger.e(TAG, "Error while fetching country code :-$ignore")
            ""
        }
    }

    /**
     * Register the network state change receiver class
     */
    override fun onAppForeground() {
        networkStateReceiver = SMTNetowrkStateReceiver(this)
        val intentFilter = IntentFilter("android.net.conn.CONNECTIVITY_CHANGE")
        context.registerReceiver(networkStateReceiver, intentFilter)
    }

    /**
     * Unregister the broadcast receiver class
     */
    override fun onAppBackground() {
        if (::networkStateReceiver.isInitialized) {
            context.unregisterReceiver(networkStateReceiver)
        }

    }

    /**
     * update network state change
     */
    override fun onNetworkStateChanged() {
        updateNetworkRelatedInfo()
    }

    /**
     * update network related info on network state change
     */
    private fun updateNetworkRelatedInfo() {
        networkMode = getNetworkType(context)
        carrier = getCarrier(context)
        countryCode = getCountryCode(context)
        radio = getRadioType(context)
    }
}