package com.netcore.android.notification.channel

import android.app.NotificationChannel
import android.app.NotificationChannelGroup
import android.app.NotificationManager
import android.content.Context
import android.content.Context.NOTIFICATION_SERVICE
import android.media.AudioAttributes
import android.media.RingtoneManager
import android.net.Uri
import android.os.Build
import com.netcore.android.notification.SMTNotificationConstants
import com.netcore.android.utility.SMTCommonUtility
import java.lang.ref.WeakReference


internal class SMTNotificationChannelHelper private constructor(val context: WeakReference<Context>) {

    companion object {
        private var INSTANCE: SMTNotificationChannelHelper? = null

        /**
         * Getting instance of the class
         */
        fun getInstance(context: WeakReference<Context>): SMTNotificationChannelHelper =
                INSTANCE
                        ?: synchronized(SMTNotificationChannelHelper::class.java) {
                            INSTANCE
                                    ?: buildInstance(context).also { INSTANCE = it }
                        }

        private fun buildInstance(context: WeakReference<Context>): SMTNotificationChannelHelper {
            return SMTNotificationChannelHelper(context)
        }
    }

    internal fun createNotificationChannelGroup(groupId: String, groupName: CharSequence) {
        try {
            if (isBuildVersionOreoAndAbove()) {
                getNotificationManagerInstance()?.createNotificationChannelGroup(NotificationChannelGroup(groupId, groupName))
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    internal fun deleteNotificationChannelGroup(groupId: String) {
        try {
            if (isBuildVersionOreoAndAbove()) {
                getNotificationManagerInstance()?.deleteNotificationChannelGroup(groupId)
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    internal fun ncGetAllNotificationChannelGroups(): MutableList<NotificationChannelGroup>? {
        try {
            if (isBuildVersionOreoAndAbove()) {
                return getNotificationManagerInstance()?.notificationChannelGroups
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
        return null
    }

    internal fun ncGetAllNotificationChannels(): MutableList<NotificationChannel>? {
        try {
            if (isBuildVersionOreoAndAbove()) {
                return getNotificationManagerInstance()?.notificationChannels
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
        return null
    }

    internal fun createNotificationChannel(smtNotificationChannel: SMTNotificationChannel) {
        try {
            if (isBuildVersionOreoAndAbove()) {
                val notificationChannel = NotificationChannel(
                        smtNotificationChannel.channelId,
                        smtNotificationChannel.channelName,
                        smtNotificationChannel.importance)

                // Channel Description
                if (!smtNotificationChannel.channelDescription.isNullOrEmpty())
                    notificationChannel.description = smtNotificationChannel.channelDescription

                // Notification Sound
                context.get()?.let {
                    if (!smtNotificationChannel.notificationSound.isNullOrEmpty()) {
                        val soundUri: Uri? = SMTCommonUtility.getSoundUri(it, smtNotificationChannel.notificationSound)
                        if (soundUri != null) {
                            notificationChannel.setSound(soundUri, AudioAttributes.Builder().setUsage(AudioAttributes.USAGE_NOTIFICATION).build())
                        }
                    }
                }

                // Channel GroupId
                if (!smtNotificationChannel.ChannelGroupId.isNullOrEmpty())
                    notificationChannel.group = smtNotificationChannel.ChannelGroupId

                //Create channel
                getNotificationManagerInstance()?.createNotificationChannel(notificationChannel)
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    internal fun deleteNotificationChannel(channelId: String) {
        try {
            if (isBuildVersionOreoAndAbove()) {
                getNotificationManagerInstance()?.deleteNotificationChannel(channelId)
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    private fun createDefaultChannel(): String {
        val smtChannelId = SMTNotificationConstants.NOTIF_DEFAULT_CHANNEL_ID
        val channelName = SMTNotificationConstants.NOTIF_DEFAULT_CHANNEL_NAME
        val channelDescription = SMTNotificationConstants.NOTIF_DEFAULT_CHANNEL_DESCRIPTION
        if (isBuildVersionOreoAndAbove()) {
            val importance: Int = NotificationManager.IMPORTANCE_HIGH
            context.get()?.let {
                val smtNotificationChannel = SMTNotificationChannel.Builder(
                        smtChannelId, channelName, importance)
                        .setChannelDescription(channelDescription)
                        .build()
                createNotificationChannel(smtNotificationChannel)
            }
        }
        return smtChannelId
    }

    /**
     * This method checks channel is present or not.
     */
    private fun isValidChannelId(channelId: String): Boolean {
        if (isBuildVersionOreoAndAbove()) {
            return getNotificationManagerInstance()?.getNotificationChannel(channelId) != null
        }
        return false
    }

    /**
     * This method return the channel ID.
     * It will create the new one if given channel not found.
     */
    internal fun getNotificationChannelID(mChannelId: String): String {
        return if (isValidChannelId(mChannelId)) mChannelId else createDefaultChannel()
    }

    /**
     * This method return the NotificationManager instance
     */
    private fun getNotificationManagerInstance(): NotificationManager? {
        var notificationManager: NotificationManager? = null
        try {
            notificationManager = context.get()?.getSystemService(NOTIFICATION_SERVICE) as NotificationManager
        } catch (e: Exception) {
            e.printStackTrace()
        }
        return notificationManager
    }

    private fun isBuildVersionOreoAndAbove(): Boolean {
        return Build.VERSION.SDK_INT >= Build.VERSION_CODES.O
    }

    /**
     * This method return the sound uri if sound is enable from server.
     * If the sound is not found or invalid it will return the default sound uri.
     */
    internal fun getSoundUri(mSoundFile: String): Uri? {
        var notificationSoundUri: Uri? = null
        context.get()?.let {
            notificationSoundUri = SMTCommonUtility.getSoundUri(it, mSoundFile)
        }
        if (notificationSoundUri == null) {
            notificationSoundUri = RingtoneManager.getDefaultUri(RingtoneManager.TYPE_NOTIFICATION)
        }
        return notificationSoundUri
    }

}