package com.netcore.android.notification

import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.os.Bundle
import androidx.core.app.NotificationCompat
import com.netcore.android.event.SMTEventRecorder
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.utility.SMTCommonUtility
import org.json.JSONObject
import java.lang.ref.WeakReference

/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * Class used to show simple notifications (Title, body and action buttons)
 * based on Notification Payload
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */

internal class SMTSimplePNGenerator : SMTBaseNotificationGenerator() {

    /**
     * Create the notification based on the Payload and notify to Notification manager to display it
     */
    fun handle(context: Context, notifModel: SMTNotificationData) {

        val randomNotifyId = SMTCommonUtility.getRandomId()
        notifModel.notificationId = randomNotifyId

        updateNotificationId(context, notifModel)

        notificationManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager

        // Build the content of the notification
        val notificationBuilder = getNotificationBuilder(context,
                notifModel.mTitle ?: "",
                notifModel.mMessage ?: "",
                notifModel.mSubtitle ?: "",
                createPendingIntent(context, notifModel),
                notifModel)
        notificationBuilder.setStyle(NotificationCompat.BigTextStyle().bigText(notifModel.mMessage
                ?: ""))

        // Check for action buttons
        checkAndAddActionButtons(context, notifModel, notificationBuilder)

        // Notify to notification manager
        notificationManager?.notify(randomNotifyId, notificationBuilder.build())
    }

    /**
     * Notification Action receiver Pending Intent created
     * @param context - App context
     * @param notifModel - Parsed Notification data model
     * @return PendingIntent - recevier pending intent
     */
    private fun createPendingIntent(context: Context, notifModel: SMTNotificationData): PendingIntent {
        val rand = System.currentTimeMillis().toInt()
        val bundle = Bundle()
        bundle.putString(SMTNotificationConstants.NOTIF_TYPE_KEY, SMTNotificationType.SIMPLE.type)
        bundle.putString(SMTNotificationConstants.SIMPLE_NOTIF_OTHER_REGION_CLICKED_KEY, SMTNotificationConstants.SIMPLE_NOTIF_OTHER_REGION_CLICKED_KEY)
        bundle.putParcelable(SMTNotificationConstants.NOTIFICATION_PARCEL, notifModel)


        val launchIntent = Intent(context, SMTPNActionReceiver::class.java)

        launchIntent.putExtras(bundle)

        return PendingIntent.getBroadcast(context, rand, launchIntent, PendingIntent.FLAG_UPDATE_CURRENT)
    }

    override fun handleNotificationDismiss(context: Context?, extras: Bundle) {
        // nothing to do for simple type notification
    }

    /**
     * Handles the image click action
     * @param context App context
     * @param extras - Extras passed through the pending intent
     */
    fun handleContentClick(context: Context, extras: Bundle) {
        if (extras.containsKey(SMTNotificationConstants.SIMPLE_NOTIF_OTHER_REGION_CLICKED_KEY)) {
            if (SMTNotificationConstants.SIMPLE_NOTIF_OTHER_REGION_CLICKED_KEY == extras.get(SMTNotificationConstants.SIMPLE_NOTIF_OTHER_REGION_CLICKED_KEY)) {
                val parcel = extras.getParcelable<SMTNotificationData>(SMTNotificationConstants.NOTIFICATION_PARCEL)
                //Update DB record whether it has been clicked
                SMTEventRecorder.getInstance(WeakReference(context)).recordNotificationClick(parcel?.mTrid
                        ?: "", parcel?.mPNMeta ?: JSONObject().toString(),
                        parcel?.mDeepLinkPath ?: "", parcel?.mSource
                        ?: 1, parcel?.mSmtAttributePayload ?: HashMap())

                SMTCommonUtility.handleNotificationClick(context, parcel?.mDeepLinkPath
                        ?: "", parcel?.mCustomPayload)
            }
        }
    }
}