package com.netcore.android.notification

import android.content.Context
import com.netcore.android.Smartech
import com.netcore.android.db.SMTDataBaseService
import com.netcore.android.event.SMTEventRecorder
import com.netcore.android.event.SMTNotificationSourceType
import com.netcore.android.logger.SMTLogger
import com.netcore.android.notification.audio.SMTRichAudioPNGenerator
import com.netcore.android.notification.carousel.SMTCarouselLandscapePNGenerator
import com.netcore.android.notification.carousel.SMTCarouselPortraitPNGenerator
import com.netcore.android.notification.gif.SMTRichGifPNGenerator
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.workmgr.SMTWorkerScheduler
import java.lang.ref.WeakReference

/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * Handles the incoming notification
 *
 * @author Netcore
 * @version 1.0
 * @since 26-02-2019
 */
internal class SMTPNHandler(private val notificationGeneratorProvider: SMTNotificationGeneratorProvider) {

    private val TAG = SMTPNHandler::class.java.simpleName

    private var mNotificationModel: SMTNotificationData? = null

    /**
     * Handled the incoming notification
     * @param context - App context
     * @param notifData - Notification JSON in string format
     */
    fun handleNotification(context: Context, notifData: String, @SMTNotificationSourceType.Source sourceType: Int, disableDeliveary: Boolean = false): Boolean {

        /**
         *
         *  Check status of EventSync worker is running.
         *  If worker is running, It will automatically pickup the delivery of notification event.
         *  If worker is not running, start Event Sync worker so it will send
         *  the delivery event to server,
         *
         */
        SMTWorkerScheduler.getInstance().checkStatusAndScheduleEventWorker(context)

        SMTLogger.v(TAG, "Notification source: $sourceType")
        // Parses the notification data
        mNotificationModel = getNotificationModel(notifData, sourceType)


        if (mNotificationModel?.isFromSmartech == false) {
            SMTLogger.i(TAG, "Notification source is not Smartech.")
            return false
        }

        // If Transaction Id is not there then don't show the Notification
        if (mNotificationModel?.mTrid == null) {
            SMTLogger.w(TAG, "Notification transaction id is null")
            return true
        }

        // Insert Notification into Notification table
        mNotificationModel?.mTrid?.let {

            // If Notification already present in the DB, then don't process it again, just ignore it
            if (SMTDataBaseService.getInstance(WeakReference(context)).findNotificationWithId(it)) {
                SMTLogger.i(TAG, "Notification already displayed to user for trId : $it")
                return true
            }

            // If user has opted for Notification control then we don't need to send delivery again'
            if (!disableDeliveary) {
                // Insert notification into NotificationTable
                val dbService = SMTDataBaseService.getInstance(WeakReference(context))
                dbService.insertPNToNotificationTable(it, notifData, sourceType)

                SMTEventRecorder.getInstance(WeakReference(context)).recordNotificationDelivery(it, notifData,
                        sourceType, mNotificationModel!!)
            }

        }

        // Check if Notification listener is enabled in manifest file then
        // send the notification to client via interface
        val isNotificationListenerEnabled = SMTNotificationUtility.getInstance().checkIfNotificationListenerEnabled(context)
        SMTLogger.v(TAG, "value of isNotificationEnabled $isNotificationListenerEnabled")
        if (isNotificationListenerEnabled == 1) {
            Smartech.getInstance(WeakReference(context)).getSMTNotificationListener()?.getSmartechNotifications(notifData, SMTNotificationSourceType.NOTIFICATION_SOURCE_PN)
        } else {
            mNotificationModel?.let {
                val notificationGenerator = notificationGeneratorProvider.getNotificationGenerator(it.mNotificationType!!, context)
                val type = it.mNotificationType?.toLowerCase() ?: SMTNotificationType.SIMPLE.type
                when (type) {
                    SMTNotificationType.CAROUSEL_LANDSCAPE.type -> (notificationGenerator as SMTCarouselLandscapePNGenerator).handle(it)
                    SMTNotificationType.CAROUSEL_PORTRAIT.type -> (notificationGenerator as SMTCarouselPortraitPNGenerator).handle(it)
                    SMTNotificationType.GIF.type -> (notificationGenerator as SMTRichGifPNGenerator).handle(context, it)
                    SMTNotificationType.AUDIO.type -> (notificationGenerator as SMTRichAudioPNGenerator).handle(context, it)
                    SMTNotificationType.BIG_IMAGE.type -> (notificationGenerator as SMTRichImagePNGenerator).handle(context, it)
                    else -> (notificationGenerator as SMTSimplePNGenerator).handle(context, it)
                }
            }
        }

        return true
    }

    /**
     * Provides notification data model after parsing the notification payload
     */
    fun getNotificationModel(notifData: String, sourceType: Int): SMTNotificationData? {
        if (mNotificationModel != null) {
            return mNotificationModel
        }
        return SMTNotificationParser().parse(notifData, sourceType)
    }

    fun renderNotification(context: Context, notifData: String, @SMTNotificationSourceType.Source notifSource: Int): Boolean {
        val notificationModel = getNotificationModel(notifData, notifSource)
        if (notificationModel != null) {

            if (!notificationModel.isFromSmartech) {
                SMTLogger.i(TAG, "Notification source is not Smartech.")
                return false
            }

            // If Transaction Id is not there then don't show the Notification
            if (notificationModel.mTrid.isNullOrEmpty()) {
                SMTLogger.w(TAG, "Notification transaction id is null")
                return true
            }

            notificationModel.let {
                val notificationGenerator = notificationGeneratorProvider.getNotificationGenerator(it.mNotificationType!!, context)
                val type = it.mNotificationType?.toLowerCase() ?: SMTNotificationType.SIMPLE.type
                when (type) {
                    SMTNotificationType.CAROUSEL_LANDSCAPE.type -> (notificationGenerator as SMTCarouselLandscapePNGenerator).handle(it)
                    SMTNotificationType.CAROUSEL_PORTRAIT.type -> (notificationGenerator as SMTCarouselPortraitPNGenerator).handle(it)
                    SMTNotificationType.GIF.type -> (notificationGenerator as SMTRichGifPNGenerator).handle(context, it)
                    SMTNotificationType.AUDIO.type -> (notificationGenerator as SMTRichAudioPNGenerator).handle(context, it)
                    SMTNotificationType.BIG_IMAGE.type -> (notificationGenerator as SMTRichImagePNGenerator).handle(context, it)
                    else -> (notificationGenerator as SMTSimplePNGenerator).handle(context, it)
                }
            }

        }
        return true
    }
}