package com.netcore.android.network

import android.util.Log
import com.netcore.android.event.SMTEventBatchProcessor
import com.netcore.android.network.models.SMTRequest
import com.netcore.android.network.models.SMTResponse
import java.net.HttpURLConnection
import java.util.concurrent.*
import com.netcore.android.logger.SMTLogger

/**
 * @author Netcore
 * created on 31/01/2019
 * @Description: Request queue class which queues the requests and processes the request.
 * This class maintains its own queue apart from just giving to ThreadPoolExecutor, to handle in case the
 * ThreadPoolExecutor rejects the request
 * Also it has retry mechanism requests will be retried in case of network failure
 */
internal open class SMTRequestQueue : SMTInternalNetworkListener {
    private val TAG = SMTRequestQueue::class.java.simpleName
    private val maxParallelRequests = 5;
    /**
     * map to keep the requests in the FIFO queue
     */
    private var mRequestQueue: LinkedHashMap<Long, SMTRequest> = linkedMapOf()

//    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
//    fun getQueue():LinkedHashMap<Long,SMTRequest>{
//        return mRequestQueue
//    }
    /**
     * hashmap to tracking the executing requests
     */
    private var mRequestProcessingQueue: HashMap<Long, SMTRequest> = hashMapOf()

//    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
//    fun getProcessingQueue():HashMap<Long, SMTRequest>{
//        return mRequestProcessingQueue
//    }

    private var mRunningTaskList: HashMap<Long, Future<*>> = hashMapOf()

    companion object {

        @Volatile
        private var INSTANCE: SMTRequestQueue? = null

        fun getInstance(): SMTRequestQueue =
                INSTANCE ?: synchronized(this) {
                    INSTANCE ?: buildInstance().also {
                        INSTANCE = it
                    }
                }

        private fun buildInstance(): SMTRequestQueue {
            return SMTRequestQueue()
        }

    }

    /**
     * Adding request to the queue and processing.
     * request will be processed based on the number of current executing tasks
     */
    fun addRequest(request: SMTRequest) {
        if (mRequestProcessingQueue.size < maxParallelRequests) {
            mRequestQueue.remove(request.getRequestId())
            processRequest(request)
        } else {
            mRequestQueue[request.getRequestId()] = request
            SMTLogger.v("Queue", "Queue size : ${mRequestQueue.size}")
        }

    }

    /**
     * initiating the request
     */
    private fun processRequest(request: SMTRequest) {

        mRequestProcessingQueue[request.getRequestId()] = request;
        request.setRetryCount(request.getRetryCount().plus(1))
        try {
            val future: Future<*>
            future = SMTThreadPoolManager.getIntance().submit(SMTRequestProcessor(request, this))

            SMTLogger.internal(TAG, "URL: ${request.getBaseUrl()}${request.getApiEndPoint()}")
            SMTLogger.internal(TAG, "RequestID: ${request.getRequestId()}")
            SMTLogger.internal(TAG, "RequestHParams: ${request.getHParams()}")
            mRunningTaskList[request.getRequestId()] = future //Adding it to list to cancel of during process

        } catch (e: Exception) {
            e.printStackTrace()
            SMTLogger.e("Request", "Request failure : ${e.localizedMessage}")
            sendFailureState(request)
        }

    }

    /**
     * Running task completed, removing from the list
     */
    private fun removeRunningTaskFromList(request: SMTRequest?) {

        mRunningTaskList.remove(request?.getRequestId())
    }

    /**
     * preparing failure response. send it back to caller or to retry in case of error while executing the request
     */
    private fun sendFailureState(request: SMTRequest) {
        var networkResponse = SMTResponse()
        networkResponse.requestId = request.getRequestId()
        networkResponse.httpCode = 400
        networkResponse.isSuccess = false
        networkResponse.shouldRetry = true
        networkResponse.smtApiTypeID = request.getAPITypeID()

        onRequestProcessComplete(networkResponse)
    }

    /**
     * Method handles whether to retry or notify client of success or failure
     */

    override fun onRequestProcessComplete(smtResponse: SMTResponse) {

        val responseListener: SMTResponseListener? = (mRequestProcessingQueue[smtResponse.requestId] as SMTRequest).getResponseListener()
        var smtRequest = mRequestProcessingQueue[smtResponse.requestId] as SMTRequest

        removeRunningTaskFromList(mRequestProcessingQueue.get(smtResponse.requestId))
        mRequestProcessingQueue.remove(smtResponse.requestId)
        SMTLogger.internal(TAG, "Event response : ${smtResponse.httpCode}")
        if (smtResponse.shouldRetry) {
            /**
             * retrying if the retried count is less than the defined
             */
            if (smtRequest.getRetryCount() < SMTNetworkManager.MAX_RETRY_COUNT) {
                processRequest(smtRequest)
                return
            }
            responseListener?.onResponseFailure(smtResponse)

        } else if (smtResponse.isSuccess) {
            if (smtResponse.httpCode == HttpURLConnection.HTTP_OK) {
                responseListener?.onResposneSuccess(smtResponse)
                SMTLogger.internal(TAG, "Event request id : ${smtRequest.getRequestId()} :  ${smtResponse.isSuccess}")
            } else {
                responseListener?.onResponseFailure(smtResponse)
                SMTLogger.internal(TAG, "Event request id : ${smtRequest.getRequestId()} :  ${smtResponse.isSuccess}")

            }
        }

        if (mRequestQueue.size != 0) {
            var id = mRequestQueue.keys.iterator().next()
            processRequest(mRequestQueue[id] as SMTRequest)
            mRequestQueue.remove(id)
        }
    }

    /* Remove all tasks in the queue and stop all running threads
     */
    /*fun cancelAllTasks() {
        synchronized(this) {
            mRunningTaskList.forEach { (key, value) ->
                if (!value.isDone) {
                    value.cancel(true)
                }
            }
            mRunningTaskList.clear()

            mRequestProcessingQueue.clear()
            mRequestQueue.clear()

        }
    }*/
}