package com.netcore.android.inbox.views

import android.content.Context
import androidx.appcompat.widget.AppCompatImageView
import androidx.appcompat.widget.AppCompatTextView
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.LinearLayout
import android.widget.ProgressBar
import com.netcore.android.R
import com.netcore.android.mediadownloader.SMTMediaDownloadManager
import com.netcore.android.notification.models.SMTActionButtonData
import com.netcore.android.notification.models.SMTNotificationData
import com.netcore.android.utility.SMTCommonUtility

/**
 * Copyright © 2019 Netcore. All rights reserved.
 *
 * Inbox Image Message Custom View - Handles showing image message
 * Also it downloads the image, record events
 *
 * @author Netcore
 * @version 1.0
 * @since 29-04-2019
 */
internal class SMTInboxImageMessageView(context: Context, attrs: AttributeSet?) : SMTBaseView(context, attrs) {

    private lateinit var mView: View

    init {
        initLayout()
    }

    private fun initLayout() {
        mView = LayoutInflater.from(context).inflate(R.layout.inbox_image_message_layout, this)

        val lp = LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT)

        this.layoutParams = lp

        this.orientation = LinearLayout.VERTICAL
    }

    override fun setNotificationData(notification: SMTNotificationData) {

        setData(notification)

        setViewDetail()

        setActionButtons()

        setLayoutAction()

        setBackgroundShape()

        fetchImage()
    }

    private fun setViewDetail() {
        mView.findViewById<AppCompatTextView>(R.id.tv_image_title).text = mNotificationData.mTitle
        mView.findViewById<AppCompatTextView>(R.id.tv_image_timestamp).text = SMTCommonUtility.getFormattedTimeDifference(mNotificationData.mPublishedTimeStamp)
        mView.findViewById<AppCompatTextView>(R.id.tv_image_message).text = mNotificationData.mMessage
    }


    /**
     * If media already downloaded then show it
     * else download
     */
    private fun fetchImage() {

        setBitmapImage(false)

        if (mNotificationData.mMediaLocalPath.isNullOrEmpty()) {
            showProgressBar()
            downloadImage()
        } else {
            hideProgressBar()
            setBitmapImage(true)
        }
    }

    private fun showProgressBar() {

        mView.findViewById<ProgressBar>(R.id.loadingProgressBar).visibility = View.VISIBLE
    }

    private fun hideProgressBar() {

        mView.findViewById<ProgressBar>(R.id.loadingProgressBar).visibility = View.GONE
    }

    /**
     * Download image
     */
    private fun downloadImage() {
        mNotificationData.mIsForInbox = true

        if (!mNotificationData.mIsDownloadInProgress) {
            mNotificationData.mIsDownloadInProgress = true

            // Download media
            SMTMediaDownloadManager().downloadMedia(context, mNotificationData, object : SMTMediaDownloadManager.MediaDownloadListener {

                override fun onDownloadSuccess(notification: SMTNotificationData) {
                    if (notification.mTrid == mNotificationData.mTrid) {
                        mNotificationData.mIsDownloadInProgress = false
                        hideProgressBar()
                        mNotificationData = notification
                        setBitmapImage(true)
                    }
                }

                override fun onDownloadFailed(notification: SMTNotificationData) {
                    if (notification.mTrid == mNotificationData.mTrid) {
                        // disable progress bar
                        hideProgressBar()
                        mNotificationData.mIsDownloadInProgress = false
                    }
                }
            })
        }
    }

    /**
     * Set Bitmap Image
     * If download success then set the downloaded image
     * else show placeholder image
     */
    private fun setBitmapImage(isDownloadSuccess: Boolean) {
        if (isDownloadSuccess) {
            findViewById<AppCompatImageView>(R.id.iv_image_view)
                    .setImageBitmap(SMTCommonUtility.loadImageFromLocalStorage(mNotificationData.mMediaLocalPath!!))
        } else {
            val notifOptions = SMTCommonUtility.getNotificationOptions(context)

            findViewById<AppCompatImageView>(R.id.iv_image_view)
                    .setImageBitmap(SMTCommonUtility.getBitmapFromResId(context, notifOptions.placeHolderIcon))
        }
    }


    /**
     * Set action listener to the layout
     */
    private fun setLayoutAction() {
        mView.setOnClickListener {
            handleClick(mNotificationData.mDeepLinkPath)
        }
    }

    /**
     * set action button and click listener if there is any action button
     */
    private fun setActionButtons() {
        if (mNotificationData.mActionButtonList?.isNotEmpty() == true) {
            mView.findViewById<LinearLayout>(R.id.image_action_layout).visibility = View.VISIBLE
            hideAllActionButtons()
            mNotificationData.mActionButtonList?.let { actionList ->
                var count = 0
                actionList.forEach { buttonData ->
                    ++count
                    when (count) {
                        1 -> setDataToActionButton(R.id.tv_image_left_action, buttonData)
                        2 -> setDataToActionButton(R.id.tv_image_middle_action, buttonData)
                        3 -> setDataToActionButton(R.id.tv_image_right_action, buttonData)
                    }
                }
            }
        } else {
            mView.findViewById<LinearLayout>(R.id.image_action_layout).visibility = View.GONE
        }
    }

    private fun hideAllActionButtons() {
        mView.findViewById<AppCompatTextView>(R.id.tv_image_left_action).visibility = View.INVISIBLE
        mView.findViewById<AppCompatTextView>(R.id.tv_image_middle_action).visibility = View.INVISIBLE
        mView.findViewById<AppCompatTextView>(R.id.tv_image_right_action).visibility = View.INVISIBLE
    }

    /**
     * Set the data to Action Buttons
     * @param viewid view id to which action to be set
     * @param buttonData data related to the action button
     */
    private fun setDataToActionButton(viewid: Int, buttonData: SMTActionButtonData) {
        val action = mView.findViewById<AppCompatTextView>(viewid)
        action?.let {
            action.text = buttonData.actionName
            action.visibility = View.VISIBLE
            action.setOnClickListener {
                handleClick(buttonData.actionDeeplink)
            }
        }
    }
}