package com.netcore.android.inapp

import android.app.Activity
import android.content.ActivityNotFoundException
import android.content.Intent
import android.net.Uri
import androidx.annotation.VisibleForTesting
import com.netcore.android.SMTActivityLifecycleCallback
import com.netcore.android.SMTConfigConstants
import com.netcore.android.SMTEventParamKeys
import com.netcore.android.event.SMTEventId
import com.netcore.android.inapp.model.SMTInAppRule
import com.netcore.android.logger.SMTLogger
import com.netcore.android.utility.SMTCommonUtility
import org.json.JSONArray
import org.json.JSONObject
import java.text.SimpleDateFormat
import java.util.*
import java.util.regex.Pattern

internal class SMTInAppUtility {


    companion object {
        private val TAG = SMTInAppHandler::class.java.simpleName

        private const val FILTER_PAYLOAD_TYPE_ARRAY_OF_STRINGS_OR_INT = 1
        private const val FILTER_PAYLOAD_TYPE_ARRAY_OF_JSON_OBJECT = 2
        private const val FILTER_PAYLOAD_TYPE_NORMAL_OBJECT = 3
        private const val FILTER_PAYLOAD_TYPE_NESTED_OBJECT = 4
        private const val FILTER_PAYLOAD_TYPE_UNKNOWN = 5


        //Frequency Type constants
        const val FREQUENCY_TYPE_DAY = "day"
        const val FREQUENCY_TYPE_SESSION = "session"
        const val FREQUENCY_TYPE_CAMPAIGN = "campaign"


        /**
         * Check if app is in foreground
         */
        fun isAppInForeground(): Boolean {

            return SMTActivityLifecycleCallback.getInstance().isAppInForeground()
        }

        /**
         * Getting the foreground activity from ActivityLifeCycleCallback
         */
        fun getForeGroundActivity(): Activity? {
            return SMTActivityLifecycleCallback.getInstance().getActivity()?.get()
        }


        /**
         * Method to identify the payload structure. Payload can be a normal object,
         * array of string/integer, array of json objects,
         * nested json object or it could be unknown type
         */
        @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
        fun getFilterPayloadType(filterString: String): Int {
            when {
                //example - s^cell_phone[].brand array of json object
                Pattern.matches("^[^.]+\\[]\\.[^.]+\$", filterString) -> {
                    return FILTER_PAYLOAD_TYPE_ARRAY_OF_JSON_OBJECT
                }

                //example - s^country[]  json array of strings or int
                Pattern.matches("^[^.]+\\[]\$", filterString) -> {
                    return FILTER_PAYLOAD_TYPE_ARRAY_OF_STRINGS_OR_INT
                }
                //example - s^students.first_name  nested json object
                Pattern.matches("^[^.]+\\.[^.]+\$", filterString) -> {
                    return FILTER_PAYLOAD_TYPE_NESTED_OBJECT
                }
                //example - i^age normal json object
                Pattern.matches("^[^.]+\$", filterString) -> {
                    return FILTER_PAYLOAD_TYPE_NORMAL_OBJECT
                }


            }
            return FILTER_PAYLOAD_TYPE_UNKNOWN
        }


        /**
         * Method it get the main key from the filter string
         * example s^students.first_name -> students is main key and first_name is the secondary key in the payload to look for
         */
        @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
        fun getFilterKey(filterString: String, filterPayloadType: Int, isPrimaryKey: Boolean = true): String? {
            var filterKey: String? = null
            when (filterPayloadType) {
                FILTER_PAYLOAD_TYPE_ARRAY_OF_STRINGS_OR_INT -> {
                    if (isPrimaryKey) {
                        filterKey = filterString.substring(filterString.indexOf("^") + 1, filterString.indexOf("["))
                    }
                }
                FILTER_PAYLOAD_TYPE_ARRAY_OF_JSON_OBJECT -> {
                    filterKey = if (isPrimaryKey) {
                        filterString.substring(filterString.indexOf("^") + 1, filterString.indexOf("["))
                    } else {
                        filterString.substring(filterString.indexOf(".") + 1)

                    }
                }
                FILTER_PAYLOAD_TYPE_NESTED_OBJECT -> {
                    filterKey = if (isPrimaryKey) {
                        filterString.substring(filterString.indexOf("^") + 1, filterString.indexOf("."))
                    } else {
                        filterString.substring(filterString.indexOf(".") + 1)

                    }
                }
                FILTER_PAYLOAD_TYPE_NORMAL_OBJECT -> {
                    filterKey = filterString.substring(filterString.indexOf("^") + 1)

                }
            }
            return filterKey
        }


        /**
         * Function to compare the integer operator with given conditions
         */
        @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
        fun validateFilterIntRules(payLoadValue: String?, filter: SMTInAppRule.Filter): Boolean {

            var isShow = false
            if (payLoadValue != null) {
                val flValue = filter.value.trim().toLowerCase()
                val operator = filter.operator.trim()
                when (operator) {
                    "==" -> isShow = java.lang.Float.parseFloat(payLoadValue) == java.lang.Float.parseFloat(flValue)
                    "!=" -> isShow = java.lang.Float.parseFloat(payLoadValue) != java.lang.Float.parseFloat(flValue)
                    "<" -> isShow = java.lang.Float.parseFloat(payLoadValue) < java.lang.Float.parseFloat(flValue)
                    ">" -> isShow = java.lang.Float.parseFloat(payLoadValue) > java.lang.Float.parseFloat(flValue)
                    "<=" -> isShow = java.lang.Float.parseFloat(payLoadValue) <= java.lang.Float.parseFloat(flValue)
                    ">=" -> isShow = java.lang.Float.parseFloat(payLoadValue) >= java.lang.Float.parseFloat(flValue)
                }
            }

            return isShow
        }

        /**
         * Function to compare the payloadvalue with the filter value using the string operator
         */
        @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
        fun validateFilterStringRules(payLoadValue: String?, filter: SMTInAppRule.Filter): Boolean {

            var isShowInApp = false
            if (payLoadValue != null) {
                val plValue = payLoadValue.toLowerCase()
                val flValue = filter.value.toLowerCase()
                val operator = filter.operator
                isShowInApp = if (!operator.isEmpty() && operator == "!") {
                    !plValue.contains(flValue)
                } else {

                    when {
                        Pattern.matches("^\\^.*.\\$$", flValue) -> //check if the rule is compare both plValue and filter value to be equal
                            Pattern.matches(flValue, plValue)
                        Pattern.matches("^\\^.*$", flValue) -> // starts with filter value and ends with any character
                            Pattern.matches("$flValue.*$", plValue)
                        Pattern.matches("^.*\\$$", flValue) -> // starts with any character and ends with filter value
                            Pattern.matches("^.*$flValue", plValue)
                        else -> plValue.contains(flValue)
                    }

                }
            }
            return isShowInApp

        }


        /**
         * Method to check the filters against the eventPayload
         * @param inAppRule InAppRule object
         * @param eventPayLoad Payload of an event
         * @param isShowRule default value
         * @return true or false
         * Description:  Method validates the given filter against the event payload. If event matches then returns true otherwise false
         *
         */
        internal fun checkRuleFilter(inAppRule: SMTInAppRule, eventPayLoad: HashMap<String, Any>, isShowRule: Boolean): Boolean {
            var isShowInApp: Boolean = isShowRule

            try {
                if (inAppRule.whomTo.filter != null && inAppRule.whomTo.filter!!.size > 0) {
                    var isShow = false
                    for (i in 0..(inAppRule.whomTo.filter!!.size - 1)) {

                        //Get the filter key from the InAppRule
                        val filterString = inAppRule.whomTo.filter!![i].filter
                        val filterPayloadType = SMTInAppUtility.getFilterPayloadType(filterString)

                        val isStringOperator = Pattern.matches("^s\\^.*$", filterString)

                        //get payload from the custom event
                        val payLoad: String? = eventPayLoad[SMTEventParamKeys.SMT_PAYLOAD].toString()
                        if (!payLoad.isNullOrEmpty()) {
                            val payload = JSONObject(payLoad!!.trim())

                            when (filterPayloadType) {
                                SMTInAppUtility.FILTER_PAYLOAD_TYPE_ARRAY_OF_STRINGS_OR_INT -> {
                                    val filterPrimaryKey = SMTInAppUtility.getFilterKey(filterString, SMTInAppUtility.FILTER_PAYLOAD_TYPE_ARRAY_OF_STRINGS_OR_INT, true)
                                    if (!filterPrimaryKey.isNullOrEmpty()) {
                                        val payLoadArray = payload.getJSONArray(filterPrimaryKey)
                                        isShow = checkFilterAgainstPayloadArray(payLoadArray, inAppRule.whomTo.filter!![i], isStringOperator)
                                    }
                                }
                                SMTInAppUtility.FILTER_PAYLOAD_TYPE_ARRAY_OF_JSON_OBJECT -> {
                                    val filterPrimaryKey = SMTInAppUtility.getFilterKey(filterString, SMTInAppUtility.FILTER_PAYLOAD_TYPE_ARRAY_OF_JSON_OBJECT, true)
                                    if (!filterPrimaryKey.isNullOrEmpty()) {
                                        val payLoadJsonArray = payload.getJSONArray(filterPrimaryKey)
                                        val filterSecondaryKey = SMTInAppUtility.getFilterKey(filterString, SMTInAppUtility.FILTER_PAYLOAD_TYPE_ARRAY_OF_JSON_OBJECT, false)

                                        isShow = checkFilterAgainstPayloadJsonArray(payLoadJsonArray, inAppRule.whomTo.filter!![i], isStringOperator, filterSecondaryKey)

                                    }

                                }
                                SMTInAppUtility.FILTER_PAYLOAD_TYPE_NESTED_OBJECT -> {
                                    val filterPrimaryKey = SMTInAppUtility.getFilterKey(filterString, SMTInAppUtility.FILTER_PAYLOAD_TYPE_NESTED_OBJECT, true)
                                    if (!filterPrimaryKey.isNullOrEmpty()) {
                                        val payLoadJsonObject = payload.getJSONObject(filterPrimaryKey)
                                        if (payLoadJsonObject != null) {
                                            val filterSecondaryKey = SMTInAppUtility.getFilterKey(filterString, SMTInAppUtility.FILTER_PAYLOAD_TYPE_NESTED_OBJECT, false)
                                            filterSecondaryKey?.let {
                                                val payLoadValue = payLoadJsonObject.get(it).toString()
                                                isShow = checkIfThePayLoadValueMatchesFilterValue(payLoadValue, inAppRule.whomTo.filter!![i], isStringOperator)

                                            }
                                        }
                                    }

                                }
                                SMTInAppUtility.FILTER_PAYLOAD_TYPE_NORMAL_OBJECT -> {
                                    val filterPrimaryKey = SMTInAppUtility.getFilterKey(filterString, SMTInAppUtility.FILTER_PAYLOAD_TYPE_NORMAL_OBJECT, true)
                                    if (!filterPrimaryKey.isNullOrEmpty()) {
                                        val payLoadValue = payload.get(filterPrimaryKey).toString()
                                        isShow = checkIfThePayLoadValueMatchesFilterValue(payLoadValue, inAppRule.whomTo.filter!![i], isStringOperator)

                                    }
                                }
                                else -> // it is unknown payload set false
                                    isShow = false
                            }

                        }

                        //if any filter is validate true then break the loop and then return
                        if (isShow) {
                            isShowInApp = isShow
                            break
                        }
                    }
                    //if all of the filter validation fails then override the default value to false
                    if (!isShow) {
                        isShowInApp = isShow
                    }
                }
            } catch (e: Exception) {
                SMTLogger.e(TAG, "Json parsing error: " + e.localizedMessage)
                isShowInApp = false
            }
            return isShowInApp
        }


        /**
         * Method to loop through array of json object to check if the filter value matches
         */
        @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
        fun checkFilterAgainstPayloadJsonArray(payLoadJsonArray: JSONArray?, filter: SMTInAppRule.Filter, stringOperator: Boolean, key: String?): Boolean {
            var isShowInApp = false
            if (payLoadJsonArray != null && key != null) {
                for (i in 0..(payLoadJsonArray.length() - 1)) {
                    var isShow = checkIfThePayLoadValueMatchesFilterValue((payLoadJsonArray[i] as JSONObject).get(key).toString(), filter, stringOperator)
                    if (isShow) {
                        isShowInApp = isShow

                        break
                    }
                }
            }
            return isShowInApp

        }

        /**
         * method to loop through the array of int/string to check if the filter value matches
         */
        @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
        fun checkFilterAgainstPayloadArray(payLoadArray: JSONArray?, filter: SMTInAppRule.Filter, stringOperator: Boolean): Boolean {
            var isShowInApp = false
            if (payLoadArray != null) {
                for (i in 0..(payLoadArray.length() - 1)) {
                    var isShow = checkIfThePayLoadValueMatchesFilterValue(payLoadArray[i].toString(), filter, stringOperator)
                    if (isShow) {
                        isShowInApp = isShow

                        break
                    }
                }
            }
            return isShowInApp
        }


        /**
         * function to compare the payload value against the filter value
         */
        @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
        fun checkIfThePayLoadValueMatchesFilterValue(payLoadValue: String, filter: SMTInAppRule.Filter, isStringOperator: Boolean): Boolean {
            return if (isStringOperator) {
                validateFilterStringRules(payLoadValue.trim(), filter)
            } else {
                validateFilterIntRules(payLoadValue.trim(), filter)
            }

        }

        /**
         * Check if the rule is applicable now
         */
        internal fun isApplicableNow(inAppRule: SMTInAppRule, date: Date): Boolean {

            val dayOfWeek = SimpleDateFormat("EEEE", Locale.getDefault()).format(date)
            var isApplicableNow = inAppRule.whenTo.occurrence.days.contains(dayOfWeek)
            if (isApplicableNow) {
                isApplicableNow = isCurrentTimeWithinTheRange(inAppRule, date)
            }
            return isApplicableNow
        }

        /**
         * Method to check if the current time is within the bounds of time range given
         */
        internal fun isCurrentTimeWithinTheRange(inAppRule: SMTInAppRule, date: Date): Boolean {
            return if (!inAppRule.whenTo.occurrence.from.isEmpty() && !inAppRule.whenTo.occurrence.to.isEmpty()) {
                val curTime = SimpleDateFormat("HH:mm", Locale.getDefault()).format(date)
                val currentTime = SimpleDateFormat("HH:mm", Locale.getDefault()).parse(curTime)
                val toTime = SimpleDateFormat("HH:mm", Locale.getDefault()).parse(inAppRule.whenTo.occurrence.to)
                val fromTime = SimpleDateFormat("HH:mm", Locale.getDefault()).parse(inAppRule.whenTo.occurrence.from)

                (toTime.after(currentTime) && fromTime.before(currentTime))
            } else {
                true
            }
        }

        internal fun getTodayInMilliSec(): Long {
//            return SimpleDateFormat("dd mm yyyy", Locale.getDefault()).parse(SimpleDateFormat("dd mm yyyy", Locale.getDefault()).format(Date())).time
            var calendar = Calendar.getInstance()
            calendar.set(Calendar.HOUR_OF_DAY, 0)
            calendar.set(Calendar.MINUTE, 0)
            calendar.set(Calendar.SECOND, 0)
            calendar.set(Calendar.MILLISECOND, 0)
            return calendar.time.time
        }

        /**
         * Method to convert the time given in yyyy-mm-dd'T'HH:mm:ss format to timestamp
         */
        internal fun getTimeInMilliSec(time: String): String {
            var simpleDateFormat = SimpleDateFormat(SMTConfigConstants.SERVER_TIME_FORMAT, Locale.getDefault())
            simpleDateFormat.timeZone = TimeZone.getTimeZone("UTC")
            return simpleDateFormat.parse(time).time.toString()
        }

        /**
         * Method to convert the time given in yyyy-mm-dd HH:mm:ss format to timestamp
         */
        internal fun getTimeInMilliSecForModifiedDate(time: String): String? {
            return try {
                var simpleDateFormat = SimpleDateFormat("yyyy-MM-dd HH:mm:ss", Locale.getDefault())
                simpleDateFormat.timeZone = TimeZone.getTimeZone("UTC")
                simpleDateFormat.parse(time).time.toString()
            } catch (e: Exception) {
                null
            }
        }

        internal fun isInAppEvent(eventId: Int): Boolean {
            var isInAppEvent = false
            when (eventId) {
                SMTEventId.EVENT_INAPP_DISMISSED,
                SMTEventId.EVENT_INAPP_VIEWED,
                SMTEventId.EVENT_INAPP_CLICKED -> {
                    isInAppEvent = true
                }
            }
            return isInAppEvent
        }


        /**
         * Custom HTML listener for providing control to user.
         * @param inAppCustomHTMLListener InAppCustomHTMLListener
         * check payload data for valid json and set payload data to listener
         * @param payload
         */
        internal fun processInAppCustomHtmlData(inAppCustomHTMLListener: InAppCustomHTMLListener?, payload: String?) {
            try {
                if (payload != null) {
                    val objPayload = JSONObject(payload)
                    inAppCustomHTMLListener?.let { listener ->
                        {
                            objPayload?.let {
                                listener.customHTMLCallback(SMTCommonUtility.jsonToMap(it))
                            }
                        }
                    } ?: SMTLogger.w(TAG, "InAppCustomHtmlListener instance is null")
                }
            } catch (e: java.lang.Exception) {
                SMTLogger.e(TAG, "Netcore Error: " + e.message)
            }
        }
    }
}