package com.netcore.android

import android.app.Activity
import android.app.Application
import android.os.Bundle
import com.netcore.android.inapp.SMTInAppHandler
import com.netcore.android.listeners.SMTAppLifeCycleListener
import com.netcore.android.logger.SMTLogger
import com.netcore.android.utility.SMTInfo
import com.netcore.android.utility.SMTNetworkInfo
import java.lang.ref.WeakReference

/**
 * Class for handling activity lifecycle events
 */
internal class SMTActivityLifecycleCallback private constructor() {

    companion object {
        @Volatile
        private var instance: SMTActivityLifecycleCallback? = null

        @Synchronized
        fun getInstance(): SMTActivityLifecycleCallback =
                instance ?: synchronized(SMTActivityLifecycleCallback::class.java) {
                    instance ?: buildInstance().also { instance = it }
                }


        private fun buildInstance(): SMTActivityLifecycleCallback {
            return SMTActivityLifecycleCallback()
        }
    }

    private val TAG = SMTActivityLifecycleCallback::class.java.simpleName
    private var activityReferences: Int = 0
    private var isActivityChangingConfigurations = false
    private var isAppRegisteredForLifecycleCallback = false
    var currentActivity: WeakReference<Activity?>? = null
    private val lifeCycleListeners = ArrayList<SMTAppLifeCycleListener>()

    /**
     * Enables lifecycle callbacks for Android devices
     * @param application App's Application object
     */
    fun register(application: Application) {
        if (isAppRegisteredForLifecycleCallback) {
            SMTLogger.v(TAG, "Lifecycle callbacks have already been registered")
            return
        }

        isAppRegisteredForLifecycleCallback = true

        application.registerActivityLifecycleCallbacks(object : Application.ActivityLifecycleCallbacks {
            override fun onActivityPaused(activity: Activity) {
                Smartech.getInstance(WeakReference(activity.applicationContext)).onAppForegroundStateChanged(false)
            }

            override fun onActivityResumed(activity: Activity) {
                currentActivity = WeakReference(activity)
                Smartech.getInstance(WeakReference(activity.applicationContext)).onAppForegroundStateChanged(true)
            }

            override fun onActivityStarted(activity: Activity) {
                SMTLogger.v(TAG, "Lifecycle callback - Activity Started")
                activityReferences++
                if (activityReferences == 1 && !isActivityChangingConfigurations) {
                    // App enters foreground
                    // notify the listener about app foreground status
                    Smartech.getInstance(WeakReference(activity.applicationContext)).onAppForeground()
                    SMTInfo.getInstance(WeakReference(activity.applicationContext)).enableNetworkStateListener(true)
                    lifeCycleListeners.forEach { listener ->
                        listener.onAppForeground()
                    }
                }
                isActivityChangingConfigurations = false
            }

            override fun onActivityDestroyed(activity: Activity) {
                SMTInAppHandler.getInstance().dismiss()
            }

            override fun onActivitySaveInstanceState(activity: Activity, outState: Bundle?) {

            }

            override fun onActivityStopped(activity: Activity) {
                SMTLogger.v(TAG, "Lifecycle callback - Activity Stopped")
                isActivityChangingConfigurations = activity.isChangingConfigurations == true
                if (activityReferences > 0) activityReferences--
                if (activityReferences == 0 && !isActivityChangingConfigurations) {
                    /**
                     * un setting the current activity to null when app goes to background
                     */
                    Smartech.getInstance(WeakReference(activity.applicationContext)).onAppBackground()
                    SMTInfo.getInstance(WeakReference(activity.applicationContext)).enableNetworkStateListener(false)
                    currentActivity = null
                    // App enters background
                    SMTLogger.v(TAG, "Lifecycle callback - App is in background")
                    // notify the listener about app background status
                    lifeCycleListeners.forEach { listener ->
                        listener.onAppBackground()
                    }
                }
            }

            override fun onActivityCreated(activity: Activity, savedInstanceState: Bundle?) {
            }
        })
        SMTLogger.i(TAG, "Activity Lifecycle Callback successfully registered")
    }

    fun isAppInForeground(): Boolean {

        return activityReferences > 0
    }

    fun getActivity(): WeakReference<Activity?>? {
        return currentActivity

    }

    fun isLifeCycleCallbackRegistered() = isAppRegisteredForLifecycleCallback

    /**
     * to set app lifecycle listener
     */
    internal fun setLifeCycleListener(listener: SMTAppLifeCycleListener) {
        lifeCycleListeners.add(listener)
    }

    /**
     * to remove app lifecycle listener
     */
    internal fun removeLifeCycleListener(listener: SMTAppLifeCycleListener) {
        lifeCycleListeners.remove(listener)
    }
}