package com.mxchip.mx_image_loader_engine_glide.transform;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.drawable.BitmapDrawable;
import android.util.Log;
import android.view.View;

import androidx.annotation.NonNull;

import com.bumptech.glide.Glide;
import com.bumptech.glide.load.Transformation;
import com.bumptech.glide.load.engine.Resource;
import com.bumptech.glide.load.engine.bitmap_recycle.BitmapPool;
import com.bumptech.glide.load.resource.bitmap.BitmapResource;

import java.security.MessageDigest;

/**
 * Created with Android Studio
 *
 * @author: chenxukun
 * @date: 2020/8/20
 * @time: 3:24 PM
 * @desc:
 */
public class BlurTransformation implements Transformation<Bitmap> {

    private static final int MAX_RADIUS = 25;
    private static final int DEFAULT_DOWN_SAMPLING = 1;

    private BitmapPool mBitmapPool;
    private int mRadius;
    private int mSample;

    public BlurTransformation(Context context) {
        this(context, MAX_RADIUS);
    }

    public BlurTransformation(Context context, BitmapPool bitmapPool) {
        this(context, bitmapPool, MAX_RADIUS);
    }

    public BlurTransformation(Context context, BitmapPool bitmapPool, int radius) {
        this(context, bitmapPool, radius, DEFAULT_DOWN_SAMPLING);
    }

    public BlurTransformation(Context context, int radius) {
        this(context, Glide.get(context).getBitmapPool(), radius);
    }

    public BlurTransformation(Context context, int radius, int sample) {
        this(context, Glide.get(context).getBitmapPool(), radius, sample);
    }

    public BlurTransformation(Context context, BitmapPool pool, int radius, int sample) {
        mBitmapPool = pool;
        mRadius = radius;
        mSample = sample;
    }

    @NonNull
    @Override
    public Resource<Bitmap> transform(@NonNull Context context, @NonNull Resource<Bitmap> resource, int outWidth, int outHeight) {
        Bitmap source = resource.get();

        int width = source.getWidth();
        int height = source.getHeight();
        int scaleWidth = width / mSample;
        int scaleHeight = height / mSample;

        Bitmap.Config config = source.getConfig() != null ? source.getConfig() : Bitmap.Config.ARGB_8888;
        Bitmap bitmap = mBitmapPool.get(scaleWidth, scaleHeight, config);
        if (bitmap == null) {
            bitmap = Bitmap.createBitmap(scaleWidth, scaleHeight, config);
        }
        Canvas canvas = new Canvas(bitmap);
        canvas.scale(1 / (float) mSample, 1 / (float) mSample);
        Paint paint = new Paint();
        paint.setFlags(Paint.FILTER_BITMAP_FLAG);
        canvas.drawBitmap(source, 0, 0, paint);
        bitmap = FastBlur.blur(bitmap, mRadius, true);
        return BitmapResource.obtain(bitmap, mBitmapPool);
    }

    @Override
    public boolean equals(Object obj) {
        return (hashCode() == obj.hashCode());
    }

    @Override
    public int hashCode() {
        return getId().hashCode();
    }

    @Override
    public void updateDiskCacheKey(MessageDigest messageDigest) {
        messageDigest.update(getId().getBytes());
    }

    private String getId() {
        return ("BlurTransformation(radius=" + mRadius + ", sample=" + mSample + ")");
    }

    public void transformToView(Bitmap source, View sourceView, View targetView) {
        int scaledWidth;
        int scaledHeight;
        float translateX;
        float translateY;

        if (sourceView != null) {
            float dx = (float) source.getWidth() / sourceView.getMeasuredWidth() / mSample;
            float dy = (float) source.getHeight() / sourceView.getMeasuredHeight() / mSample;
            scaledHeight = (int) (targetView.getMeasuredHeight() * dy);
            scaledWidth = (int) (targetView.getMeasuredWidth() * dx);
            translateX = -targetView.getLeft() * dx;
            translateY = -targetView.getTop() * dy;
        } else {
            scaledWidth = targetView.getWidth() / mSample;
            scaledHeight = targetView.getHeight() / mSample;
            translateX = -targetView.getLeft() / mSample;
            translateY = -targetView.getTop() / mSample;
        }

        if (scaledHeight == 0 || scaledWidth == 0) {
            Log.d("blurtest", "size==0");
            return;
        }
        Bitmap bitmap = mBitmapPool.get(scaledWidth, scaledHeight, Bitmap.Config.ARGB_8888);
        if (bitmap == null) {
            bitmap = Bitmap.createBitmap(scaledWidth, scaledHeight, Bitmap.Config.ARGB_8888);
        }
        Canvas canvas = new Canvas(bitmap);
        canvas.translate(translateX, translateY);
        canvas.scale(1 / (float) mSample, 1 / (float) mSample);
        Paint paint = new Paint();
        paint.setFlags(Paint.FILTER_BITMAP_FLAG);
        canvas.drawBitmap(source, 0, 0, paint);
        bitmap = FastBlur.blur(bitmap, mRadius, true);
        targetView.setBackground(new BitmapDrawable(targetView.getResources(), bitmap));
        targetView.invalidate();
    }
}
