package com.mxchip.mx_image_loader_engine_glide.engine;

import android.app.Activity;
import android.content.Context;
import android.graphics.Bitmap;
import android.os.Handler;
import android.os.Looper;
import android.view.View;
import android.view.ViewTreeObserver;
import android.widget.ImageView;

import androidx.annotation.Nullable;

import com.bumptech.glide.Glide;
import com.bumptech.glide.MemoryCategory;
import com.bumptech.glide.RequestBuilder;
import com.bumptech.glide.RequestManager;
import com.bumptech.glide.load.DataSource;
import com.bumptech.glide.load.engine.GlideException;
import com.bumptech.glide.load.resource.gif.GifDrawable;
import com.bumptech.glide.request.RequestListener;
import com.bumptech.glide.request.RequestOptions;
import com.bumptech.glide.request.target.ImageViewTarget;
import com.bumptech.glide.request.target.Target;
import com.mxchip.config.DisplayConfig;
import com.mxchip.config.RadiusConfig;
import com.mxchip.factory.DisplayConfigFactory;
import com.mxchip.interfaces.IImageLoader;
import com.mxchip.mx_image_loader_engine_glide.factory.RequestOptionFactory;
import com.mxchip.mx_image_loader_engine_glide.transform.BlurTransformation;

import java.io.File;
import java.lang.ref.WeakReference;

/**
 * Created with Android Studio
 *
 * @author: chenxukun
 * @date: 2020/8/19
 * @time: 7:48 PM
 * @desc:
 */
public class ImageLoaderGlideEngine implements IImageLoader {

    private Context mContext;
    private RequestListener mRequestListener;

    /**
     * unique request listener
     */
    private boolean isInit = false;

    private String adapt(String url) {
        if (url == null) {
            return "";
        }
        /**
         * imageloader的url 兼容
         */
        if (url.startsWith("assets://")) {
            return url.replace("assets://", "file:///android_asset/");
        } else if (url.startsWith("/")) {
            return "file://" + url;
        } else {
            return url;
        }
    }

    private DisplayConfig generateDisplayConfig(int radius, int direction) {
        return new DisplayConfig.Builder().radius(radius).radiusDircetion(direction).build();
    }

    @Override
    public void display(ImageView imageView, String url) {
        display(imageView, url, -1);
    }

    @Override
    public void display(ImageView imageView, String url, int radius) {
        display(imageView, url, radius, RadiusConfig.RadiusAll);
    }

    @Override
    public void display(ImageView imageView, String url, int radius, int direction) {
        DisplayConfig config = null;
        if (radius > 0) {
            config = generateDisplayConfig(radius, direction);
        }
        display(imageView, url, config);
    }

    @Override
    public void display(ImageView imageView, String url, DisplayConfig config) {
        display(Glide.with(mContext), url, imageView, config);
    }

    @Override
    public void display(Activity activity, ImageView imageView, String url) {
        display(activity, imageView, url, -1);
    }

    @Override
    public void display(Activity activity, ImageView imageView, String url, DisplayConfig config) {
        display(Glide.with(activity), url, imageView, config);
    }

    @Override
    public void display(Activity activity, ImageView imageView, String url, int radius) {
        display(activity, imageView, url, radius, RadiusConfig.RadiusAll);
    }

    @Override
    public void display(Activity activity, ImageView imageView, String url, int radius, int direction) {
        DisplayConfig config = null;
        if (radius > 0) {
            config = generateDisplayConfig(radius, direction);
        }
        display(activity, imageView, url, config);
    }

    @Override
    public void display(Context context, ImageView imageView, String url) {
        display(context, imageView, url, -1);
    }

    @Override
    public void display(Context context, ImageView imageView, String url, DisplayConfig config) {
        display(Glide.with(context), url, imageView, config);
    }

    @Override
    public void display(Context context, ImageView imageView, String url, int radius) {
        display(context, imageView, url, radius, RadiusConfig.RadiusAll);
    }

    @Override
    public void display(Context context, ImageView imageView, String url, int radius, int direction) {
        DisplayConfig config = null;
        if (radius > 0) {
            config = generateDisplayConfig(radius, direction);
        }
        display(context, imageView, url, config);
    }

    @Override
    public void display(ImageView imageView, int resId) {
        display(imageView, resId, -1);
    }

    @Override
    public void display(ImageView imageView, int resId, int radius) {
        display(imageView, resId, radius, RadiusConfig.RadiusAll);
    }

    @Override
    public void display(ImageView imageView, int resId, int radius, int direction) {
        DisplayConfig config = null;
        if (radius > 0) {
            config = generateDisplayConfig(radius, direction);
        }
        display(imageView, resId, config);
    }

    @Override
    public void display(ImageView imageView, int resId, DisplayConfig config) {
        RequestBuilder builder = config.isGif() ? Glide.with(mContext).asGif().load(resId) : Glide.with(mContext).load(resId);
        display(builder, imageView, config);
    }

    @Override
    public void display(Activity activity, ImageView imageView, int resId) {
        display(activity, imageView, resId, -1);
    }

    @Override
    public void display(Activity activity, ImageView imageView, int resId, DisplayConfig op) {
        if (activity != null && !activity.isFinishing() && !activity.isDestroyed()) {
            RequestBuilder builder = op!= null && op.isGif() ? Glide.with(activity).asGif().load(resId) : Glide.with(activity).load(resId);
            display(builder, imageView, op);
        }
    }

    @Override
    public void display(Activity activity, ImageView imageView, int resId, int radius) {
        display(activity, imageView, resId, radius, RadiusConfig.RadiusAll);
    }

    @Override
    public void display(Activity activity, ImageView imageView, int resId, int radius, int direction) {
        DisplayConfig config = null;
        if (radius > 0) {
            config = generateDisplayConfig(radius, direction);
        }
        display(activity, imageView, resId, config);
    }

    @Override
    public void display(Context context, ImageView imageView, int resId) {
        display(context, imageView, resId, -1);
    }

    @Override
    public void display(Context context, ImageView imageView, int resId, DisplayConfig op) {
        if (context != null) {
            RequestBuilder builder = op!= null && op.isGif() ? Glide.with(context).asGif().load(resId) : Glide.with(context).load(resId);
            display(builder, imageView, op);
        }
    }

    @Override
    public void display(Context context, ImageView imageView, int resId, int radius) {
        display(context, imageView, resId, -1, RadiusConfig.RadiusAll);
    }

    @Override
    public void display(Context context, ImageView imageView, int resId, int radius, int direction) {
        DisplayConfig config = null;
        if (radius > 0) {
            config = generateDisplayConfig(radius, direction);
        }
        display(context, imageView, resId, config);
    }

    @Override
    public void display(ImageView imageView, File file) {
        display(imageView, file, -1);
    }

    @Override
    public void display(ImageView imageView, File file, DisplayConfig config) {
        RequestBuilder builder = config.isGif()? Glide.with(mContext).asGif().load(file) : Glide.with(mContext).load(file);
        display(builder, imageView, config);
    }

    @Override
    public void display(ImageView imageView, File file, int radius) {
        display(imageView, file, radius, RadiusConfig.RadiusAll);
    }

    @Override
    public void display(ImageView imageView, File file, int radius, int direction) {
        DisplayConfig config = null;
        if (radius > 0) {
            config = generateDisplayConfig(radius, direction);
        }
        display(imageView, file, config);
    }

    @Override
    public void display(Activity activity, ImageView imageView, File file) {
        display(activity, imageView, file, -1);
    }

    @Override
    public void display(Activity activity, ImageView imageView, File file, DisplayConfig config) {
        if (activity != null && !activity.isFinishing() && !activity.isDestroyed()) {
            RequestBuilder builder = config.isGif() ? Glide.with(activity).asGif().load(file) : Glide.with(activity).load(file);
            display(builder, imageView, config);
        }
    }

    @Override
    public void display(Activity activity, ImageView imageView, File file, int radius) {
        display(activity, imageView, file, radius, RadiusConfig.RadiusAll);
    }

    @Override
    public void display(Activity activity, ImageView imageView, File file, int radius, int direction) {
        DisplayConfig config = null;
        if (radius > 0) {
            config = generateDisplayConfig(radius, direction);
        }
        display(activity, imageView, file, config);
    }

    @Override
    public void display(Context context, ImageView imageView, File file) {
        display(context, imageView, file, -1);
    }

    @Override
    public void display(Context context, ImageView imageView, File file, DisplayConfig config) {
        if (context != null) {
            RequestBuilder builder = config.isGif() ? Glide.with(context).asGif().load(file) : Glide.with(context).load(file);
            display(builder, imageView, config);
        }
    }

    @Override
    public void display(Context context, ImageView imageView, File file, int radius) {
        display(context, imageView, file, radius, RadiusConfig.RadiusAll);
    }

    @Override
    public void display(Context context, ImageView imageView, File file, int radius, int direction) {
        DisplayConfig config = null;
        if (radius > 0) {
            config = generateDisplayConfig(radius, direction);
        }
        display(context, imageView, file, config);
    }

    private void display(RequestManager manager, String url, ImageView imageView, DisplayConfig config) {
        RequestBuilder builder;
        if (config.isGif()) {
            builder = manager.asGif();
        } else {
            builder = manager.asBitmap();
        }
        builder.load(adapt(url));
        display(builder, imageView, config);
    }

    private void display(RequestBuilder builder, ImageView imageView, final DisplayConfig config) {

        /**
         * 对imageView使用弱引用，避免activity或者fragment长期不释放，导致imageView中的bitmap资源无法回收
         */
        WeakReference<ImageView> imageViewWeakReference = new WeakReference<>(imageView);
        if (imageViewWeakReference.get() == null) {
            return;
        }

        if (config == null) {
            if (imageViewWeakReference.get() != null) {
                builder.into(imageViewWeakReference.get());
            }
            return;
        }
        DisplayConfigFactory<RequestOptions> factory = new RequestOptionFactory(mContext);
        RequestOptions options = factory.toTargetConfig(config);
        if (options != null) {
            builder.apply(options);
        }
        if (config.getThumbnail() != 0) {
            if (config.getThumbnail() < 0f)
                throw new IllegalArgumentException("thumbnail value cannot a minus value");
            builder.thumbnail(config.getThumbnail());
        }

        if (config.isPreLoad()) {
            builder.preload();
        } else {
            if (config.getEffectOutputView() != null) {
                final DisplayConfig configTmp = config;
                RequestListener requestListener = new RequestListener() {
                    @Override
                    public boolean onLoadFailed(@Nullable GlideException e, Object model, Target target, boolean isFirstResource) {
                        return false;
                    }

                    @Override
                    public boolean onResourceReady(final Object resource, Object model, Target target, DataSource dataSource, boolean isFirstResource) {
                        if (resource instanceof Bitmap) {
                            if (target instanceof ImageViewTarget) {
                                ImageViewTarget ivTarget = (ImageViewTarget) target;
                                final View view = ivTarget.getView();
                                if (configTmp.getEffectOutputView().getMeasuredHeight() == 0) {
                                    ViewTreeObserver viewTreeObserver = configTmp.getEffectOutputView().getViewTreeObserver();
                                    if (viewTreeObserver.isAlive()) {
                                        viewTreeObserver.addOnPreDrawListener(new ViewTreeObserver.OnPreDrawListener() {
                                            @Override
                                            public boolean onPreDraw() {
                                                configTmp.getEffectOutputView().getViewTreeObserver().removeOnPreDrawListener(this);
                                                new BlurTransformation(mContext, Glide.get(mContext).getBitmapPool(), configTmp.getBlurRadius(), configTmp.getBlurSample()).transformToView((Bitmap) resource, view, configTmp.getEffectOutputView());
                                                return false;
                                            }
                                        });
                                    }
                                } else {
                                    new BlurTransformation(mContext, Glide.get(mContext).getBitmapPool(), configTmp.getBlurRadius(), configTmp.getBlurSample()).transformToView((Bitmap) resource, view, configTmp.getEffectOutputView());
                                }
                            }
                        }
                        return false;
                    }
                };
                if (imageViewWeakReference.get() != null) {
                    builder.listener(requestListener).into(imageViewWeakReference.get());
                }
            } else {
                if (imageViewWeakReference.get() != null) {
                    builder.into(imageViewWeakReference.get());
                }
            }
        }
    }

    @Override
    public void startGif(ImageView imageView) {
        if (imageView.getDrawable() != null && imageView.getDrawable() instanceof GifDrawable) {
            ((GifDrawable) imageView.getDrawable()).start();
        }
    }

    @Override
    public void stopGif(ImageView imageView) {
        if (imageView.getDrawable() != null && imageView.getDrawable() instanceof GifDrawable) {
            ((GifDrawable) imageView.getDrawable()).stop();
        }
    }

    @Override
    public void recycleGif(ImageView imageView) {
        if (imageView.getDrawable() != null && imageView.getDrawable() instanceof GifDrawable) {
            ((GifDrawable) imageView.getDrawable()).recycle();
        }
    }

    @Override
    public void cancel(ImageView imageView) {
        Glide.with(mContext).clear(imageView);
    }

    @Override
    public void pauseRequest() {
        Glide.with(mContext).pauseRequests();
    }

    @Override
    public void resumeRequest() {
        Glide.with(mContext).resumeRequests();
    }

    @Override
    public void clearMemoryCache() {
        Glide.get(mContext).clearMemory();
    }

    @Override
    public void clearDiskCache() {
        // 需要操作文件，必须在子线程
        new Thread(new Runnable() {
            @Override
            public void run() {
                Glide.get(mContext).clearDiskCache();
            }
        });
    }

    @Override
    public void onTrimMemory(int level) {
        Glide.get(mContext).onTrimMemory(level);
    }

    @Override
    public void onLowMemory() {
        Glide.get(mContext).onLowMemory();
    }

    @Override
    public void init(Context context) {
        mContext = context;
        Glide.get(context.getApplicationContext()).setMemoryCategory(MemoryCategory.NORMAL);
        isInit = true;
    }

    @Override
    public void release() {
        if (isInit) {
            if (Looper.getMainLooper() == Looper.myLooper()) {
                clearMemoryCache();
                Glide.with(mContext).onDestroy();
                isInit = false;
            } else  {
                Handler handler = new Handler(Looper.getMainLooper());
                handler.post(new Runnable() {
                    @Override
                    public void run() {
                        clearMemoryCache();
                        Glide.with(mContext).onDestroy();
                        isInit = false;
                    }
                });
            }
        }
    }
}
