/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage.core.model;

import org.mule.runtime.api.component.ComponentIdentifier;

import static org.mule.munit.common.util.Preconditions.checkArgument;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import java.io.Serializable;

/**
 * Unique identifier for a configuration option. Every configuration option has a namespace and an identifier.
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class CoverageComponentIdentifier implements ComponentIdentifier, Serializable {

  private static final long serialVersionUID = -7904681927277956932L;

  private String namespace;
  private String name;
  public static final String CORE_PREFIX = "mule";

  public CoverageComponentIdentifier(String namespace, String name) {
    checkArgument(namespace != null && !namespace.trim().isEmpty(), "Prefix URI must be not blank");
    checkArgument(name != null && !name.trim().isEmpty(), "Name must be not blank");

    this.namespace = namespace;
    this.name = name;
  }

  /**
   * {@inheritDoc}
   */
  public String getNamespace() {
    return namespace;
  }

  /**
   * {@inheritDoc}
   */
  public String getName() {
    return name;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    CoverageComponentIdentifier that = (CoverageComponentIdentifier) o;

    if (!getNamespace().equalsIgnoreCase(that.getNamespace())) {
      return false;
    }
    return getName().equals(that.getName());
  }

  @Override
  public int hashCode() {
    int result = getNamespace().toLowerCase().hashCode();
    result = 31 * result + getName().hashCode();
    return result;
  }

  @Override
  public String toString() {
    return getNamespace().equals(CORE_PREFIX) ? getName() : getNamespace() + ":" + getName();
  }


  public static CoverageComponentIdentifier fromComponentIdentifier(ComponentIdentifier identifier) {
    return new CoverageComponentIdentifier(identifier.getNamespace(), identifier.getName());
  }

  public static CoverageComponentIdentifier parseComponentIdentifier(String componentIdentifier) {
    // This is reading the alias of the namespace in a given xml, not the actual namespace
    checkArgument(!isEmpty(componentIdentifier), "identifier cannot be an empty string or null");
    String[] values = componentIdentifier.split(":");
    String namespace;
    String identifier;
    if (values.length == 2) {
      namespace = values[0];
      identifier = values[1];
    } else {
      namespace = CORE_PREFIX;
      identifier = values[0];
    }
    return new CoverageComponentIdentifier(namespace, identifier);
  }

  @Override
  public String getNamespaceUri() {
    return namespace;
  }


}
