/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage.core.interception;

import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.INTERCEPTING;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.OPERATION;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;

/**
 * Defines the logic regarding how to treat {@link ComponentLocation} in different situations
 * 
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class ComponentLocationFilter {

  private static final List<String> NON_INTERCEPTABLE_NAMESPACES = Arrays.asList("munit", "munit-tools");

  /**
   * Defines if a location should be taken under account for the purposes of coverage calculation
   * 
   * @param location
   * @return false if the location can be discarded, true otherwise
   */
  public static boolean isRepresentativeForCoverage(ComponentLocation location) {
    return shouldIntercept(location);
  }

  /**
   * Defines if a location should be intercepted by the coverage interceptor
   * 
   * @param location
   * @return false if the location should not be intercepted, true otherwise
   */
  public static boolean shouldIntercept(ComponentLocation location) {
    if (isLocationInsideMunitTest(location)) {
      return false;
    }

    TypedComponentIdentifier identifier = location.getComponentIdentifier();
    if (NON_INTERCEPTABLE_NAMESPACES.contains(identifier.getIdentifier().getNamespace())) {
      return false;
    }

    if (!OPERATION.equals(identifier.getType()) && !INTERCEPTING.equals(identifier.getType())) {
      return false;
    }

    return true;
  }

  private static Boolean isLocationInsideMunitTest(ComponentLocation location) {
    Optional<TypedComponentIdentifier> partIdentifier = location.getParts().get(0).getPartIdentifier();

    if (partIdentifier.isPresent()) {
      if ("munit".equals(partIdentifier.get().getIdentifier().getNamespace())) {
        return true;
      }
    }
    return false;
  }

}
