/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.runner.config;

import static org.mule.munit.runner.config.MunitXmlNamespaceInfoProvider.NAME_SPACE;
import static org.mule.runtime.dsl.api.component.AttributeDefinition.Builder.fromChildCollectionConfiguration;
import static org.mule.runtime.dsl.api.component.AttributeDefinition.Builder.fromChildConfiguration;
import static org.mule.runtime.dsl.api.component.AttributeDefinition.Builder.fromFixedValue;
import static org.mule.runtime.dsl.api.component.AttributeDefinition.Builder.fromSimpleParameter;
import static org.mule.runtime.dsl.api.component.AttributeDefinition.Builder.fromSimpleReferenceParameter;
import static org.mule.runtime.dsl.api.component.TypeDefinition.fromType;

import static org.apache.commons.lang3.StringUtils.EMPTY;

import org.mule.munit.common.api.model.EventAttributes;
import org.mule.munit.common.api.model.Payload;
import org.mule.munit.common.api.model.UntypedEventError;
import org.mule.munit.common.api.model.Variable;
import org.mule.munit.runner.component.factory.TestProcessorChainFactory;
import org.mule.munit.runner.flow.AfterSuite;
import org.mule.munit.runner.flow.AfterTest;
import org.mule.munit.runner.flow.BeforeSuite;
import org.mule.munit.runner.flow.BeforeTest;
import org.mule.munit.runner.flow.TestFlow;
import org.mule.munit.runner.processors.EnableFlowSources.FlowRef;
import org.mule.munit.runner.processors.MunitModule;
import org.mule.munit.runner.processors.Parameterization;
import org.mule.munit.runner.processors.Parameterizations;
import org.mule.munit.runner.processors.Parameters;
import org.mule.munit.runner.processors.SetEventProcessor;
import org.mule.runtime.core.privileged.processor.chain.MessageProcessorChain;
import org.mule.runtime.dsl.api.component.ComponentBuildingDefinition;
import org.mule.runtime.dsl.api.component.ComponentBuildingDefinitionProvider;

import java.util.LinkedList;
import java.util.List;

/**
 * <p>
 * Provides the Component Definition for the MUnit schema
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class MunitComponentBuildingDefinitionProvider implements ComponentBuildingDefinitionProvider {

  public static final String CONFIG = "config";
  public static final String TEST = "test";
  public static final String BEFORE_TEST = "before-test";
  public static final String AFTER_TEST = "after-test";
  public static final String BEFORE_SUITE = "before-suite";
  public static final String AFTER_SUITE = "after-suite";

  private static final String SET_EVENT = "set-event";
  private static final String SET_NULL_PAYLOAD = "set-null-payload";

  private static ComponentBuildingDefinition.Builder baseDefinition =
      new ComponentBuildingDefinition.Builder().withNamespace(NAME_SPACE);

  @Override
  public void init() {}

  @Override
  public List<ComponentBuildingDefinition> getComponentBuildingDefinitions() {
    LinkedList<ComponentBuildingDefinition> componentBuildingDefinitions = new LinkedList<>();


    componentBuildingDefinitions.add(
                                     baseDefinition
                                         .withIdentifier(CONFIG)
                                         .withTypeDefinition(fromType(MunitModule.class))
                                         .withSetterParameterDefinition("parameterizations",
                                                                        fromChildConfiguration(Parameterizations.class)
                                                                            .withIdentifier("parameterizations").build())
                                         .withSetterParameterDefinition("ignore",
                                                                        fromSimpleParameter("ignore").withDefaultValue(false)
                                                                            .build())
                                         .withSetterParameterDefinition("minMuleVersion",
                                                                        fromSimpleParameter("minMuleVersion").build())
                                         .withSetterParameterDefinition("requiredProduct",
                                                                        fromSimpleParameter("requiredProduct").build())
                                         .build());

    componentBuildingDefinitions
        .add(baseDefinition.withIdentifier("parameterizations").withTypeDefinition(fromType(Parameterizations.class))
            .withSetterParameterDefinition("parameterizations",
                                           fromChildCollectionConfiguration(Parameterization.class).build())
            .withSetterParameterDefinition("file", fromSimpleParameter("file").build())
            .build());

    componentBuildingDefinitions
        .add(baseDefinition.withIdentifier("parameterization").withTypeDefinition(fromType(Parameterization.class))
            .withSetterParameterDefinition("parameters",
                                           fromChildConfiguration(Parameters.class).build())
            .build());

    componentBuildingDefinitions
        .add(baseDefinition.withIdentifier("parameters").withTypeDefinition(fromType(Parameterization.class))
            .withSetterParameterDefinition("parameters",
                                           fromChildCollectionConfiguration(Parameters.Parameter.class).build())
            .build());

    componentBuildingDefinitions
        .add(baseDefinition.withIdentifier("parameter").withTypeDefinition(fromType(Parameters.Parameter.class))
            .withSetterParameterDefinition("propertyName", fromSimpleParameter("propertyName").build())
            .withSetterParameterDefinition("value", fromSimpleParameter("value").build())
            .build());


    componentBuildingDefinitions.add(
                                     baseDefinition
                                         .withIdentifier(TEST)
                                         .withTypeDefinition(fromType(TestFlow.class))
                                         .withSetterParameterDefinition("ignore",
                                                                        fromSimpleParameter("ignore").withDefaultValue(false)
                                                                            .build())
                                         .withSetterParameterDefinition("description",
                                                                        fromSimpleParameter("description").withDefaultValue(EMPTY)
                                                                            .build())
                                         .withSetterParameterDefinition("expectedErrorType",
                                                                        fromSimpleParameter("expectedErrorType").build())
                                         .withSetterParameterDefinition("expectedException",
                                                                        fromSimpleParameter("expectedException").build())
                                         .withSetterParameterDefinition("expectedErrorDescription",
                                                                        fromSimpleParameter("expectedErrorDescription").build())
                                         .withSetterParameterDefinition("tags", fromSimpleParameter("tags").build())
                                         .withSetterParameterDefinition("enableFlowSources",
                                                                        fromChildCollectionConfiguration(List.class)
                                                                            .withWrapperIdentifier("enable-flow-sources")
                                                                            .build())
                                         .withSetterParameterDefinition("processorChains",
                                                                        fromChildCollectionConfiguration(MessageProcessorChain.class)
                                                                            .build())
                                         .withSetterParameterDefinition("timeOut", fromSimpleParameter("timeOut").build())
                                         .build());

    componentBuildingDefinitions
        .add(baseDefinition.withIdentifier("enable-flow-source").withTypeDefinition(fromType(FlowRef.class))
            .withSetterParameterDefinition("flow", fromSimpleReferenceParameter("value").build())
            .build());


    componentBuildingDefinitions.add(buildSimpleFlowDefinition(BEFORE_SUITE, BeforeSuite.class));
    componentBuildingDefinitions.add(buildSimpleFlowDefinition(BEFORE_TEST, BeforeTest.class));
    componentBuildingDefinitions.add(buildSimpleFlowDefinition(AFTER_TEST, AfterTest.class));
    componentBuildingDefinitions.add(buildSimpleFlowDefinition(AFTER_SUITE, AfterSuite.class));


    componentBuildingDefinitions.add(
                                     baseDefinition
                                         .withIdentifier(SET_NULL_PAYLOAD)
                                         .asPrototype()
                                         .withTypeDefinition(fromType(SetEventProcessor.class))
                                         .withSetterParameterDefinition("payload", fromSimpleParameter("payload").build())
                                         .build());

    componentBuildingDefinitions.add(
                                     baseDefinition
                                         .withIdentifier(SET_EVENT)
                                         .asPrototype()
                                         .withTypeDefinition(fromType(SetEventProcessor.class))
                                         .withSetterParameterDefinition("cloneOriginalEvent",
                                                                        fromSimpleParameter("cloneOriginalEvent").build())
                                         .withSetterParameterDefinition("payload",
                                                                        fromChildConfiguration(Payload.class)
                                                                            .withIdentifier("payload").build())
                                         .withSetterParameterDefinition("attributes",
                                                                        fromChildConfiguration(EventAttributes.class)
                                                                            .withIdentifier("attributes").build())
                                         .withSetterParameterDefinition("error",
                                                                        fromChildConfiguration(UntypedEventError.class)
                                                                            .withIdentifier("error").build())
                                         .withSetterParameterDefinition("variables",
                                                                        fromChildConfiguration(List.class)
                                                                            .withIdentifier("variables").build())
                                         .build());

    componentBuildingDefinitions.add(
                                     baseDefinition
                                         .withIdentifier("payload")
                                         .withTypeDefinition(fromType(Payload.class))
                                         .withSetterParameterDefinition("value", fromSimpleParameter("value").build())
                                         .withSetterParameterDefinition("mediaType", fromSimpleParameter("mediaType").build())
                                         .withSetterParameterDefinition("encoding", fromSimpleParameter("encoding").build())
                                         .build());

    componentBuildingDefinitions.add(
                                     baseDefinition
                                         .withIdentifier("attributes")
                                         .withTypeDefinition(fromType(EventAttributes.class))
                                         .withSetterParameterDefinition("value", fromSimpleParameter("value").build())
                                         .withSetterParameterDefinition("mediaType", fromSimpleParameter("mediaType").build())
                                         .withSetterParameterDefinition("encoding", fromSimpleParameter("encoding").build())
                                         .build());

    componentBuildingDefinitions.add(
                                     baseDefinition
                                         .withIdentifier("error")
                                         .withTypeDefinition(fromType(UntypedEventError.class))
                                         .withSetterParameterDefinition("typeId", fromSimpleParameter("id").build())
                                         .withSetterParameterDefinition("cause", fromSimpleParameter("exception").build())
                                         .build());

    componentBuildingDefinitions
        .add(baseDefinition.withIdentifier("variables").withTypeDefinition(fromType(List.class)).build());

    componentBuildingDefinitions.add(buildGenericPropertyDefinition("variable", Variable.class));

    componentBuildingDefinitions.add(buildDefaultProcessorChainDefinition("processorChain"));

    componentBuildingDefinitions.add(buildDefaultProcessorChainDefinition("behavior"));
    componentBuildingDefinitions.add(buildDefaultProcessorChainDefinition("execution"));
    componentBuildingDefinitions.add(buildDefaultProcessorChainDefinition("validation"));

    return componentBuildingDefinitions;
  }

  private ComponentBuildingDefinition buildSimpleFlowDefinition(String identifier, Class typeClass) {
    return baseDefinition.withIdentifier(identifier)
        .withTypeDefinition(fromType(typeClass))
        .withSetterParameterDefinition("description", fromSimpleParameter("description").build())
        .withSetterParameterDefinition("processors", fromChildCollectionConfiguration(Object.class).build())
        .build();
  }

  private ComponentBuildingDefinition buildGenericPropertyDefinition(String identifier, Class clazz) {
    return baseDefinition
        .withIdentifier(identifier)
        .withTypeDefinition(fromType(clazz))
        .withSetterParameterDefinition("key", fromSimpleParameter("key").build())
        .withSetterParameterDefinition("value", fromSimpleParameter("value").build())
        .withSetterParameterDefinition("mediaType", fromSimpleParameter("mediaType").build())
        .withSetterParameterDefinition("encoding", fromSimpleParameter("encoding").build())
        .build();
  }

  private ComponentBuildingDefinition buildDefaultProcessorChainDefinition(String name) {
    return baseDefinition.withIdentifier(name)
        .withTypeDefinition(fromType(MessageProcessorChain.class))
        .withObjectFactoryType(TestProcessorChainFactory.class)
        .withSetterParameterDefinition("name", fromFixedValue(name).build())
        .withSetterParameterDefinition("messageProcessors", fromChildCollectionConfiguration(Object.class).build())
        .build();
  }

}
