/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.common.util;

import org.mule.runtime.core.api.config.i18n.CoreMessages;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import jakarta.activation.DataHandler;
import jakarta.activation.FileDataSource;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.Properties;


/**
 * <code>IOUtils</code> contains useful methods for dealing with I/O operations This is based on the mule core class so we can
 * have control over it
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class IOUtils extends org.apache.commons.io.IOUtils {

  private static final Logger logger = LoggerFactory.getLogger(IOUtils.class);

  /**
   * Attempts to load a resource from the file system or from the classpath, in that order.
   *
   * @param resourceName The name of the resource to load
   * @param callingClass The Class object of the calling object
   * @return an URL to the resource or null if resource not found
   */
  public static URL getResourceAsUrl(final String resourceName, final Class callingClass) {
    return getResourceAsUrl(resourceName, callingClass, true, true);
  }

  /**
   * Attempts to load a resource from the file system or from the classpath, in that order.
   *
   * @param resourceName The name of the resource to load
   * @param callingClass The Class object of the calling object
   * @param tryAsFile - try to load the resource from the local file system
   * @param tryAsUrl - try to load the resource as a Url string
   * @return an URL to the resource or null if resource not found
   */
  public static URL getResourceAsUrl(final String resourceName, final Class callingClass, boolean tryAsFile, boolean tryAsUrl) {
    if (resourceName == null) {
      throw new IllegalArgumentException(CoreMessages.objectIsNull("Resource name").getMessage());
    }
    URL url = null;

    // Try to load the resource from the file system.
    if (tryAsFile) {
      try {
        File file = FileUtils.newFile(resourceName);
        if (file.exists()) {
          url = file.getAbsoluteFile().toURL();
        } else {
          logger.debug("Unable to load resource from the file system: " + file.getAbsolutePath());
        }
      } catch (Exception e) {
        logger.debug("Unable to load resource from the file system: " + e.getMessage());
      }
    }

    // Try to load the resource from the classpath.
    if (url == null) {
      try {
        url = (URL) AccessController
            .doPrivileged((PrivilegedAction) () -> ClassUtils.getResource(resourceName, callingClass));
        if (url == null) {
          logger.debug("Unable to load resource " + resourceName + " from the classpath");
        }
      } catch (Exception e) {
        logger.debug("Unable to load resource " + resourceName + " from the classpath: " + e.getMessage());
      }
    }

    if (url == null) {
      try {
        url = new URL(resourceName);
      } catch (MalformedURLException e) {
        // ignore
      }
    }
    return url;
  }

  public static Properties toProperties(File propertiesFile) throws IOException {
    if (propertiesFile == null || !propertiesFile.exists()) {
      throw new IOException("Invalid or missing file: " + propertiesFile);
    }
    return toProperties(new FileInputStream(propertiesFile));
  }

  public static Properties toProperties(URL url) throws IOException {
    if (url == null) {
      throw new IOException("Invalid file URL!");
    }
    return toProperties(url.openStream());
  }

  private static Properties toProperties(InputStream stream) throws IOException {
    Properties properties = new Properties();
    properties.load(stream);
    return properties;
  }

}
