package com.moengage.core;

import android.app.job.JobParameters;
import android.content.Context;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import com.moe.pushlibrary.models.Event;
import com.moe.pushlibrary.models.UserAttribute;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import java.util.ArrayList;
import org.json.JSONArray;
import org.json.JSONObject;

/**
 * Read events from Datapoints table -> Create a batch with batch-id and write to batch table ->
 * initiate data sync
 *
 * @author Umang Chamaria
 */
class CreatingDataBatchTask extends SDKTask {

  private String requestTime;
  private String batchId;
  private String requestId;

  private JobParameters jobParameters;
  private OnJobComplete jobComplete;

  CreatingDataBatchTask(Context context){
    super(context);
  }

  CreatingDataBatchTask(Context context, @Nullable OnJobComplete jobComplete,
      @Nullable JobParameters parameters) {
    super(context);
    this.jobParameters = parameters;
    this.jobComplete = jobComplete;
  }

  @Override public TaskResult execute() {
    try {
      Logger.v("CreatingDataBatchTask: executing task");
      if (!ConfigurationProvider.getInstance(mContext).isAppEnabled()) return null;
      trackInstallReferrerIfRequired();
      createBatchRequests();
      MoEDispatcher.getInstance(mContext).addTaskToQueue(new SendInteractionDataTask(mContext,
          jobComplete, jobParameters));
      Logger.v("CreatingDataBatchTask: completed task execution");
    } catch (Exception e) {
      Logger.f("CreatingDataBatchTask: execute() ", e);
    }
    Logger.v("CreatingDataBatchTask: completed execution");
    return null;
  }

  private void trackInstallReferrerIfRequired() {
    String referrer = MoEHelperUtils.getInstallReferrer(mContext);
    if (!TextUtils.isEmpty(referrer)) {
      JSONObject userJson = new JSONObject();
      try {
        userJson.put(MoEHelperConstants.PREF_KEY_INSTALL_REFERRER, referrer);
        MoEUtils.trackEventInternal(MoEConstants.EVENT_ACTION_USER_ATTRIBUTE, userJson, mContext);
      } catch (Exception e) {
        Logger.f("SendInteractionDataTask:setUserAttribute", e);
      }
      MoEHelperUtils.removeInstallReferrer(mContext);
    }
  }

  @Override public String getTaskTag() {
    return TAG_CREATE_DATA_BATCH;
  }

  @Override public boolean isSynchronous() {
    return true;
  }

  private void createBatchRequests() {
    ArrayList<Event> batchedData = null;
    for (; ; ) {
      //read events from datapoints table
      batchedData = MoEDAO.getInstance(mContext).getInteractionData(100);
      Logger.d("CreatingDataBatchTask : createBatchRequests:Fetching interaction data in batches");
      if (batchedData == null || batchedData.isEmpty()) {
        Logger.d("CreatingDataBatchTask : createBatchRequests: Found Nothing to send");
        return;
      }
      String interactionData = convertEventsToJSON(batchedData);
      if (interactionData == null) {
        return;
      }
      //write to batch table
      MoEDAO.getInstance(mContext).writeBatch(interactionData);
      //delete events from datapoint table
      MoEDAO.getInstance(mContext).deleteInteractionData(batchedData, mContext);
      batchedData.clear();
    }
  }

  @Nullable private String convertEventsToJSON(ArrayList<Event> eventList) {
    if (eventList == null || eventList.isEmpty()) {
      return null;
    }
    JSONArray jsonInteractions = new JSONArray();
    for (Event event : eventList) {
      try {
        jsonInteractions.put(new JSONObject(event.details));
      } catch (Exception e) {
        Logger.f("CreatingDataBatchTask:convertEventsToJSON", e);
      }
    }
    if (jsonInteractions.length() == 0) {
      return null;
    }
    JSONObject jsonInteractionsObject = new JSONObject();
    try {
      jsonInteractionsObject.put(MoEConstants.ATTR_INTERACTION_VIEWS_COUNT,
          jsonInteractions.length());
      jsonInteractionsObject.put(MoEConstants.ATTR_INTERACTION_VIEWS_INFO, jsonInteractions);
      JSONObject sdkMeta = getSDKMeta();
      if (sdkMeta != null) {
        jsonInteractionsObject.put(MoEConstants.ATTR_SDK_META, sdkMeta);
      }
      JSONObject sdkIdentifiers = getSDKIdentifiersJSON();
      if (sdkIdentifiers != null) {
        jsonInteractionsObject.put(MoEConstants.ATTR_SDK_IDENTIFIERS, sdkIdentifiers);
      }
    } catch (Exception e) {
      Logger.f("CreatingDataBatchTask:convertEventsToJSON", e);
      return null;
    }
    return jsonInteractionsObject.toString();
  }

  @Nullable private JSONObject getSDKIdentifiersJSON() {
    try {
      JSONObject identifierJSON = new JSONObject();
      UserAttribute userAttributeUniqueId = MoEDAO.getInstance(mContext)
          .getUserAttributesForKey(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID);
      if (userAttributeUniqueId != null) {
        identifierJSON.put(MoEConstants.ATTR_MOE_USER_ID, userAttributeUniqueId.userAttributeValue);
      }
      String segmentAnonymousId =
          ConfigurationProvider.getInstance(mContext).getSegmentAnonymousId();
      if (!TextUtils.isEmpty(segmentAnonymousId)) {
        identifierJSON.put(MoEConstants.ATTR_SEGMENT_ID, segmentAnonymousId);
      }
      /*requestId =  MoEUtils.getRequestId(batchId +
          requestTime + ConfigurationProvider.getInstance(mContext).getCurrentUserId());
      identifierJSON.put(MoEConstants.REQUEST_ATTR_REQUEST_ID, requestId);*/
      if (identifierJSON.length() != 0) {
        return identifierJSON;
      }
    } catch (Exception e) {
      Logger.f("CreatingDataBatchTask: getSDKIdentifiersJSON() ", e);
    }
    return null;
  }

  @Nullable private JSONObject getSDKMeta() {
    try {
      JSONObject metaJSON = new JSONObject();
      batchId = MoEUtils.getBatchId();
      metaJSON.put(MoEConstants.ATTR_BATCH_ID, batchId);
      ConfigurationProvider provider = ConfigurationProvider.getInstance(mContext);
      JSONObject preferences = new JSONObject();
      // add opt-out preferences only if opted out
      if (provider.isPushNotificationOptedOut()) {
        preferences.put(MoEConstants.REQUEST_ATTR_PUSH_PREFERENCE,
            !provider.isPushNotificationOptedOut());
      }
      if (provider.isInAppOptedOut()) {
        preferences.put(MoEConstants.REQUEST_ATTR_IN_APP_PREFERENCE, !provider.isInAppOptedOut());
      }
      if (provider.isDataTrackingOptedOut()) {
        preferences.put(MoEConstants.REQUEST_ATTR_DATA_TRACKING_PREFERENCE,
            !provider.isDataTrackingOptedOut());
      }
      if (preferences.length() != 0) {
        metaJSON.put(MoEConstants.REQUEST_ATTR_DEVICE_PREFERENCE, preferences);
      }
      /*requestTime = MoEUtils.getCurrentISOTime();
      metaJSON.put(MoEConstants.REQUEST_ATTR_REQUEST_TIME, requestTime);*/

      if (metaJSON.length() != 0) {
        return metaJSON;
      }
    } catch (Exception e) {
      Logger.f("CreatingDataBatchTask: getSDKMeta() ", e);
    }
    return null;
  }
}
