package com.moengage.mi;

import android.app.ActivityManager;
import android.app.ActivityManager.RunningAppProcessInfo;
import android.app.Application;
import android.content.Context;
import android.os.Process;
import android.text.TextUtils;
import com.moengage.core.Logger;
import com.moengage.core.MoEDispatcher;
import com.xiaomi.mipush.sdk.MiPushClient;
import java.util.List;

/**
 * @author Umang Chamaria
 */
class MiPushController {

  private static final String TAG = "MiPushController";
  private MiPushEvaluator evaluator;
  private MiPushRepository repository;

  private static MiPushController instance;

  private MiPushController() {
    evaluator = new MiPushEvaluator();
    repository = new MiPushRepository();
  }

  static MiPushController getInstance() {
    if (instance == null) {
      instance = new MiPushController();
    }
    return instance;
  }

  void initialiseMiPush(Application application) {
    try {
      if (!evaluator.shouldInitialiseMiPush(repository.isMiPushEnabled(),
          repository.getDeviceManufacturer())) {
        Logger.e(TAG
            + " initialiseMiPush() : Either it is not a Xiaomi device or Mi Push is "
            + "disabled.");
        return;
      }
      if (repository.isPushNotificationOptedOut(application.getApplicationContext())){
        Logger.e( TAG + " initialiseMiPush() : Push notification disabled will not register for "
            + "push token.");
        return;
      }
      String miAppId = repository.getMiAppId();
      String miAppKey = repository.getMiAppKey();
      if (TextUtils.isEmpty(miAppId)){
        Logger.e( TAG + " initialiseMiPush() : Mi App Id is empty cannot register for Mi Push");
        return;
      }
      if (TextUtils.isEmpty(miAppKey)){
        Logger.e( TAG + " initialiseMiPush() : Mi App Key is empty cannot register for Mi Push");
        return;
      }
      if (!shouldInit(application.getApplicationContext())) {
        Logger.v(TAG + " initialiseMiPush() : Mi Push service already running need to re-initialise");
        return;
      }
      MiPushClient.registerPush(application, miAppId, miAppKey);
    } catch (Exception e) {
      Logger.f( TAG + " initialiseMiPush() : Exception: ", e);
    }
  }

  private boolean shouldInit(Context context) {
    ActivityManager am = ((ActivityManager) context.getSystemService(Context.ACTIVITY_SERVICE));
    if (am == null) return false;
    List<RunningAppProcessInfo> processInfos = am.getRunningAppProcesses();
    String mainProcessName = context.getPackageName();
    int myPid = Process.myPid();
    for (RunningAppProcessInfo info : processInfos) {
      if (info.pid == myPid && mainProcessName.equals(info.processName)) {
        return true;
      }
    }
    return false;
  }

  void processPushToken(Context context, String pushToken) {
    Logger.v(TAG + " processPushToken() : Token: " + pushToken);
    if (TextUtils.isEmpty(pushToken)){
      Logger.e( TAG + " processPushToken() : Passed push token is empty no processing required.");
      return;
    }
    if (evaluator.shouldSendTokenToServer(repository.getSavedToken(context), pushToken,
        repository.hasSentMiPushTokenToServer(context))){
      Logger.v(TAG + " processPushToken() : Will try to send token to server. Token: " + pushToken);
      MoEDispatcher.getInstance(context).getDeviceAddManager().registerMiToken(context);
    }else {
      Logger.v(TAG + " processPushToken() : Token already sent to server. Need not resend again.");
    }
    repository.savePushToken(context, pushToken);
  }
}
