package com.moengage.addon.trigger;

import android.app.AlarmManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.WorkerThread;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.PayloadBuilder;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.core.segmentation.FilterEvaluator;
import com.moengage.push.PushManager;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.TimeZone;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */

class DTController {

  public enum NETWORK_CALL_TYPE {
    SYNC_API, USER_IN_SEGMENT
  }

  private static final String TAG = "DTController";

  private HashSet<String> mTriggerEvents;

  private Context mContext;

  private DTConditionEvaluator mConditionEvaluator;

  private DTDAO dbHandler;

  private DTController(@NonNull Context context) {
    mContext = context;
    dbHandler = DTDAO.getInstance(mContext);
    updateDTCache();
    mConditionEvaluator = new DTConditionEvaluator();
  }

  private static DTController _INSTANCE;

  public static DTController getInstance(Context context) {
    if (_INSTANCE == null) {
      _INSTANCE = new DTController(context);
    }
    return _INSTANCE;
  }

  @WorkerThread void updateDTCache() {
    try {
      mTriggerEvents = dbHandler.getTriggerEventsIfAny();
      if (mTriggerEvents != null){
        Logger.v(TAG + " updateDTCache() : device trigger event " + mTriggerEvents.toString());
      }else {
        Logger.v(TAG + " updateDTCache() : no device trigger events");
      }
      dbHandler.removeExpiredCampaigns();
    } catch (Exception e) {
      Logger.f( TAG + " updateDTCache() : ", e);
    }
  }

  @Nullable TriggerMessage getCampaignToShown(String eventName, JSONObject eventAttributes) {
    try {
      LinkedList<TriggerMessage> campaignsForEvent =
          dbHandler.getCampaignsForEvent(eventName);
      if (campaignsForEvent == null) return null;
      for (TriggerMessage campaign : campaignsForEvent) {
        try {
          Logger.v(TAG + " getCampaignToShown() : evaluating conditions for campaign id: " +
              campaign.campaignId);
          campaign.dump();
          if (!mConditionEvaluator.canShowTriggerMessage(campaign, getLastSyncTime(),
              System.currentTimeMillis())) continue;
          //campaign payload sanity check
          if (campaign.campaignPayload == null) continue;
          if (!campaign.campaignPayload.has(DTConstants.RESPONSE_ATTR_CONDITION)) return campaign;
          JSONObject conditions = campaign.campaignPayload.getJSONObject(DTConstants
              .RESPONSE_ATTR_CONDITION);
          JSONArray filters =
              conditions.getJSONArray(DTConstants.RESPONSE_ATTR_FILTER);
          String filterOperator = campaign.campaignPayload.optString(DTConstants
              .RESPONSE_ATTR_FILTER_OPERATOR, DTConstants.FILTER_OPERATOR_AND);
          if (filters == null) continue;
          // campaign without filters
          if (filters.length() == 0){
            Logger.v(TAG + " getCampaignToShown() : campaign does not have filters only trigger "
                + "event will try to show campaign with id: " + campaign.campaignId);
            return campaign;
          }

          ArrayList<Boolean> isSatisfied = new ArrayList<>(filters.length());
          //evaluating conditions
          for (int i = 0; i < filters.length(); i++) {
            JSONObject filter = filters.getJSONObject(i);
            boolean filterEvaluationResult = new FilterEvaluator(filter, eventAttributes)
                .evaluate();
            Logger.v(TAG + " getCampaignToShown() : filter evaluation result for filter : " +
                filter.toString() + "is: " + filterEvaluationResult);
            isSatisfied.add(filterEvaluationResult);
          }
          // checking aggregated results of each condition
          switch (filterOperator){
            case DTConstants.FILTER_OPERATOR_AND:
              if (!isSatisfied.contains(Boolean.FALSE)) return campaign;
              break;
            case DTConstants.FILTER_OPERATOR_OR:
              if (isSatisfied.contains(Boolean.TRUE)) return campaign;
              break;
          }
        } catch (Exception e) {
          Logger.f(TAG + "getCampaignToShown() : inside for loop ", e);
        }
      }
    } catch (Exception e) {
      Logger.f(TAG + "getCampaignToShown() : ", e);
    }
    return null;
  }


  /**
   * Show or schedule notification when user is online
   *
   * @param message Campaign object
   */
  void showOrScheduleNotificationOnline(TriggerMessage message) {
    if (mConditionEvaluator.isPayloadEmpty(message)) return;
    if (message.rules.showDelay == 0) {
      showPushAndUpdateCounter(message);
    } else {
      schedulePushNotification(message, false);
    }
  }

  /**
   * Try to show notification offline is possible
   *
   * @param message Campaign Object
   * @throws JSONException pass on the exception to the caller
   */
  void showOrScheduleNotificationOffline(TriggerMessage message) throws JSONException {
    if (!message.rules.shouldShowOffline || mConditionEvaluator.isPayloadEmpty(message)) return;
    // identifier to track notifications shown offline
    message.payload.put("shownOffline", true);

    // show or schedule notification
    if (message.rules.showDelay == 0) {
      tryShowingNotificationOffline(message);
    } else {
      schedulePushNotification(message, true);
    }
  }

  /**
   * Update campaign counters.
   *
   * @param message Campaign object
   */
  private void updateCampaignState(TriggerMessage message) {
    message.state.showCount++;
    message.state.lastShowTime = System.currentTimeMillis();
    ConfigurationProvider.getInstance(mContext).saveLastDTShowTime(message.state.lastShowTime);
    dbHandler.updateCampaignState(message);
  }

  /**
   * Try to show offline notification if DND is not active.
   *
   * @param message Campaign object
   */
  void tryShowingNotificationOffline(TriggerMessage message){
    try {
      Calendar rightNow = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
      int currentHour = rightNow.get(Calendar.HOUR_OF_DAY);
      int currentMinutes = rightNow.get(Calendar.MINUTE);

      if (mConditionEvaluator.isDNDActive(getDNDStartTime(), getDNDEndTime(), currentHour,
          currentMinutes) && !message.rules.shouldIgnoreDND) {
        Logger.e( TAG + " tryShowingNotificationOffline() : dnd is active cannot show notification");
        return;
      }

      String campaignId = message.payload.getString(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID);

      // event for backend stats job
      PayloadBuilder builder = new PayloadBuilder();
      builder.putAttrString(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID, campaignId);
      MoEHelper.getInstance(mContext).trackEvent("NOTIFICATION_OFFLINE_MOE", builder.build());

      // adding additional timestamp to by pass duplicate check
      message.payload.put(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID, campaignId + "DTSDK" +
          System.currentTimeMillis());
      showPushAndUpdateCounter(message);
    } catch (Exception e) {
      Logger.e( TAG + " tryShowingNotificationOffline() : ", e);
    }
  }

  void showScheduledNotification(String campaignId, boolean isOffline, String payload){
    try {
      TriggerMessage message = dbHandler.getCampaignById(campaignId);
      if (message != null){
        message.payload = new JSONObject(payload);
        if (isOffline) tryShowingNotificationOffline(message);
        else showPushAndUpdateCounter(message);
      }else {
        Logger.e( TAG + " showScheduledNotification() : did not find campaign with id: " +
            campaignId);
      }
    } catch (Exception e) {
      Logger.e(TAG + " showScheduledNotification() : ", e);
    }
  }

  /**
   * Pass push payload to push module
   *
   * @param message Campaign object
   */
  void showPushAndUpdateCounter(TriggerMessage message) {
    try {
      if (message.rules.expiryTime < System.currentTimeMillis() || mConditionEvaluator.isPayloadEmpty(message)) {
        Logger.f(TAG + " showPushAndUpdateCounter() : cannot show trigger message for campaign "
            + "id: " + message.campaignId);
        return;
      }
      // convert payload from JSON to Bundle
      Bundle pushPayload = MoEUtils.jsonToBundle(message.payload);
      if (pushPayload == null){
        Logger.f( TAG + " showPushAndUpdateCounter() : could not convert json to bundle, cannot "
            + "show campaign for campaign id: " + message.campaignId);
        return;
      }
      //pass payload to push module
      PushManager.getInstance().getPushHandler().handlePushPayload(mContext, pushPayload);
      // update count
      updateCampaignState(message);
    } catch (Exception e) {
      Logger.f( TAG + " showPushAndUpdateCounter() : ", e);
    }
  }

  /**
   * Schedule alarm to show notification later.
   *
   * @param message Campaign object
   * @param isOffline true if online notification was scheduled, else false
   */
  void schedulePushNotification(TriggerMessage message, boolean isOffline) {
    try {
      Logger.v(TAG + " schedulePushNotification() : will schedule notification for campaign id: "
          + message.campaignId);
      Intent laterIntent = new Intent(mContext, DTIntentService.class);
      laterIntent.putExtra(DTConstants.EXTRA_ATTR_OFFLINE, isOffline);
      laterIntent.putExtra(DTConstants.EXTRA_SHOW_NOTIFICATION, message.campaignId);
      laterIntent.putExtra(DTConstants.EXTRA_NOTIFICATION_PAYLOAD, message.payload.toString());
      laterIntent.setAction(DTConstants.ACTION_SHOW_NOTIFICATION);
      final int _id = (int)System.currentTimeMillis();
      PendingIntent alarmPendingIntent =
          PendingIntent.getService(mContext, _id, laterIntent, PendingIntent.FLAG_UPDATE_CURRENT);
      AlarmManager alarmManager = (AlarmManager) mContext.getSystemService(Context.ALARM_SERVICE);
      if (alarmManager != null) {
        alarmManager.set(AlarmManager.RTC_WAKEUP, System.currentTimeMillis() + message.rules.showDelay,
            alarmPendingIntent);
      }
      //track event to log campaign was scheduled
      PayloadBuilder builder = new PayloadBuilder();
      builder.putAttrString("campaign_id", message.campaignId);
      MoEHelper.getInstance(mContext).trackEvent("DT_CAMPAIGN_SCHEDULED", builder.build());
    } catch (Exception e) {
      Logger.f( TAG + " schedulePushNotification() : ", e);
    }
  }

  @Nullable HashSet<String> getTriggerEvents() {
    return mTriggerEvents;
  }

  long getGlobalMinimumDelay(){
    return ConfigurationProvider.getInstance(mContext).getDTMinimumDelay();
  }

  long getLastShowTime(){
    return ConfigurationProvider.getInstance(mContext).getDTLastShowTime();
  }

  long getDNDStartTime(){
    return ConfigurationProvider.getInstance(mContext).getDTDNDStartTime();
  }

  long getDNDEndTime(){
    return ConfigurationProvider.getInstance(mContext).getDNDEndTime();
  }

  long getLastSyncTime(){
    return ConfigurationProvider.getInstance(mContext).getDTLastSyncTime();
  }
}
