package com.moengage.addon.trigger;

import android.support.annotation.Nullable;
import com.moengage.core.Logger;
import java.util.HashSet;

/**
 * @author Umang Chamaria
 */
public class DTConditionEvaluator {

  private static final String TAG = "DTConditionEvaluator";

  /**
   * Checks whether a given event is a device trigger event or not
   *
   * @param eventName triggered event name
   * @return true if the event is a device trigger event else false
   */
  boolean isDTEvent(@Nullable HashSet<String> triggerEvents, String eventName) {
    return triggerEvents != null && triggerEvents.contains(eventName);
  }

  boolean shouldShowTrigger(long lastShowTime, long globalMinimumDelay, long currentTime) {
    return lastShowTime + globalMinimumDelay < currentTime;
  }

  /**
   * Check whether DND is currently active or not.
   *
   * @return false if DND is not active, else true
   */
  boolean isDNDActive(long mDNDStartTime, long mDNDEndTime, int currentHour, int currentMinutes) {
    if (mDNDEndTime == -1
        || mDNDStartTime == -1
        || (mDNDStartTime == 0 && mDNDEndTime == 0)
        || mDNDEndTime == mDNDStartTime) {
      return false;
    }

    // get dnd start hours and minutes
    int dndStartTimeHours = 0;
    int dndStartTimeMinutes = 0;
    if (mDNDStartTime != 0) {
      dndStartTimeHours = (int) mDNDStartTime / 100;
      dndStartTimeMinutes = (int) mDNDStartTime % 100;
    }

    // get dnd stop hours and minutes
    int dndStopTimeHours = 0;
    int dndStopTimeMinutes = 0;
    if (mDNDEndTime != 0) {
      dndStopTimeHours = (int) mDNDEndTime / 100;
      dndStopTimeMinutes = (int) mDNDEndTime % 100;
    }

    //dnd spanning over 2 different days
    if (dndStartTimeHours > dndStopTimeHours) {
      if (dndStartTimeHours < currentHour || dndStopTimeHours > currentHour) return true;
      if (dndStartTimeHours == currentHour) return currentMinutes >= dndStartTimeMinutes;
      if (dndStopTimeHours == currentHour) return currentMinutes <= dndStopTimeMinutes;
    }
    //dnd on the same day
    else if (dndStartTimeHours < dndStopTimeHours) {
      if (currentHour > dndStartTimeHours && currentHour < dndStopTimeHours) return true;
      if (dndStartTimeHours == currentHour) return currentMinutes >= dndStartTimeMinutes;
      if (dndStopTimeHours == currentHour) return currentMinutes <= dndStopTimeMinutes;
    }
    // start and end has same hours
    else if (dndStartTimeHours == dndStopTimeHours) {
      return currentHour == dndStartTimeHours
          && currentMinutes >= dndStartTimeMinutes
          && currentMinutes <= dndStopTimeMinutes;
    }

    return false;
  }

  /**
   * Check whether a campaign is fit to be shown or not.
   *
   * @param triggerMessage campaign details.
   * @return true is the campaign is fit to be shown else false
   */
  boolean canShowTriggerMessage(TriggerMessage triggerMessage, long lastSyncTime, long
      currentTime) {
    //checking if active
    if (!triggerMessage.campaignType.equals(DTConstants.CAMPAIGN_TYPE_GENERAL)) {
      Logger.f(TAG
          + " canShowTriggerMessage() : "
          + triggerMessage.campaignId
          + " is not of type general. "
          + "Cannot show");
      return false;
    }
    if (!triggerMessage.state.isActive
        || triggerMessage.rules.expiryTime < currentTime) {
      Logger.f(TAG
          + " canShowTriggerMessage() : "
          + triggerMessage.campaignId
          + " is no longer active "
          + "cannot show");
      return false;
    }
    if (triggerMessage.rules.maxSyncDelay + lastSyncTime < currentTime && lastSyncTime != 0) {
      Logger.f(TAG
          + " canShowTriggerMessage() : "
          + triggerMessage.campaignId
          + " has not been updated"
          + " in a while. Cannot show without update");
      return false;
    }
    if (triggerMessage.rules.minimumDelay + triggerMessage.state.lastShowTime
        > currentTime) {
      Logger.f(TAG
          + " canShowTriggerMessage() : "
          + triggerMessage.campaignId
          + " was shown recently. "
          + "Cannot show now");
      return false;
    }
    if (triggerMessage.rules.maxCount != -9090909
        && triggerMessage.rules.maxCount <= triggerMessage.state.showCount) {
      Logger.e(TAG
          + "canShowTriggerMessage() : "
          + triggerMessage.campaignId
          + " has been shown "
          + "maximum number of times. Cannot be shown again");
      return false;
    }
    return true;
  }

  /**
   * Checks whether notification payload is empty or not
   *
   * @param message Campaign object
   * @return true if campaign has notification payload
   */
  boolean isPayloadEmpty(TriggerMessage message) {
    return message.payload == null || message.payload.length() == 0;
  }
}
