package com.game.sdk.ui.payment;

import android.graphics.Bitmap;
import android.os.Bundle;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.text.TextUtils;
import android.util.Log;
import android.view.View;
import android.webkit.WebView;

import com.android.billingclient.api.BillingClient;
import com.android.billingclient.api.BillingClientStateListener;
import com.android.billingclient.api.BillingResult;
import com.android.billingclient.api.ConsumeResponseListener;
import com.android.billingclient.api.Purchase;
import com.android.billingclient.api.PurchasesUpdatedListener;
import com.android.billingclient.api.SkuDetails;
import com.android.billingclient.api.SkuDetailsResponseListener;
import com.game.sdk.comon.cmd.CmdPaymentV3;
import com.game.sdk.comon.config.GameConfigs;
import com.game.sdk.comon.game.BaseDialogWebFragment;
import com.game.sdk.comon.js.JsBase;
import com.game.sdk.comon.js.JsPayment;
import com.game.sdk.comon.listener.IWebViewClientListener;
import com.game.sdk.comon.object.BaseObj;
import com.game.sdk.comon.object.ListPurchaseHistoryObj;
import com.game.sdk.comon.object.ProductObj;
import com.game.sdk.comon.object.PurchaseHistoryObj;
import com.game.sdk.comon.object.err.VerifyPurchaseErrObj;
import com.game.sdk.comon.object.request.VerifyPurchaseRequestObj;
import com.game.sdk.comon.object.response.VerifyPurchaseResponseObj;
import com.game.sdk.comon.presenter.BaseView;
import com.game.sdk.comon.sharePref.PrefManager;
import com.game.sdk.comon.tracking.TrackingUtil;
import com.game.sdk.comon.utils.DeviceUtils;
import com.game.sdk.comon.utils.DialogUtils;
import com.game.sdk.comon.utils.LogUtils;
import com.game.sdk.comon.utils.PurchaseUtils;
import com.game.sdk.comon.utils.Utils;
import com.mobgame.R;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

public class PaymentFragment extends BaseDialogWebFragment {
    public static String TAG = PaymentFragment.class.getSimpleName();
    IPaymentPresenter iPaymentPresenter;
    String account_id;
    int current;
    PurchaseHistoryObj currentPurchase;
    PaymentGGSuccess paymentGGSuccess;
    boolean sendHistory;
    ArrayList<IPaymentPresenter> listPresenter;

    List<SkuDetails> skuDetails;
    ProductObj productObj;
    com.android.billingclient.api.Purchase purchase;


    public static PaymentFragment newInstance(String url, PaymentGGSuccess paymentGGSuccess) {
        PaymentFragment frag = new PaymentFragment();
        Bundle args = new Bundle();
        args.putString(URL_WEBVIEW, url);
        LogUtils.d(TAG, url);
        frag.setArguments(args);
        frag.setPaymentGGSuccess(paymentGGSuccess);
        return frag;
    }

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);


        CmdPaymentV3.getInstance().initIABv3(getActivity(), new BillingClientStateListener() {
            @Override
            public void onBillingSetupFinished(@NonNull BillingResult billingResult) {
                int responseCode = billingResult.getResponseCode();
                String debugMessage = billingResult.getDebugMessage();
                Log.d(TAG, "onBillingSetupFinished: " + responseCode + " " + debugMessage);
            }

            @Override
            public void onBillingServiceDisconnected() {
                Log.d(TAG, "onBillingServiceDisconnected: ");
            }
        }, new PurchasesUpdatedListener() {
            @Override
            public void onPurchasesUpdated(@NonNull BillingResult billingResult, @Nullable List<Purchase> list) {
                int responseCode = billingResult.getResponseCode();
                String debugMessage = billingResult.getDebugMessage();
                Log.d(TAG, "onPurchasesUpdated:  " + responseCode + " " + debugMessage);
                switch (responseCode) {
                    case BillingClient.BillingResponseCode.OK:
                        processPurchases(list);
                        break;
                    case BillingClient.BillingResponseCode.USER_CANCELED:
                        Log.i(TAG, "onPurchasesUpdated: User canceled the purchase");
                        skuDetails.clear();

                        break;
                    case BillingClient.BillingResponseCode.ITEM_UNAVAILABLE:
                        Log.i(TAG, "onPurchasesUpdated: This item is unavailable");

                        break;
                    case BillingClient.BillingResponseCode.ITEM_ALREADY_OWNED:
                        Log.i(TAG, "onPurchasesUpdated: The user already owns this item");

                        break;
                    case BillingClient.BillingResponseCode.DEVELOPER_ERROR:
                        Log.e(TAG, "onPurchasesUpdated: Developer error means that Google Play " +
                                "does not recognize the configuration. If you are just getting started, " +
                                "make sure you have configured the application correctly in the " +
                                "Google Play Console. The SKU product ID must match and the APK you " +
                                "are using must be signed with release keys."
                        );
                        break;
                    default:
                        Log.i(TAG, "onPurchasesUpdated: " + debugMessage);

                }
            }
        });
    }


    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        try {
            TrackingUtil.getInstance().trackPaymentScreenOpened();
            account_id = GameConfigs.getInstance().getUser().getAccount().getAccountId();
            listPresenter = new ArrayList<>();
            skuDetails = new ArrayList<>();
        } catch (Exception e) {
            e.printStackTrace();
        }
        current = -1;
        sendHistory = false;

    }

    @Override
    public void onResume() {
        super.onResume();
        if (sendHistory) {
            sendHistory = true;
//            checkPurchaseHistory();
        }
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        CmdPaymentV3.getInstance().billingEndConnection();
    }

    public void sendHistory(PurchaseHistoryObj obj) {
        IPaymentPresenter presenter = new PaymentPresenterImpl(new BaseView() {
            @Override
            public void showProgress(String message) {

            }

            @Override
            public void hideProgress() {

            }

            @Override
            public void success(Object x) {
                if (x instanceof VerifyPurchaseResponseObj) {
                    VerifyPurchaseResponseObj responseObj = (VerifyPurchaseResponseObj) x;
                    int status = responseObj.getData().getStatus();
                    if (status == 3 || status == 2 || status == 17)
                        TrackingUtil.getInstance().trackPaymentVerifySuccess(responseObj.getData());
                    else
                        TrackingUtil.getInstance().trackPaymentVerifyFail(obj.getOrder_no(), "code", String.valueOf(status), responseObj.getData().getDescription());
                    obj.setAccount_id(account_id);
                    PurchaseUtils.removeSuccessPurchase(obj, mActivity);
                    current -= 1;
                    checkPurchaseHistory();
                    LogUtils.e(TAG, "Verify Success - " + obj.getOrder_no());
                }
            }

            @Override
            public void error(Object o) {
                if (o instanceof VerifyPurchaseErrObj) {
                    VerifyPurchaseErrObj err = (VerifyPurchaseErrObj) o;
                    LogUtils.e(TAG, "Verify failed - " + obj.getOrder_no() + " :" + err.getMessage());
                    checkPurchaseHistory();
                }
                if (o instanceof BaseObj) {
                    BaseObj err = (BaseObj) o;
                    TrackingUtil.getInstance().trackPaymentVerifyFail(obj.getOrder_no(), "http", String.valueOf(err.getStatus()), err.getMessage());
                }

            }
        });
        TrackingUtil.getInstance().trackPaymentBeforVerify(obj.getOrder_no());
        listPresenter.add(presenter);
        onVerifyPurchase(presenter, obj.getReceipt(), obj.getOrder_no());
    }

    public void checkPurchaseHistory() {
        current += 1;
        ListPurchaseHistoryObj listPurchaseHistoryObj = PrefManager.getHistoryPurchase(mActivity, account_id);
        if (listPurchaseHistoryObj.getData() == null || listPurchaseHistoryObj.getData().size() == 0 || listPurchaseHistoryObj.getData().size() <= current && current >= 0)
            return;
        PurchaseHistoryObj obj = listPurchaseHistoryObj.getData().get(current);
        sendHistory(obj);
    }


    public void onVerifyPurchase(IPaymentPresenter presenter, VerifyPurchaseRequestObj.Receipt receipt, String order_no) {
        VerifyPurchaseRequestObj obj = new VerifyPurchaseRequestObj();
        obj.setOrder_no(order_no);
        obj.setMethod("2");
        obj.setReceipt(receipt);
        obj.setApp_version(Utils.getGameVersion(mActivity));
        obj.setSdk_version(Utils.getSDKVersion(mActivity));
        obj.setDevice_name(DeviceUtils.getDevice());
        obj.setDevice_os(DeviceUtils.getOSInfo());
        obj.setResolution(DeviceUtils.getResolution(mActivity));
        obj.setNetwork(Utils.getNetwork(mActivity));
        obj.setAdvertising_id(DeviceUtils.getAdvertisingID(mActivity));
        obj.setAppsflyer_id(DeviceUtils.getAppsflyerUID(mActivity));
        obj.setLocale(GameConfigs.getInstance().getLang());
        if (presenter != null)
            presenter.verifyPurchase(obj);
    }

    @Override
    protected JsBase getJsHandler() {
        return new JsPayment(new JsPayment.Listener() {
            @Override
            public void onPaymentStartIAPV3(String params) {
                Log.d(TAG, "onPaymentStartIAPV3: " + params);

                productObj = new ProductObj(params);
                String noOrder = productObj.getOrder_no();
                String productId = productObj.getProduct_id();

                if (TextUtils.isEmpty(noOrder)) {
                    DialogUtils.showErrorDialog(mActivity, mActivity.getString(R.string.err_and_try_again_late));
                    return;
                }
                CmdPaymentV3.getInstance().querySkuDetails(productId, skuDetailsResponseListener);
            }

            @Override
            public void onCloseWindow() {
                dismiss();
            }

            @Override
            public void onOpenWindow() {

            }

            @Override
            public void onBackToWindowns() {
                goBack();
            }
        });
    }

    private void processPurchases(List<com.android.billingclient.api.Purchase> purchasesList) {
        if (purchasesList != null) {
            Log.d(TAG, "processPurchases: " + purchasesList.size() + " purchase(s)");
            for (com.android.billingclient.api.Purchase purchase : purchasesList) {
                handlePurchase(purchase);
            }

        } else {
            Log.d(TAG, "processPurchases: with no purchases");
            Log.d(TAG, "onPurchasesUpdated: null purchase list");
        }

    }

    private void handlePurchase(com.android.billingclient.api.Purchase purchase) {
/*      UNSPECIFIED_STATE = 0;
        PURCHASED = 1;
        PENDING = 2;
*/
        Log.i(TAG, "handlePurchase: " + purchase.getPurchaseState());

        if (purchase.getPurchaseState() == com.android.billingclient.api.Purchase.PurchaseState.PURCHASED) {
            Log.i(TAG, "PURCHASE IS PURCHASED ");
            this.purchase = purchase;
            CmdPaymentV3.getInstance().consumeAsyncPurchase(purchase, new ConsumeResponseListener() {
                @Override
                public void onConsumeResponse(@NonNull BillingResult billingResult, @NonNull String s) {


                    skuDetails.clear();
                    if (billingResult.getResponseCode() == BillingClient.BillingResponseCode.OK) {

                        currentPurchase = new PurchaseHistoryObj();
                        String character_id = PrefManager.getString(mActivity, com.game.sdk.comon.constants.Constants.ROLE_ID, "");
                        String area_id = PrefManager.getString(mActivity, com.game.sdk.comon.constants.Constants.AREA_ID, "");
                        currentPurchase.setArea_id(area_id);
                        currentPurchase.setCharactor_id(character_id);
                        currentPurchase.setOrder_no(productObj.getOrder_no());
                        Log.i(TAG, "currentPurchase: " + currentPurchase.getOrder_no());

                        try {
                            JSONObject joReceipt = new JSONObject(purchase.getOriginalJson());
                            VerifyPurchaseRequestObj.Receipt rec = new VerifyPurchaseRequestObj.Receipt();
                            rec.setData(joReceipt.toString());
                            rec.setItemType("inapp");
                            rec.setSignature(purchase.getSignature());
                            currentPurchase.setReceipt(rec);
                        } catch (Exception e) {
                            LogUtils.e("Err Purchase", e.getMessage());
                        }
                        currentPurchase.setIs_send(false);
                        currentPurchase.setAccount_id(account_id);
                        LogUtils.d("RECEIPT 1", purchase.getOriginalJson());
                        for (IPaymentPresenter p : listPresenter) {
                            if (p != null)
                                p.cancelRequest();
                        }
                        PrefManager.saveUsePurchaseHistory(mActivity, account_id, currentPurchase);
                        Log.i(TAG, "currentPurchase: " + currentPurchase.getOrder_no());
                        if (paymentGGSuccess != null) {
                            dismiss();
                            paymentGGSuccess.onPaymentGGSuccess(currentPurchase);
                        }
                    }
                }
            });


        } else if (purchase.getPurchaseState() == com.android.billingclient.api.Purchase.PurchaseState.PENDING) {
            Log.i(TAG, "PURCHASE IS PENDING ");
        }
    }


    private SkuDetailsResponseListener skuDetailsResponseListener = new SkuDetailsResponseListener() {
        @Override
        public void onSkuDetailsResponse(@NonNull BillingResult billingResult, @Nullable List<SkuDetails> list) {
            int responseCode = billingResult.getResponseCode();
            String debugMessage = billingResult.getDebugMessage();

            switch (responseCode) {
                case BillingClient.BillingResponseCode.OK:
                    Log.d(TAG, "onSkuDetailsResponse: " + responseCode);
                    if (list != null) {
                        skuDetails = list;
                        CmdPaymentV3.getInstance().launchBillingFlow(getActivity(), skuDetails.get(skuDetails.size() - 1));
                    }
                    break;
                case BillingClient.BillingResponseCode.SERVICE_DISCONNECTED:
                case BillingClient.BillingResponseCode.SERVICE_UNAVAILABLE:
                case BillingClient.BillingResponseCode.BILLING_UNAVAILABLE:
                case BillingClient.BillingResponseCode.ITEM_UNAVAILABLE:
                case BillingClient.BillingResponseCode.DEVELOPER_ERROR:
                case BillingClient.BillingResponseCode.ERROR:
                    Log.e(TAG, "onSkuDetailsResponse: " + responseCode + " " + debugMessage);
                    break;
                case BillingClient.BillingResponseCode.USER_CANCELED:
                    Log.i(TAG, "onSkuDetailsResponse: " + responseCode + " " + debugMessage);
                    break;
                // These response codes are not expected.
                case BillingClient.BillingResponseCode.FEATURE_NOT_SUPPORTED:
                case BillingClient.BillingResponseCode.ITEM_ALREADY_OWNED:
                case BillingClient.BillingResponseCode.ITEM_NOT_OWNED:
                default:
                    Log.wtf(TAG, "onSkuDetailsResponse: " + responseCode + " " + debugMessage);
            }
        }
    };


    @Override
    protected IWebViewClientListener getWebListener() {
        return new IWebViewClientListener() {
            @Override
            public void shouldOverrideUrlLoading(WebView view, String url) {

            }

            @Override
            public void onPageStarted(WebView view, String url, Bitmap favicon) {

            }

            @Override
            public void onReceivedError(WebView view, int errorCode, String description, String failingUrl) {
                TrackingUtil.getInstance().trackPaymentLoadScreenFail(String.valueOf(errorCode), description);
            }

            @Override
            public void onPageFinished(WebView view, String url) {
                TrackingUtil.getInstance().trackPaymentLoadScreenSuccess();
            }
        };
    }

    public interface PaymentGGSuccess {
        void onPaymentGGSuccess(PurchaseHistoryObj obj);
    }


    public void setPaymentGGSuccess(PaymentGGSuccess paymentGGSuccess) {
        this.paymentGGSuccess = paymentGGSuccess;
    }

}