package org.p2p.solanaj.core;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.List;

import org.bitcoinj.crypto.*;
import org.bitcoinj.core.Base58;
import org.p2p.solanaj.utils.TweetNaclFast;
import org.p2p.solanaj.utils.bip32.wallet.SolanaBip44;
import org.p2p.solanaj.utils.bip32.wallet.DerivableType;

public class Account {
    private final TweetNaclFast.Signature.KeyPair keyPair;

    public Account() {
        this.keyPair = TweetNaclFast.Signature.keyPair();
    }

    public Account(byte[] secretKey) {
        this.keyPair = TweetNaclFast.Signature.keyPair_fromSecretKey(secretKey);
    }

    private Account(TweetNaclFast.Signature.KeyPair keyPair) {
        this.keyPair = keyPair;
    }

    @Deprecated
    public static Account fromMnemonic(List<String> words, String passphrase) {
        byte[] seed = MnemonicCode.toSeed(words, passphrase);
        DeterministicKey masterPrivateKey = HDKeyDerivation.createMasterPrivateKey(seed);
        DeterministicHierarchy deterministicHierarchy = new DeterministicHierarchy(masterPrivateKey);
        DeterministicKey child = deterministicHierarchy.get(HDUtils.parsePath("M/501H/0H/0/0"), true, true);
        TweetNaclFast.Signature.KeyPair keyPair = TweetNaclFast.Signature.keyPair_fromSeed(child.getPrivKeyBytes());
        return new Account(keyPair);
    }

    /**
     * Derive a Solana account from a Mnemonic generated by the Solana CLI using bip44 Mnemonic with deviation path of
     * m/55H/501H/0H
     * @param words seed words
     * @param passphrase seed passphrase
     * @return Solana account
     */
    public static Account fromBip44Mnemonic(List<String> words, String passphrase) {
        SolanaBip44 solanaBip44 = new SolanaBip44();
        byte[] seed = MnemonicCode.toSeed(words, passphrase);
        byte[] privateKey = solanaBip44.getPrivateKeyFromSeed(seed, DerivableType.BIP44);
        return new Account(TweetNaclFast.Signature.keyPair_fromSeed(privateKey));
    }

    /**
     * Derive a Solana account from a Mnemonic generated by the Solana CLI using bip44 Mnemonic with deviation path of
     * m/55H/501H/0H/0H
     * @param words seed words
     * @param passphrase seed passphrase
     * @return Solana account
     */
    public static Account fromBip44MnemonicWithChange(List<String> words, String passphrase) {
        SolanaBip44 solanaBip44 = new SolanaBip44();
        byte[] seed = MnemonicCode.toSeed(words, passphrase);
        byte[] privateKey = solanaBip44.getPrivateKeyFromSeed(seed, DerivableType.BIP44CHANGE);
        return new Account(TweetNaclFast.Signature.keyPair_fromSeed(privateKey));
    }

    /**
     * Derive a Solana account from a Mnemonic generated by the Solana CLI
     * @param words seed words
     * @param passphrase seed passphrase
     * @return Solana account
     */
    public static Account fromBip39Mnemonic(List<String> words, String passphrase) {
        byte[] seed = MnemonicCode.toSeed(words, passphrase);
        TweetNaclFast.Signature.KeyPair keyPair = TweetNaclFast.Signature.keyPair_fromSeed(seed);

        return new Account(keyPair);
    }

    /**
     * Creates an {@link Account} object from a Sollet-exported JSON string (array)
     * @param json Sollet-exported JSON string (array)
     * @return {@link Account} built from Sollet-exported private key
     */
    public static Account fromJson(String json) {
        return new Account(convertJsonStringToByteArray(json));
    }

    public PublicKey getPublicKey() {
        return new PublicKey(keyPair.getPublicKey());
    }

    public byte[] getSecretKey() {
        return keyPair.getSecretKey();
    }

    /**
     * Convert's a Sollet-exported JSON string into a byte array usable for {@link Account} instantiation
     * @param characters Sollet-exported JSON string
     * @return byte array usable in {@link Account} instantiation
     */
    private static byte[] convertJsonStringToByteArray(String characters) {
        // Create resulting byte array
        ByteBuffer buffer = ByteBuffer.allocate(64);

        // Convert json array into String array
        String sanitizedJson = characters.replaceAll("\\[", "")
                .replaceAll("]", "")
                .trim();

        String[] chars = sanitizedJson.split(",");

        // Convert each String character into byte and put it in the buffer
        Arrays.stream(chars).forEach(character -> {
            byte byteValue = (byte) Integer.parseInt(character.trim());
            buffer.put(byteValue);
        });

        return buffer.array();
    }

    /**
     * Creates an Account from a base58-encoded private key string.
     * @param base58PrivateKey The base58-encoded private key
     * @return A new Account instance
     */
    public static Account fromBase58PrivateKey(String base58PrivateKey) {
        byte[] privateKey = Base58.decode(base58PrivateKey);
        return new Account(privateKey);
    }

    /**
     * Returns the account's public key as a base58-encoded string.
     * @return The base58-encoded public key
     */
    public String getPublicKeyBase58() {
        return this.getPublicKey().toBase58();
    }

    /**
     * Returns the account's private key as a base58-encoded string.
     * @return The base58-encoded private key
     */
    public String getPrivateKeyBase58() {
        return Base58.encode(this.getSecretKey());
    }
}