// Copyright (c) Microsoft Corporation.
// All rights reserved.
//
// This code is licensed under the MIT License.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files(the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and / or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions :
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
package com.microsoft.identity.common.internal.providers.oauth2;

import com.google.gson.annotations.Expose;

/**
 * This is the class encapsulating the details of the Authorization Response (OAuth2.0/OIDC).
 * https://tools.ietf.org/html/rfc6749#section-4.1.2
 * It should include all of the required and optional parameters based on the protocol and
 * support an extension to allow the authorization server / openid provider to send back additional information
 */
public class AuthorizationResponse implements ISuccessResponse {

    /**
     * REQUIRED for auth flow. The authorization code generated by the
     * authorization server. The authorization code MUST expire
     * shortly after it is issued to mitigate the risk of leaks.  A
     * maximum authorization code lifetime of 10 minutes is
     * RECOMMENDED. The client MUST NOT use the authorization code more than once.  If an authorization code is used more than
     * once, the authorization server MUST deny the request and SHOULD
     * revoke (when possible) all tokens previously issued based on
     * that authorization code. The authorization code is bound to
     * the client identifier and redirection URI.
     *
     * NULLABLE for device code flow. Device code authorization does not generate this field.
     */
    private String mCode;

    /**
     * REQUIRED if the "state" parameter was present in the client
     * authorization request.  The exact value received from the
     * client.
     */
    @Expose()
    private String mState;

    /**
     * Constructor of {@link AuthorizationResponse}.
     *
     * @param code The authorization code generated by the authorization server.
     */
    public AuthorizationResponse(final String code) {
        this(code, null);
    }

    /**
     * Constructor of {@link AuthorizationResponse}.
     *
     * @param code  The authorization code generated by the authorization server.
     * @param state "state" parameter from the client authorization request.
     */
    public AuthorizationResponse(final String code, final String state) {
        mCode = code;
        mState = state;
    }

    /**
     * Gets the responses code.
     *
     * @return The code to get.
     */
    public String getCode() {
        return mCode;
    }

    /**
     * Sets the response code.
     *
     * @param code The code to set.
     */
    public void setCode(String code) {
        mCode = code;
    }

    /**
     * Gets the response state.
     *
     * @return The state to get.
     */
    public String getState() {
        return mState;
    }

    /**
     * Sets the response state.
     *
     * @param state The state to set.
     */
    public void setState(String state) {
        mState = state;
    }

    public static class Fields {
        public static final String CODE = "code";
        public static final String STATE = "state";
    }
}
