//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
//
#include <cstring>
#include "com_microsoft_cognitiveservices_speech_intent_IntentRecognizer.h"
#include "jni_utils.h"
#include "speechapi_c_factory.h"
#include "speechapi_c_intent_recognizer.h"
#include "speechapi_c_pattern_matching_model.h"

/*
 * Class:     com_microsoft_cognitiveservices_speech_intent_IntentRecognizer
 * Method:    createIntentRecognizerFromConfig
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_intent_IntentRecognizer_createIntentRecognizerFromConfig
  (JNIEnv *env, jobject, jobject recognizerHandle, jobject speechConfigHandle, jobject audioConfigHandle)
{
    SPXRECOHANDLE recoHandle = SPXHANDLE_INVALID;
    jlong speechConfig = GetObjectHandle(env, speechConfigHandle);
    jlong audioConfig = GetObjectHandle(env, audioConfigHandle);
    SPXHR hr = recognizer_create_intent_recognizer_from_config(&recoHandle, (SPXSPEECHCONFIGHANDLE)speechConfig, (SPXAUDIOCONFIGHANDLE)audioConfig);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, recognizerHandle, (jlong)recoHandle);
    }
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_intent_IntentRecognizer
 * Method:    addIntent
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Ljava/lang/String;Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_intent_IntentRecognizer_addIntent
  (JNIEnv *env, jobject, jobject recognizerHandle, jstring intentId, jobject triggerHandle)
{
    jlong recoHandle = GetObjectHandle(env, recognizerHandle);
    jlong trigger = GetObjectHandle(env, triggerHandle);
    const char* id = GetStringUTFChars(env, intentId);
    SPXHR hr = intent_recognizer_add_intent((SPXRECOHANDLE) recoHandle, id, (SPXTRIGGERHANDLE)trigger);
    ReleaseStringUTFChars(env, intentId, id);
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_intent_IntentRecognizer
 * Method:    clearLanguageModels
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_intent_IntentRecognizer_clearLanguageModels
  (JNIEnv *env, jobject, jobject recognizerHandle)
{
    jlong recoHandle = GetObjectHandle(env, recognizerHandle);
    SPXHR hr = intent_recognizer_clear_language_models((SPXRECOHANDLE) recoHandle);
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_intent_IntentRecognizer
 * Method:    importPatternMatchingModel
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Ljava/lang/String;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_intent_IntentRecognizer_importPatternMatchingModel
  (JNIEnv *env, jobject, jobject recognizerHandle, jstring jsonInput)
{
    jlong recoHandle = GetObjectHandle(env, recognizerHandle);
    const char* jsonPtr = GetStringUTFChars(env, jsonInput);
    SPXHR hr = intent_recognizer_import_pattern_matching_model((SPXRECOHANDLE) recoHandle, jsonPtr);
    ReleaseStringUTFChars(env, jsonInput, jsonPtr);
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_intent_IntentRecognizer
 * Method:    recognizeTextOnce
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Ljava/lang/String;Lcom/microsoft/cognitiveservices/speech/util/IntRef;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_intent_IntentRecognizer_recognizeTextOnce
  (JNIEnv *env, jobject, jobject recognizerHandle, jstring inputText, jobject resultRef)
{
    jlong recoHandle = GetObjectHandle(env, recognizerHandle);
    const char* textPtr = GetStringUTFChars(env, inputText);
    SPXRESULTHANDLE result = SPXHANDLE_INVALID;
    SPXHR hr = intent_recognizer_recognize_text_once((SPXRECOHANDLE) recoHandle, textPtr, &result);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, resultRef, (jlong)result);
    }
    ReleaseStringUTFChars(env, inputText, textPtr);
    return (jlong)hr;
}

JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_intent_IntentRecognizer_createPatternMatchingModel
  (JNIEnv *env, jobject, jobject langModelHandle, jobject recognizerHandle, jstring modelId)
{
    jlong hReco = GetObjectHandle(env, recognizerHandle);
    SPXLUMODELHANDLE hLangModel = SPXHANDLE_INVALID;
    NativeUtf8String modelIdUtf8(env, modelId);

    SPXHR hr = pattern_matching_model_create(&hLangModel, (SPXRECOHANDLE)hReco, modelIdUtf8);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, langModelHandle, (jlong)hLangModel);
    }

    return (jlong)hr;
}

struct ByIndexContext
{
    JNIEnv* const env;
    jobjectArray const array;
    NativeUtf8String lastUtf8;

    ByIndexContext(JNIEnv *env, jobjectArray array) : env(env), array(array) {}

    size_t arrayLength() const
    {
        size_t length = 0;
        if (array != nullptr)
        {
            length = (size_t)env->GetArrayLength(array);
            CheckException(env);
        }

        return length;
    }
};

static SPXHR SPXAPI_CALLTYPE get_string_by_index(void* context, size_t index, const char** str, size_t* len)
{
    SPX_RETURN_HR_IF(SPXERR_INVALID_ARG, context == nullptr || str == nullptr || len == nullptr);
    try
    {
        auto ctxt = (ByIndexContext*)context;
        if (ctxt->env == nullptr || ctxt->array == nullptr)
        {
            return SPXERR_INVALID_ARG;
        }

        auto element = (jstring)ctxt->env->GetObjectArrayElement(ctxt->array, (jsize)index);
        if (CheckException(ctxt->env) || element == nullptr)
        {
            return SPXERR_UNHANDLED_EXCEPTION;
        }

        ctxt->lastUtf8 = NativeUtf8String(ctxt->env, element);
        *str = ctxt->lastUtf8;
        *len = strlen(ctxt->lastUtf8);

        return SPX_NOERROR;
    }
    catch (...)
    {
        return SPXERR_UNHANDLED_EXCEPTION;
    }
}

JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_intent_IntentRecognizer_patternMatchingModelAddEntity
  (JNIEnv *env, jobject, jobject modelHandle, jstring entityId, jint entityType, jint matchMode, jobjectArray phrases)
{
    jlong hModel = GetObjectHandle(env, modelHandle);
    NativeUtf8String entityIdUtf8(env, entityId);
    ByIndexContext context(env, phrases);

    SPXHR hr = pattern_matching_model_add_entity(
        (SPXLUMODELHANDLE)hModel,
        entityIdUtf8,
        (int32_t)entityType,
        (int32_t)matchMode,
        context.arrayLength(),
        &context,
        get_string_by_index);

    return (jlong)hr;
}

JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_intent_IntentRecognizer_patternMatchingModelAddIntent
  (JNIEnv *env, jobject, jobject modelHandle, jstring intentId, jint priority, jobjectArray phrases)
{
    jlong hModel = GetObjectHandle(env, modelHandle);
    NativeUtf8String intentIdUtd8(env, intentId);
    ByIndexContext context(env, phrases);

    SPXHR hr = pattern_matching_model_add_intent(
        (SPXLUMODELHANDLE)hModel,
        intentIdUtd8,
        (uint32_t)priority,
        context.arrayLength(),
        &context,
        get_string_by_index);

    return (jlong)hr;
}

JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_intent_IntentRecognizer_addIntentWithModelId
  (JNIEnv *env, jobject, jobject recognizerHandle, jobject intentTriggerHandle, jstring modelId)
{
    jlong hReco = GetObjectHandle(env, recognizerHandle);
    jlong hTrigger = GetObjectHandle(env, intentTriggerHandle);
    NativeUtf8String modelIdUtf8(env, modelId);

    SPXHR hr = intent_recognizer_add_intent_with_model_id((SPXRECOHANDLE)hReco, (SPXTRIGGERHANDLE)hTrigger, modelIdUtf8);
    return (jlong)hr;
}
