//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
//
#include "com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions.h"
#include "jni_utils.h"
#include "speechapi_c_audio_processing_options.h"
#include <vector>

SPXHR GetMicrophoneArrayType(JNIEnv* env, jobject microphoneArrayGeometry, jclass microphoneArrayGeometryClass, AudioProcessingOptions_MicrophoneArrayType* microphoneArrayType)
{
    jmethodID getMicrophoneArrayTypeMethodID = env->GetMethodID(microphoneArrayGeometryClass, "getMicrophoneArrayType", "()Lcom/microsoft/cognitiveservices/speech/audio/MicrophoneArrayType;");
    if (CheckException(env))
    {
        return SPXERR_RUNTIME_ERROR;
    }
    jobject microphoneArrayTypeObj = env->CallObjectMethod(microphoneArrayGeometry, getMicrophoneArrayTypeMethodID);
    if (CheckException(env))
    {
        return SPXERR_RUNTIME_ERROR;
    }
    jclass microphoneArrayTypeClass = env->GetObjectClass(microphoneArrayTypeObj);
    if (CheckException(env))
    {
        return SPXERR_RUNTIME_ERROR;
    }
    jmethodID ordinalMethodID = env->GetMethodID(microphoneArrayTypeClass, "ordinal", "()I");
    if (CheckException(env))
    {
        return SPXERR_RUNTIME_ERROR;
    }
    jint ordinal = env->CallIntMethod(microphoneArrayTypeObj, ordinalMethodID);
    if (CheckException(env))
    {
        return SPXERR_RUNTIME_ERROR;
    }
    *microphoneArrayType = (AudioProcessingOptions_MicrophoneArrayType)ordinal;
    return SPX_NOERROR;
}

SPXHR GetBeamformingStartAngle(JNIEnv* env, jobject microphoneArrayGeometry, jclass microphoneArrayGeometryClass, uint16_t* beamformingStartAngle)
{
    jmethodID getBeamformingStartAngleMethodID = env->GetMethodID(microphoneArrayGeometryClass, "getBeamformingStartAngle", "()I");
    if (CheckException(env))
    {
        return SPXERR_RUNTIME_ERROR;
    }
    jint angle = env->CallIntMethod(microphoneArrayGeometry, getBeamformingStartAngleMethodID);
    if (CheckException(env))
    {
        return SPXERR_RUNTIME_ERROR;
    }
    *beamformingStartAngle = (uint16_t)angle;
    return SPX_NOERROR;
}

SPXHR GetBeamformingEndAngle(JNIEnv* env, jobject microphoneArrayGeometry, jclass microphoneArrayGeometryClass, uint16_t* beamformingEndAngle)
{
    jmethodID getBeamformingEndAngleMethodID = env->GetMethodID(microphoneArrayGeometryClass, "getBeamformingEndAngle", "()I");
    if (CheckException(env))
    {
        return SPXERR_RUNTIME_ERROR;
    }
    jint angle = env->CallIntMethod(microphoneArrayGeometry, getBeamformingEndAngleMethodID);
    if (CheckException(env))
    {
        return SPXERR_RUNTIME_ERROR;
    }
    *beamformingEndAngle = (uint16_t)angle;
    return SPX_NOERROR;
}

SPXHR GetMicrophoneCoordinates(JNIEnv* env, jobject microphoneArrayGeometry, jclass microphoneArrayGeometryClass, std::vector<AudioProcessingOptions_MicrophoneCoordinates>& microphoneCoordinates)
{
    jmethodID getMicrophoneCoordinatesMethodID = env->GetMethodID(microphoneArrayGeometryClass, "getMicrophoneCoordinates", "()[Lcom/microsoft/cognitiveservices/speech/audio/MicrophoneCoordinates;");
    if (CheckException(env))
    {
        return SPXERR_RUNTIME_ERROR;
    }
    jobjectArray coordinates = (jobjectArray)env->CallObjectMethod(microphoneArrayGeometry, getMicrophoneCoordinatesMethodID);
    if (CheckException(env))
    {
        return SPXERR_RUNTIME_ERROR;
    }
    size_t arraySize = (size_t)env->GetArrayLength(coordinates);
    if (CheckException(env))
    {
        return SPXERR_RUNTIME_ERROR;
    }
    microphoneCoordinates.resize(arraySize);
    for (size_t index = 0; index < arraySize; index++)
    {
        jobject coords = env->GetObjectArrayElement(coordinates, (jsize)index);
        if (CheckException(env))
        {
            return SPXERR_RUNTIME_ERROR;
        }
        jclass microphoneCoordinatesClass = env->GetObjectClass(coords);
        if (CheckException(env))
        {
            return SPXERR_RUNTIME_ERROR;
        }
        jmethodID getXMethodID = env->GetMethodID(microphoneCoordinatesClass, "getX", "()I");
        if (CheckException(env))
        {
            return SPXERR_RUNTIME_ERROR;
        }
        int xCoord = (int)env->CallIntMethod(coords, getXMethodID);
        if (CheckException(env))
        {
            return SPXERR_RUNTIME_ERROR;
        }
        jmethodID getYMethodID = env->GetMethodID(microphoneCoordinatesClass, "getY", "()I");
        if (CheckException(env))
        {
            return SPXERR_RUNTIME_ERROR;
        }
        int yCoord = (int)env->CallIntMethod(coords, getYMethodID);
        if (CheckException(env))
        {
            return SPXERR_RUNTIME_ERROR;
        }
        jmethodID getZMethodID = env->GetMethodID(microphoneCoordinatesClass, "getZ", "()I");
        if (CheckException(env))
        {
            return SPXERR_RUNTIME_ERROR;
        }
        int zCoord = (int)env->CallIntMethod(coords, getZMethodID);
        if (CheckException(env))
        {
            return SPXERR_RUNTIME_ERROR;
        }
        microphoneCoordinates[index] = { xCoord, yCoord, zCoord };
    }
    return SPX_NOERROR;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions
 * Method:    create
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/IntRef;I)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions_create
  (JNIEnv *env, jclass, jobject objHandle, jint audioProcessingFlags)
{
    SPXAUDIOPROCESSINGOPTIONSHANDLE audioProcessingOptionsHandle = SPXHANDLE_INVALID;
    SPXHR hr = audio_processing_options_create(&audioProcessingOptionsHandle, audioProcessingFlags);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, objHandle, (jlong)audioProcessingOptionsHandle);
    }
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions
 * Method:    createFromPresetMicrophoneArrayGeometry
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/IntRef;III)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions_createFromPresetMicrophoneArrayGeometry
  (JNIEnv *env, jclass, jobject objHandle, jint audioProcessingFlags, jint microphoneArrayGeometry, jint speakerReferenceChannel)
{
    SPXAUDIOPROCESSINGOPTIONSHANDLE audioProcessingOptionsHandle = SPXHANDLE_INVALID;
    SPXHR hr = audio_processing_options_create_from_preset_microphone_array_geometry(
        &audioProcessingOptionsHandle,
        audioProcessingFlags,
        (AudioProcessingOptions_PresetMicrophoneArrayGeometry)microphoneArrayGeometry,
        (AudioProcessingOptions_SpeakerReferenceChannel)speakerReferenceChannel);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, objHandle, (jlong)audioProcessingOptionsHandle);
    }
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions
 * Method:    createFromMicrophoneArrayGeometry
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/IntRef;ILcom/microsoft/cognitiveservices/speech/audio/MicrophoneArrayGeometry;I)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions_createFromMicrophoneArrayGeometry
  (JNIEnv *env, jclass, jobject objHandle, jint audioProcessingFlags, jobject microphoneArrayGeometry, jint speakerReferenceChannel)
{
    SPXHR hr = SPX_NOERROR;
    SPXAUDIOPROCESSINGOPTIONSHANDLE audioProcessingOptionsHandle = SPXHANDLE_INVALID;
    jclass microphoneArrayGeometryClass = env->GetObjectClass(microphoneArrayGeometry);
    if (CheckException(env))
    {
        return SPXERR_RUNTIME_ERROR;
    }
    AudioProcessingOptions_MicrophoneArrayType microphoneArrayType;
    hr = GetMicrophoneArrayType(env, microphoneArrayGeometry, microphoneArrayGeometryClass, &microphoneArrayType);
    if (SPX_FAILED(hr))
    {
        return (jlong)hr;
    }
    uint16_t beamformingStartAngle;
    hr = GetBeamformingStartAngle(env, microphoneArrayGeometry, microphoneArrayGeometryClass, &beamformingStartAngle);
    if (SPX_FAILED(hr))
    {
        return (jlong)hr;
    }
    uint16_t beamformingEndAngle;
    hr = GetBeamformingEndAngle(env, microphoneArrayGeometry, microphoneArrayGeometryClass, &beamformingEndAngle);
    if (SPX_FAILED(hr))
    {
        return (jlong)hr;
    }
    std::vector<AudioProcessingOptions_MicrophoneCoordinates> microphoneCoordinates;
    hr = GetMicrophoneCoordinates(env, microphoneArrayGeometry, microphoneArrayGeometryClass, microphoneCoordinates);
    if (SPX_FAILED(hr))
    {
        return (jlong)hr;
    }
    AudioProcessingOptions_MicrophoneArrayGeometry geometry
    {
        microphoneArrayType,
        beamformingStartAngle,
        beamformingEndAngle,
        (uint16_t)microphoneCoordinates.size(),
        microphoneCoordinates.data()
    };
    hr = audio_processing_options_create_from_microphone_array_geometry(&audioProcessingOptionsHandle, audioProcessingFlags, &geometry, (AudioProcessingOptions_SpeakerReferenceChannel)speakerReferenceChannel);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, objHandle, (jlong)audioProcessingOptionsHandle);
    }
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions
 * Method:    getAudioProcessingFlags
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/IntRef;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions_getAudioProcessingFlags
  (JNIEnv *env, jobject, jobject audioProcessingOptionsHandle, jobject audioProcessingFlags)
{
    jlong audioProcessingOptions = GetObjectHandle(env, audioProcessingOptionsHandle);
    int flags = 0;
    SPXHR hr = audio_processing_options_get_audio_processing_flags((SPXAUDIOPROCESSINGOPTIONSHANDLE)audioProcessingOptions, &flags);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, audioProcessingFlags, (jlong)flags);
    }
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions
 * Method:    getPresetMicrophoneArrayGeometry
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/IntRef;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions_getPresetMicrophoneArrayGeometry
  (JNIEnv *env, jobject, jobject audioProcessingOptionsHandle, jobject presetMicrophoneArrayGeometryRef)
{
    jlong audioProcessingOptions = GetObjectHandle(env, audioProcessingOptionsHandle);
    AudioProcessingOptions_PresetMicrophoneArrayGeometry microphoneArrayGeometry;
    SPXHR hr = audio_processing_options_get_preset_microphone_array_geometry((SPXAUDIOPROCESSINGOPTIONSHANDLE)audioProcessingOptions, &microphoneArrayGeometry);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, presetMicrophoneArrayGeometryRef, (jlong)microphoneArrayGeometry);
    }
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions
 * Method:    getMicrophoneArrayType
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/IntRef;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions_getMicrophoneArrayType
  (JNIEnv *env, jobject, jobject audioProcessingOptionsHandle, jobject microphoneArrayTypeRef)
{
    jlong audioProcessingOptions = GetObjectHandle(env, audioProcessingOptionsHandle);
    AudioProcessingOptions_MicrophoneArrayType microphoneArrayType;
    SPXHR hr = audio_processing_options_get_microphone_array_type((SPXAUDIOPROCESSINGOPTIONSHANDLE)audioProcessingOptions, &microphoneArrayType);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, microphoneArrayTypeRef, (jlong)microphoneArrayType);
    }
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions
 * Method:    getBeamformingStartAngle
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/IntRef;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions_getBeamformingStartAngle
  (JNIEnv *env, jobject, jobject audioProcessingOptionsHandle, jobject beamformingStartAngle)
{
    jlong audioProcessingOptions = GetObjectHandle(env, audioProcessingOptionsHandle);
    uint16_t startAngle;
    SPXHR hr = audio_processing_options_get_beamforming_start_angle((SPXAUDIOPROCESSINGOPTIONSHANDLE)audioProcessingOptions, &startAngle);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, beamformingStartAngle, (jlong)startAngle);
    }
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions
 * Method:    getBeamformingEndAngle
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/IntRef;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions_getBeamformingEndAngle
  (JNIEnv *env, jobject, jobject audioProcessingOptionsHandle, jobject beamformingEndAngle)
{
    jlong audioProcessingOptions = GetObjectHandle(env, audioProcessingOptionsHandle);
    uint16_t endAngle;
    SPXHR hr = audio_processing_options_get_beamforming_end_angle((SPXAUDIOPROCESSINGOPTIONSHANDLE)audioProcessingOptions, &endAngle);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, beamformingEndAngle, (jlong)endAngle);
    }
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions
 * Method:    getMicrophoneCoordinates
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/IntRef;)[Lcom/microsoft/cognitiveservices/speech/audio/MicrophoneCoordinates;
 */
JNIEXPORT jobjectArray JNICALL Java_com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions_getMicrophoneCoordinates
  (JNIEnv *env, jobject, jobject audioProcessingOptionsHandle, jobject resultHr)
{
    jobjectArray microphoneCoordinates = NULL;
    uint16_t microphoneCount = 0;
    SPXHR hr = SPX_NOERROR;

    jlong audioProcessingOptions = GetObjectHandle(env, audioProcessingOptionsHandle);
    hr = audio_processing_options_get_microphone_count((SPXAUDIOPROCESSINGOPTIONSHANDLE)audioProcessingOptions, &microphoneCount);
    if (SPX_SUCCEEDED(hr) && (microphoneCount > 0))
    {
        std::vector<AudioProcessingOptions_MicrophoneCoordinates> coordinates(microphoneCount);
        hr = audio_processing_options_get_microphone_coordinates((SPXAUDIOPROCESSINGOPTIONSHANDLE)audioProcessingOptions, coordinates.data(), microphoneCount);
        if (SPX_SUCCEEDED(hr))
        {
            jclass microphoneCoordinatesClass = env->FindClass("com/microsoft/cognitiveservices/speech/audio/MicrophoneCoordinates");
            if (CheckException(env))
            {
                hr = SPXERR_RUNTIME_ERROR;
                goto Exit;
            }
            jmethodID microphoneCoordinatesMethodID = env->GetMethodID(microphoneCoordinatesClass, "<init>", "(III)V");
            if (CheckException(env))
            {
                hr = SPXERR_RUNTIME_ERROR;
                goto Exit;
            }
            microphoneCoordinates = env->NewObjectArray(microphoneCount, microphoneCoordinatesClass, NULL);
            if (CheckException(env) || (microphoneCoordinates == NULL))
            {
                hr = SPXERR_OUT_OF_MEMORY;
                goto Exit;
            }
            for (size_t index = 0; index < (size_t)microphoneCount; index++)
            {
                jobject coords = env->NewObject(microphoneCoordinatesClass, microphoneCoordinatesMethodID, coordinates[index].X, coordinates[index].Y, coordinates[index].Z);
                if (CheckException(env) || (coords == NULL))
                {
                    hr = SPXERR_OUT_OF_MEMORY;
                    goto Exit;
                }
                env->SetObjectArrayElement(microphoneCoordinates, (jsize)index, coords);
                if (CheckException(env))
                {
                    hr = SPXERR_RUNTIME_ERROR;
                    goto Exit;
                }
            }
        }
    }

Exit:
    SetObjectHandle(env, resultHr, (jlong)hr);
    return microphoneCoordinates;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions
 * Method:    getSpeakerReferenceChannel
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/IntRef;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_audio_AudioProcessingOptions_getSpeakerReferenceChannel
  (JNIEnv *env, jobject, jobject audioProcessingOptionsHandle, jobject speakerReferenceChannelRef)
{
    jlong audioProcessingOptions = GetObjectHandle(env, audioProcessingOptionsHandle);
    AudioProcessingOptions_SpeakerReferenceChannel speakerReferenceChannel;
    SPXHR hr = audio_processing_options_get_speaker_reference_channel((SPXAUDIOPROCESSINGOPTIONSHANDLE)audioProcessingOptions, &speakerReferenceChannel);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, speakerReferenceChannelRef, (jlong)speakerReferenceChannel);
    }
    return (jlong)hr;
}
