//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.util;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import com.microsoft.cognitiveservices.speech.util.KeyedItem;

/**
 * Hash table based implementation of the {@code Map} interface.  This
 * implementation provides all of the optional map operations, and permits
 * {@code null} values and the {@code null} key.  (The {@code HashMap}
 * class is roughly equivalent to {@code Hashtable}, except that it is
 * unsynchronized and permits nulls.)  This class makes no guarantees as to
 * the order of the map; in particular, it does not guarantee that the order
 * will remain constant over time.
 *
 * Unlike a typical hash map, this implementation uses KeyedItem's as the 
 * value. A KeyedItem must implement the {@code KeyedItem} interface and 
 * provide a unique ID to avoid collisions within the map.
 * 
 * For more details see the {@code HashMap} class.
 */
public final class KeyedItemHashMap<T extends KeyedItem> implements Map<String,T> {
    private HashMap<String,T> map;

    /**
     * Constructs an empty HashMap with the default initial capacity (16) and 
     * the default load factor (0.75).
     */
    public KeyedItemHashMap() {
        map = new HashMap<>();
    }

    /**
     * Returns the number of key-value mappings in this map.
     * @return The number of key-value mappings in this map.
     */
    public int size() {
        return map.size();
    }

    /**
     * Returns {@code true} if this map contains no key-value mappings.
     * @return {@code true} if this map contains no key-value mappings.
     */
    public boolean isEmpty() {
        return map.isEmpty();
    }

    /**
     * Returns {@code true} if this map contains a mapping for the specified key.
     * @param key The key whose presence in this map is to be tested.
     * @return {@code true} if this map contains a mapping for the specified key.
     */
    public boolean containsKey(Object key) {
        return map.containsKey(key);
    }

    /**
     * Returns {@code true} if this map maps a key to the
     * specified value.
     *
     * @param keyedItem value whose presence in this map is to be tested.
     * @return {@code true} if this map maps the value to an id of an item.
     */
    public boolean containsValue(Object keyedItem) {
        return map.containsValue(keyedItem);
    }

    /**
     * Returns the string representation of the {@code Object} argument.
     *
     * @param   obj   an {@code Object}.
     * @return  if the argument is {@code null}, then a string equal to
     *          {@code "null"}; otherwise, the value of
     *          {@code obj.toString()} is returned.
     * @see     java.lang.Object#toString()
     */
    public T get(Object obj) {
        return map.get(String.valueOf(obj));
    }

    /**
     * Associates the specified value with the keyedItem's ID in this map.
     * If the map previously contained a mapping for the keyedItem's ID, the old
     * value is replaced.
     *
     * @param key UNUSED here but necessary for override.
     * @param keyedItem Value that implements the KeyedItem interface. The getId() function will
     * be used to determine the key for the map.
     * @return the previous value associated with {@code keyedItem.getId()}, or
     *         {@code null} if there was no mapping for {@code keyedItem.getId()}.
     *         (A {@code null} return can also indicate that the map
     *         previously associated {@code null} with {@code keyedItem.getId()}.)
     */
    public T put(String key, T keyedItem) {
        return map.put(keyedItem.getId(), keyedItem);
    }

    /**
     * Associates the specified value with the keyedItem's ID in this map.
     * If the map previously contained a mapping for the keyedItem's ID, the old
     * value is replaced.
     *
     * @param keyedItem Value that implements the KeyedItem interface. The getId() function will
     * be used to determine the key for the map.
     * @return the previous value associated with {@code keyedItem.getId()}, or
     *         {@code null} if there was no mapping for {@code keyedItem.getId()}.
     *         (A {@code null} return can also indicate that the map
     *         previously associated {@code null} with {@code keyedItem.getId()}.)
     */
    public T put(T keyedItem) {
        return map.put(keyedItem.getId(), keyedItem);
    }

    /**
     * Removes the mapping for the specified keyedItem's ID from this map if present.
     *
     * @param  keyedItem KeyedItem whose mapping is to be removed from the map.
     * @return the previous value associated with {@code keyedItem.getId()}, or
     *         {@code null} if there was no mapping for {@code keyedItem.getId()}.
     *         (A {@code null} return can also indicate that the map
     *         previously associated {@code null} with {@code keyedItem.getId()}.)
     * @throws ClassCastException if the keyedItem does not implement the KeyedItem interface.
     */
    public T remove(Object keyedItem) {
        return map.remove(((T)keyedItem).getId());
    }

    /**
     * Copies all of the mappings from the specified map to this map.
     * These mappings will replace any mappings that this map had for
     * any of the keys currently in the specified map.
     *
     * @param map mappings to be stored in this map
     * @throws NullPointerException if the specified map is null
     */
    public void putAll(Map<? extends String, ? extends T> map) {
        this.map.putAll(map);
    }

    /**
     * Removes all of the mappings from this map.
     * The map will be empty after this call returns.
     */
    public void clear() {
        map.clear();
    }

    /**
     * Returns a {@code Set} view of the keys contained in this map.
     * The set is backed by the map, so changes to the map are
     * reflected in the set, and vice-versa.  If the map is modified
     * while an iteration over the set is in progress (except through
     * the iterator's own {@code remove} operation), the results of
     * the iteration are undefined.  The set supports element removal,
     * which removes the corresponding mapping from the map, via the
     * {@code Iterator.remove}, {@code Set.remove},
     * {@code removeAll}, {@code retainAll}, and {@code clear}
     * operations.  It does not support the {@code add} or {@code addAll}
     * operations.
     *
     * @return a set view of the keys contained in this map
     */
    public Set<String> keySet() {
        return map.keySet();
    }

    /**
     * Returns a {@code Collection} view of the values contained in this map.
     * The collection is backed by the map, so changes to the map are
     * reflected in the collection, and vice-versa.  If the map is
     * modified while an iteration over the collection is in progress
     * (except through the iterator's own {@code remove} operation),
     * the results of the iteration are undefined.  The collection
     * supports element removal, which removes the corresponding
     * mapping from the map, via the {@code Iterator.remove},
     * {@code Collection.remove}, {@code removeAll},
     * {@code retainAll} and {@code clear} operations.  It does not
     * support the {@code add} or {@code addAll} operations.
     *
     * @return a view of the values contained in this map
     */
    public Collection<T> values() {
        return map.values();
    }

    /**
     * Returns a {@code Set} view of the mappings contained in this map.
     * The set is backed by the map, so changes to the map are
     * reflected in the set, and vice-versa.  If the map is modified
     * while an iteration over the set is in progress (except through
     * the iterator's own {@code remove} operation, or through the
     * {@code setValue} operation on a map entry returned by the
     * iterator) the results of the iteration are undefined.  The set
     * supports element removal, which removes the corresponding
     * mapping from the map, via the {@code Iterator.remove},
     * {@code Set.remove}, {@code removeAll}, {@code retainAll} and
     * {@code clear} operations.  It does not support the
     * {@code add} or {@code addAll} operations.
     *
     * @return a set view of the mappings contained in this map
     */
    public Set<Entry<String, T>> entrySet() {
        return map.entrySet();
    }
}