//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//

package com.microsoft.cognitiveservices.speech.util;

import com.microsoft.cognitiveservices.speech.util.JsonValue;

/**
 * The type Json value jni.
 */
public final class JsonValueJNI {

    /**
     * Create parser json value.
     *
     * @param json the json
     * @return the JsonValue object
     */
    public static JsonValue createParser(String json) {
        IntRef handle = new IntRef(0);
        int item = createJsonParserHandle(json, handle);
        SafeHandle parserHandle = new SafeHandle(handle.getValue(), SafeHandleType.JSONVALUE);
        return new JsonValue(item, parserHandle);
    }

    /**
     * As string.
     *
     * @param parserHandle the parser handle
     * @param item         the item
     * @return the string value
     */
    public static String asString(SafeHandle parserHandle, int item) {

        if (kind(parserHandle, item) != ValueKind.STRING.getValue()) {
            Contracts.throwIfFail(INVALID_ARG);
        }
        return asString(parserHandle, item, null);
    }

    /**
     * As int.
     *
     * @param parserHandle the parser handle
     * @param item         the item
     * @return the long value
     */
    public static long asInt(SafeHandle parserHandle, int item) {

        if (kind(parserHandle, item) != ValueKind.NUMBER.getValue()) {
            Contracts.throwIfFail(INVALID_ARG);
        }
        return asInt(parserHandle, item, 0);
    }

    /**
     * As float.
     *
     * @param parserHandle the parser handle
     * @param item         the item
     * @return the float value
     */
    public static float asFloat(SafeHandle parserHandle, int item) {

        if (kind(parserHandle, item) != ValueKind.NUMBER.getValue()) {
            Contracts.throwIfFail(INVALID_ARG);
        }
        return (float) asDouble(parserHandle, item, 0.0);
    }

    /**
     * As boolean.
     *
     * @param parserHandle the parser handle
     * @param item         the item
     * @return the boolean value
     */
    public static boolean asBoolean(SafeHandle parserHandle, int item) {

        if (kind(parserHandle, item) != ValueKind.BOOLEAN.getValue()) {
            Contracts.throwIfFail(INVALID_ARG);
        }
        return asBoolean(parserHandle, item, false);
    }

    /**
     * Gets value.
     *
     * @param parserHandle the parser handle
     * @param item         the item
     * @param index        the index
     * @param key          the key
     * @return the value
     */
    public static JsonValue getValue(SafeHandle parserHandle, int item, int index, String key) {
        int newItem = get(parserHandle, item, index, key);
        return new JsonValue(newItem, parserHandle);
    }

    /**
     * Gets the name/key of the pair.
     *
     * @param parserHandle the parser handle
     * @param item         the item
     * @param index        the index
     * @param key          the key
     * @return String the name/key of the pair.
     */
    public static String getName(SafeHandle parserHandle, int item, int index, String key) {
        String result = getName(parserHandle, item, index);
        return result;
    }

    /**
     * Has value.
     *
     * @param parserHandle the parser handle
     * @param item         the item
     * @param index        the index
     * @param key          the key
     * @return the boolean value
     */
    public static boolean hasValue(SafeHandle parserHandle, int item, int index, String key) {
        return get(parserHandle, item, index, key) > 0;
    }

    /**
     * The enum Value kind.
     */
    enum ValueKind {
        /**
         * Error value kind.
         */
        ERROR(-1),
        /**
         * End value kind.
         */
        END(0),
        /**
         * Object value kind.
         */
        OBJECT('{'),
        /**
         * Array value kind.
         */
        ARRAY('['),
        /**
         * String value kind.
         */
        STRING('\"'),
        /**
         * Number value kind.
         */
        NUMBER('1'),
        /**
         * Boolean value kind.
         */
        BOOLEAN('b'),
        /**
         * Null value kind.
         */
        NULL('n'),
        /**
         * Unspecified value kind.
         */
        UNSPECIFIED('?');

        private ValueKind(int id) {
            this.id = id;
        }

        /**
         * Gets value.
         *
         * @return the value
         */
        public int getValue() {
            return this.id;
        }

        private int id;
    }

    /**
     * Count int.
     *
     * @param parser the parser
     * @param item   the item
     * @return the count
     */
    public final native static int count(SafeHandle parser, int item);

    /**
     * As json string.
     *
     * @param parser the parser
     * @param item   the item
     * @return the string
     */
    public final native static String asJson(SafeHandle parser, int item);

    private final native static int createJsonParserHandle(String json, IntRef parser);
    private final native static long asInt(SafeHandle parser, int item, long defaultValue);
    private final native static String asString(SafeHandle parser, int item, String defaultValue);
    private final native static double asDouble(SafeHandle parser, int item, double defaultValue);
    private final native static boolean asBoolean(SafeHandle parser, int item, boolean defaultValue);
    private final native static int kind(SafeHandle parser, int item);
    private final native static int get(SafeHandle parser, int item, int index, String key);
    private final native static String getName(SafeHandle parser, int item, int index);
    private static long INVALID_ARG = 0x005;
}
