//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.transcription;

import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.RecognitionEventArgs;
import com.microsoft.cognitiveservices.speech.util.IntRef;

/**
 * Class that defines meeting transcription event arguments.
 */
public class MeetingTranscriptionEventArgs extends RecognitionEventArgs {

    /*! \cond INTERNAL */

    /**
     * Constructs an instance of a MeetingTranscriptionEventArgs object.
     * @param eventArgs recognition event args object.
     */
    MeetingTranscriptionEventArgs(long eventArgs) {
        super(eventArgs);
        storeEventData(false);

    }
    
    MeetingTranscriptionEventArgs(long eventArgs, boolean dispose) {
        super(eventArgs);
        storeEventData(dispose);
    }
    
    /*! \endcond */

    /**
     * Represents the meeting transcription result.
     * @return The meeting transcription result.
     */
    public final MeetingTranscriptionResult getResult() {
        return result;
    }

    /**
     * Returns a String that represents the meeting transcription result event.
     * @return A String that represents the meeting transcription result event.
     */
    @Override
    public String toString() {
        return "SessionId:" + this.getSessionId() +
                " ResultId:" + result.getResultId() +
                " Reason:" + result.getReason() +
                " UserId:" + result.getUserId() +
                " UtteranceId:" + result.getUtteranceId() +
                " Recognized text:<" + result.getText() + ">.";
    }

    private void storeEventData(boolean disposeNativeResources) {
        Contracts.throwIfNull(eventHandle, "eventHandle");
        IntRef resultHandle = new IntRef(0);
        Contracts.throwIfFail(getRecognitionResult(eventHandle, resultHandle));
        this.result = new MeetingTranscriptionResult(resultHandle.getValue());
        Contracts.throwIfNull(this.getSessionId(), "SessionId");
        if (disposeNativeResources == true) {
            super.close();
        }
    }
    
    private MeetingTranscriptionResult result;
}
