//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.speaker;

import java.lang.AutoCloseable;
import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.PropertyCollection;
import com.microsoft.cognitiveservices.speech.PropertyId;
import com.microsoft.cognitiveservices.speech.RecognitionResult;
import com.microsoft.cognitiveservices.speech.ResultReason;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.util.StringRef;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;

/**
 * Defines result of speaker recognition.
 */
public final class SpeakerRecognitionResult implements AutoCloseable
{

    SpeakerRecognitionResult(long result) {
        Contracts.throwIfNull(result, "result");
        this.resultHandle = new SafeHandle(result, SafeHandleType.SpeakerRecognitionResult);

        StringRef stringVal = new StringRef("");
        Contracts.throwIfFail(getResultId(resultHandle, stringVal));
        this.resultId = stringVal.getValue();
                
        IntRef intVal = new IntRef(0);
        Contracts.throwIfFail(getResultReason(resultHandle, intVal));
        this.reason = ResultReason.values()[(int)intVal.getValue()];
        
        IntRef propertyHandle = new IntRef(0);
        Contracts.throwIfFail(getPropertyBagFromResult(resultHandle, propertyHandle));
        this.properties = new PropertyCollection(propertyHandle);

        this.profileId = properties.getProperty("speakerrecognition.profileid");
        String value = properties.getProperty("speakerrecognition.score");
        this.score = value.isEmpty() ? 0 : Double.parseDouble(value);
    }

    /**
     * Presents the result identifier.
     * @return The result identifier.
     */
    public String getResultId() {  
        return this.resultId;
    }

    /**
     * Specifies status of speaker recognition result.
     * @return The reason of the result.
     */
    public ResultReason getReason() {
        return this.reason;
    }

    /**
     * Presents the recognized profile id.
     * @return The profile identifier.
     */
    public String getProfileId() {  
        return this.profileId;
    }

    /**
     * Presents the similarity score of the recognized speaker.
     * The score is a float number indicating the similarity between input audio and targeted voice profile.This number is between 0 and 1. A higher number means higher similarity.
     * @return The similarity score of the recognized speaker.
     */
    public Double getScore() {  
        return this.score;
    }

    /**  
     * The set of properties exposed in the result.
     * @return The set of properties exposed in the result.
     */
    public PropertyCollection getProperties() {
        return this.properties;
    }

    /**
     * Explicitly frees any external resource attached to the object
     */
    public void close() {
        if (this.resultHandle != null) {
            resultHandle.close();
            this.resultHandle = null;
        }

        if (this.properties != null) {
            this.properties.close();
            this.properties = null;
        }
    }

    /**
     * Returns a String that represents the speaker recognition result.
     * @return A String that represents the speaker recognition result.
     */
    @Override
    public String toString() {
        return "ResultId:" + getResultId() +
                " Reason:" + getReason() +
                " Recognized profileId:" + getProfileId() +
                " Json:" + properties.getProperty(PropertyId.SpeechServiceResponse_JsonResult);
    }

    /*! \cond INTERNAL */
    /**
     * Returns the speaker recognition result implementation.
     * @return The implementation of the result.
     */
    public SafeHandle getImpl() {
        Contracts.throwIfNull(this.resultHandle, "result");
        return this.resultHandle;
    }
    /*! \endcond */

    private SafeHandle resultHandle = null;
    private PropertyCollection properties = null;
    private String resultId = "";
    private ResultReason reason;
    private String profileId = "";
    private double score = 0.0;

    private final native long getResultId(SafeHandle resultHandle, StringRef resultId);
    private final native long getResultReason(SafeHandle resultHandle, IntRef reasonVal);
    private final native long getPropertyBagFromResult(SafeHandle resultHandle, IntRef propertyHandle);
}
