//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.speaker;


import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.CancellationErrorCode;
import com.microsoft.cognitiveservices.speech.CancellationReason;
import com.microsoft.cognitiveservices.speech.PropertyId;

/**
 * Contains detailed information about why a speaker recognition was canceled.
 */
public class SpeakerRecognitionCancellationDetails {

    /**
     * Creates an instance of SpeakerRecognitionCancellationDetails object for the canceled SpeakerRecognitionResult.
     * @param result The speaker recognition result that was canceled.
     * @return The cancellation details object being created.
     */
    public static com.microsoft.cognitiveservices.speech.speaker.SpeakerRecognitionCancellationDetails fromResult(SpeakerRecognitionResult result) {
        return new com.microsoft.cognitiveservices.speech.speaker.SpeakerRecognitionCancellationDetails(result);
    }

    private SpeakerRecognitionCancellationDetails(SpeakerRecognitionResult result) {
        Contracts.throwIfNull(result, "result");
        Contracts.throwIfNull(result.getImpl(), "resultHandle");

        IntRef valueRef = new IntRef(0);
        Contracts.throwIfFail(getCanceledReason(result.getImpl(), valueRef));
        this.reason = CancellationReason.values()[(int)valueRef.getValue() - 1]; // Native CancellationReason enum starts at 1!!
        Contracts.throwIfFail(getCanceledErrorCode(result.getImpl(), valueRef));
        this.errorCode = CancellationErrorCode.values()[(int)valueRef.getValue()];
        this.errorDetails = result.getProperties().getProperty(PropertyId.SpeechServiceResponse_JsonErrorDetails);
    }

    /**
     * Explicitly frees any external resource attached to the object
     */
    public void close() {
    }

    /**
     * The reason the recognition was canceled.
     * @return Specifies the reason canceled.
     */
    public CancellationReason getReason() {
        return this.reason;
    }

    /**
     * The error code in case of an unsuccessful recognition (when getReason() returns Error).
     * Added in version 1.1.0.
     * @return An error code that represents the error reason.
     */
    public CancellationErrorCode getErrorCode() {
        return this.errorCode;
    }

    /**
     * The error message in case of an unsuccessful recognition (when getReason() returns Error).
     * @return A String that represents the error details.
     */
    public String getErrorDetails() {
        return this.errorDetails;
    }

    /**
     * Returns a String that represents the cancellation details.
     * @return A String that represents the cancellation details.
     */
    @Override
    public String toString() {
        return "CancellationReason:" + this.reason +
                " ErrorCode: " + this.errorCode +
                " ErrorDetails:" + this.errorDetails;
    }

    private CancellationReason reason;
    private String errorDetails;
    private CancellationErrorCode errorCode;

    private final native long getCanceledReason(SafeHandle resultHandle, IntRef valueRef);
    private final native long getCanceledErrorCode(SafeHandle resultHandle, IntRef valueRef);
}
