//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech;

import java.lang.AutoCloseable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.IntRef;

/**
 * Contains detailed speech translation model information.
 * Note: close() must be called in order to release underlying resources held by the object.
 */
public class SpeechTranslationModel implements AutoCloseable {

    SpeechTranslationModel(IntRef modelInfo) {
        Contracts.throwIfNull(modelInfo, "modelInfo");

        this.modelInfoHandle = new SafeHandle(modelInfo.getValue(), SafeHandleType.SpeechTranslationModel);

        this.name = getName(modelInfoHandle);

        String sourceLanguagesString = getSourceLanguagesString(modelInfoHandle);
        if (sourceLanguagesString.isEmpty()) {
            this.sourceLanguages = new ArrayList<String>();
        } else {
            this.sourceLanguages = Arrays.asList(sourceLanguagesString.split("\\|"));
        }

        String targetLanguagesString = getTargetLanguagesString(modelInfoHandle);
        if (targetLanguagesString.isEmpty()) {
            this.targetLanguages = new ArrayList<String>();
        } else {
            this.targetLanguages = Arrays.asList(targetLanguagesString.split("\\|"));
        }

        this.path = getPath(modelInfoHandle);
        this.version = getVersion(modelInfoHandle);
    }

    /**
     * Gets the model name.
     * @return The model name.
     */
    public String getName() {
        return this.name;
    }

    /**
     * Gets the source languages that the model supports.
     * @return Source languages of the model.
     */
    public List<String> getSourceLanguages() {
        return this.sourceLanguages;
    }

    /**
     * Gets the target languages that the model supports.
     * @return Target languages of the model.
     */
    public List<String> getTargetLanguages() {
        return this.targetLanguages;
    }

    /**
     * Gets the model path (only valid for offline models).
     * @return The model path.
     */
    public String getPath() {
        return this.path;
    }

    /**
     * Gets the model version.
     * @return The model version.
     */
    public String getVersion() {
        return this.version;
    }

    /**
     * Explicitly frees any external resource attached to the object.
     */
    public void close() {
        if (this.modelInfoHandle != null) {
            this.modelInfoHandle.close();
            this.modelInfoHandle = null;
        }
    }

    /*! \cond INTERNAL */

    /**
     * Returns the model info implementation.
     * @return The implementation handle.
     */
    public SafeHandle getImpl() {
        return this.modelInfoHandle;
    }

    /*! \endcond */

    private final native String getName(SafeHandle modelInfoHandle);
    private final native String getSourceLanguagesString(SafeHandle modelInfoHandle);
    private final native String getTargetLanguagesString(SafeHandle modelInfoHandle);
    private final native String getPath(SafeHandle modelInfoHandle);
    private final native String getVersion(SafeHandle modelInfoHandle);

    private String name;
    private List<String> sourceLanguages;
    private List<String> targetLanguages;
    private String path;
    private String version;
    private SafeHandle modelInfoHandle = null;
}
