//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech;

import java.lang.AutoCloseable;
import java.io.Closeable;
import java.io.File;
import java.io.FileFilter;
import java.io.FilenameFilter;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.util.Arrays;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.IOError;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLDecoder;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.Files;
import java.nio.file.FileVisitResult;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.DirectoryStream;

/**
 * A helper class for loading native Speech SDK libraries from Java
 *
 * - Native libraries are placed in an OS-specific package assets folder
 *   (/ASSETS/{mac,linux,window}64)
 * - From this folder, a known list of libraries is extracted into a temporary folder.
 * - On Windows and OSX, these libraries are loaded in the list order (bottom
 *   to top dependency), by full extracted path name.
 *   TODO this may need to be revisited on OSX
 * - On Linux, only the last of these libraries (top dependency) is loaded,
 *   which resolves static dependency via RPATH; dynamic dependencies will be
 *   loaded later, also via RPATH.
 * - On Windows, optional external native libraries (such as carbon-tts-mock) are
 *   also copied from the folder where user's runnable jar is present to a temporary
 *   folder. If these optional external libraries are not found, no errors are reported
 *   and no exceptions are thrown.
 *
 * Workaround for a Windows (added in 1.17): Turns out that the temporary folder created here and the native DLL
 * files copied to it never got delete even though they were marked for deleteOnExit(). This resulted in ever-increasing
 * disk usage on Windows in the %TEMP% folder, as reported by a customer.
 * This is because on Windows the call to System.load() holds a lock on the files and they cannot be deleted when JVM shuts
 * down (there is no way to "unload" them first since there is no System.unload in Java).
 * The workaround is to make sure next time JVM loads this class we delete all previous unused folders. To mark a folder as "in use",
 * we add an empty file next to it with the same name and extension ".lock". We call deleteOnExit() on this lock file.
 * That file should be deleted when the JVM shuts down, since we did not call System.load() on that file. Hence folders without
 * a corresponding .lock file next to them are okay to be deleted when this class loads.
 * For example, the code in this class creates the following lock file and folder (here "353862700187557707" is an example
 * of the random number Windows picks):
 *   C:\Users\<username>\AppData\Local\Temp\speech-sdk-native-353862700187557707.lock - An empty file indicating the folder below is in use.
 *   C:\Users\<username>\AppData\Local\Temp\speech-sdk-native-353862700187557707      - A folder containing the native .dll files.
 * Deleting unlocked folders is done in a separate thread, so we do not slow down the loading of this class for the (very unlikely)
 * case where there are many unlocked folders to delete (e.g. when the device previously run many multiple recognizers in parallel).
 * We start running the cleanup thread *after* the new local temp folder is created and locked.
 * See relevant test named "testTempFolderCleanup" in NativeLibraryLoaderTests.java.
 */
class NativeLibraryLoader {

    private static class NativeLibrary {
        private String name;
        private boolean required;

        public NativeLibrary(String name, boolean required) {
            this.name = name;
            this.required = required;
        }

        public String getName() { return name; }
        public boolean getRequired() { return required; }

        private void setName() { this.name = name; }
        private void setRequired() { this.required = required; }
    }

    private static final String tempDirPrefix = "speech-sdk-native-";
    private static final String lockFileExtension = ".lock";

    private static NativeLibrary[] nativeList = new NativeLibrary[0];
    private static NativeLibrary[] externalNativeList = new NativeLibrary[0];
    private static String operatingSystem;
    private static Boolean windowsOperatingSystem;
    private static Boolean extractionDone = false;
    private static Boolean loadAll = false;
    private static File tempDir;

    static {
        operatingSystem = ("" + System.getProperty("os.name")).toLowerCase();
        windowsOperatingSystem = operatingSystem.contains("windows");

        try {
            if (windowsOperatingSystem) {

                // Start by creating the lock file, which will guard the folder to be created next
                final File lockFile = Files.createTempFile(tempDirPrefix, lockFileExtension).toFile();
                lockFile.createNewFile();
                lockFile.deleteOnExit();

                // Create a folder with a name derived from the lock file name, where the native binaries will be copied
                String tempDirName = lockFile.getAbsolutePath();
                tempDirName = tempDirName.substring(0, tempDirName.length() - lockFileExtension.length());
                tempDir = Files.createDirectory(Paths.get(tempDirName)).toFile();
                tempDir.deleteOnExit();

                // Start a thread to clean up all un-locked temp folders from other Java apps that run before
                Thread thread = new Thread(new DeleteUnlockedTempFolders());
                thread.start();

            } else {

                tempDir = Files.createTempDirectory(tempDirPrefix).toFile();
                tempDir.deleteOnExit();

            }
        }
        catch (IOException e) {
            throw new IOError(e);
        }
    }

    /**
     * Extract and load OS-specific libraries from the JAR file.
     */
    public static void loadNativeBinding() {
        try {
            extractNativeLibraries();

            // Load extracted libraries (either all or the last one)
            for (int i = 0; i < nativeList.length; i++)
            {
                if (loadAll || (i == nativeList.length - 1)) {
                    String libName = nativeList[i].getName();
                    String fullPathName = new File(tempDir.getCanonicalPath(), libName).getCanonicalPath();
                    if(!fullPathName.startsWith(tempDir.getCanonicalPath())) {
                        throw new SecurityException("illegal path");
                    }
                    try {
                        System.load(fullPathName);
                    }
                    catch (UnsatisfiedLinkError e) {
                        // Required library
                        if (nativeList[i].getRequired() == true) {
                            throw new UnsatisfiedLinkError(
                            String.format("Could not load a required Speech SDK library because of the following error: %s", e.getMessage()));
                        }
                    }
                }
            }
        }
        catch (Exception e) {
            // If nothing worked, throw exception
            throw new UnsatisfiedLinkError(
                    String.format("Could not extract/load all Speech SDK libraries because we encountered the following error: %s", e.getMessage()));
        }
    }

    private static void extractNativeLibraries() throws Exception {
        try {
            if (!extractionDone) {
                nativeList = getResourceLines();

                // Extract all operatingSystem specific native libraries to temporary location
                for (NativeLibrary library: nativeList) {
                    extractResourceFromPath(library, getResourcesPath());
                }

                externalNativeList = getExternalResourceLines();

                // For Windows, copy external native libraries to temporary location if they exist in current directory
                if (windowsOperatingSystem) {
                    String path = null;
                    try {
                        URL classResource = ClassLoader.getSystemClassLoader().getResource(".");
                        if (classResource != null)
                        {
                            path = new File(classResource.getPath()).getAbsolutePath();
                        }
                        else
                        {
                            path = new File(NativeLibraryLoader.class.getProtectionDomain().getCodeSource().getLocation().toURI()).getPath();
                            if (path.endsWith("jar"))
                            {
                                // get parent
                                path = new File(path).getParent();
                            }
                        }
                        path = URLDecoder.decode(path, "UTF-8");
                        if (path != null) {
                            for (NativeLibrary library: externalNativeList) {
                                copyLibraryFromPath(library, path);
                            }
                        }
                    }
                    catch (Exception e) {
                        System.err.println(
                        String.format("Could not copy external Speech SDK libraries because of the following error: %s", e.getMessage()));
                    }
                }
            }
        }
        finally {
            extractionDone = true;
        }
    }

    /**
     * Depending on the OS, return the list of libraries to be extracted, the
     * first of which is to be loaded as well.
     */
    private static NativeLibrary[] getResourceLines() throws IOException {

        if (operatingSystem.contains("linux")) {
            return new NativeLibrary[] {
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.core.so", true),
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.extension.kws.ort.so", false),
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.extension.kws.so", false),
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.extension.codec.so", false),
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.extension.audio.sys.so", false),
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.extension.lu.so", false),
                    // Bottom to top dependency: onnx -> runtime -> extension wrapper
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.extension.onnxruntime.so", false),
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.extension.embedded.sr.runtime.so", false),
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.extension.embedded.sr.so", false),
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.extension.embedded.tts.runtime.so", false),
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.extension.embedded.tts.so", false),
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.extension.telemetry.so", false),
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.extension.mas.so", false),
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.extension.vad.so", false),
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.java.bindings.so", true)
            };
        }
        else if (windowsOperatingSystem) {
            return new NativeLibrary[] {
                    new NativeLibrary("Microsoft.CognitiveServices.Speech.core.dll", true),
                    // Note: the Speech SDK core library loads extension DLLs
                    // relative to its location, so this one is currently only
                    // needed for extraction (TODO however due to 'loadAll ==
                    // true' below, we'll still load it later; should fix).
                    new NativeLibrary("Microsoft.CognitiveServices.Speech.extension.kws.ort.dll", false),
                    new NativeLibrary("Microsoft.CognitiveServices.Speech.extension.kws.dll", false),
                    new NativeLibrary("Microsoft.CognitiveServices.Speech.extension.codec.dll", false),
                    new NativeLibrary("Microsoft.CognitiveServices.Speech.extension.audio.sys.dll", false),
                    new NativeLibrary("Microsoft.CognitiveServices.Speech.extension.lu.dll", false),
                    new NativeLibrary("Microsoft.CognitiveServices.Speech.extension.silk_codec.dll", false),
                    // Bottom to top dependency: onnx -> runtime -> extension wrapper
                    new NativeLibrary("Microsoft.CognitiveServices.Speech.extension.onnxruntime.dll", false),
                    new NativeLibrary("Microsoft.CognitiveServices.Speech.extension.embedded.sr.runtime.dll", false),
                    new NativeLibrary("Microsoft.CognitiveServices.Speech.extension.embedded.sr.dll", false),
                    new NativeLibrary("Microsoft.CognitiveServices.Speech.extension.embedded.tts.dll", false),
                    new NativeLibrary("Microsoft.CognitiveServices.Speech.extension.telemetry.dll", false),
                    new NativeLibrary("Microsoft.CognitiveServices.Speech.extension.mas.dll", false),
                    new NativeLibrary("Microsoft.CognitiveServices.Speech.extension.vad.dll", false),
                    new NativeLibrary("Microsoft.CognitiveServices.Speech.java.bindings.dll", true)
            };
        }
        else if (operatingSystem.contains("mac") || operatingSystem.contains("darwin")) {
            return new NativeLibrary[] {
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.core.dylib", true),
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.extension.audio.sys.dylib", false),
                    // Bottom to top dependency: onnx -> runtime -> extension wrapper
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.extension.onnxruntime.dylib", false),
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.extension.embedded.sr.runtime.dylib", false),
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.extension.embedded.sr.dylib", false),
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.extension.embedded.tts.dylib", false),
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.extension.telemetry.dylib", false),
                    new NativeLibrary("libMicrosoft.CognitiveServices.Speech.java.bindings.dylib", true)
            };
        }

        throw new UnsatisfiedLinkError(
                String.format("The Speech SDK doesn't currently have native support for operating system: %s", operatingSystem));
    }

    /**
     * Depending on the OS, return the list of external optional libraries to be copied.
     */
    private static NativeLibrary[] getExternalResourceLines() {

        if (windowsOperatingSystem) {
            return new NativeLibrary[] {
                new NativeLibrary("carbon-tts-mock.dll", false)
            };
        }
        else if (operatingSystem.contains("linux")) {
            return new NativeLibrary[] {
            };
        }
        else if (operatingSystem.contains("mac") || operatingSystem.contains("darwin")) {
            return new NativeLibrary[] {
            };
        }

        return new NativeLibrary[0];
    }

    private static boolean useCentos7Binaries() {
        try {
            String overrideEOLSystem = System.getenv("SpeechSDK_UseCentos7Binaries");
            if (overrideEOLSystem != null) {
                return overrideEOLSystem.equalsIgnoreCase("true");
            }
            else {

                //lists all the files ending with -release in the etc folder
                File dir = new File("/etc/");
                File fileList[] = new File[0];
                if(dir.exists()){
                    fileList =  dir.listFiles(new FilenameFilter() {
                        public boolean accept(File dir, String filename) {
                            return filename.endsWith("-release");
                        }
                    });
                }
                //prints all the version-related files
                for (File f : fileList) {
                    try {
                        BufferedReader myReader = new BufferedReader(new FileReader(f));
                        String strLine = null;
                        while ((strLine = myReader.readLine()) != null) {
                            if(strLine.contains("Red Hat Enterprise Linux Server release 7.")) {
                                return true;
                            }
                            if(strLine.contains("CentOS Linux release 7.")) {
                                return true;
                            }
                        }
                        myReader.close();
                    } catch (Exception e) {
                        System.err.println("Error: " + e.getMessage());
                    }
                }

                return false;
            }
        }
        catch (Exception e) {
            System.err.println( String.format("Could not detect EOL Linux Distribution because of the following error: %s", e.getMessage()));
        }
        return false;
    }

    private static String getResourcesPath() {

        String speechPrefix = "/ASSETS/%s%s/";

        // determine if the VM runs on 64 or 32 bit
        String dataModelSize = System.getProperty("sun.arch.data.model");
        if(dataModelSize != null && dataModelSize.equals("64")) {
            dataModelSize = "64";
        }
        else {
            dataModelSize = "32";
        }

        if (operatingSystem.contains("linux")) {
            String osArchitecture = System.getProperty("os.arch");
            if (osArchitecture.contains("aarch64") || osArchitecture.contains("arm")) {
                return String.format(speechPrefix, "linux-arm", dataModelSize);
            }
            else if (osArchitecture.contains("amd64") || osArchitecture.contains("x86_64")) {
                if (useCentos7Binaries()) {
                    return String.format(speechPrefix, "centos7-x", dataModelSize);
                }
                else {
                    return String.format(speechPrefix, "linux-x", dataModelSize);
                }
            }
        }
        else if (windowsOperatingSystem) {
            loadAll = true; // signal to load all libraries
            return String.format(speechPrefix, "windows", dataModelSize);
        }
        else if (operatingSystem.contains("mac")|| operatingSystem.contains("darwin")) {
            String osArchitecture = System.getProperty("os.arch");
            if (osArchitecture.contains("aarch64") || osArchitecture.contains("arm")) {
                return String.format(speechPrefix, "osx-arm", dataModelSize);
            }
            else if (osArchitecture.contains("amd64") || osArchitecture.contains("x86_64")) {
                return String.format(speechPrefix, "osx-x", dataModelSize);
            }
        }

        throw new UnsatisfiedLinkError(
                String.format("The Speech SDK doesn't currently have native support for operating system: %s data model size %s", operatingSystem, dataModelSize));
    }

    private static void extractResourceFromPath(NativeLibrary library, String prefix) throws IOException {
        String libName = library.getName();
        File temp = new File(tempDir.getCanonicalPath(), libName);
        if (!temp.getCanonicalPath().startsWith(tempDir.getCanonicalPath())) {
            throw new SecurityException("illegal name " + temp.getCanonicalPath());
        }

        temp.createNewFile();
        temp.deleteOnExit();

        if (!temp.exists()) {
            throw new FileNotFoundException(String.format(
                    "Temporary file %s could not be created. Make sure you can write to this location.",
                    temp.getCanonicalPath()));
        }

        String path = prefix + libName;
        InputStream inStream = SpeechConfig.class.getResourceAsStream(path);
        if (inStream == null) {
            if (library.getRequired() == true) {
                throw new FileNotFoundException(String.format("Could not find resource %s in jar.", path));
            }
            else {
                // Optional library
                return;
            }
        }

        FileOutputStream outStream = null;
        byte[] buffer = new byte[1024*1024];
        int bytesRead;

        try {
            outStream = new FileOutputStream(temp);
            while ((bytesRead = inStream.read(buffer)) >= 0) {
                outStream.write(buffer, 0, bytesRead);
            }
        } finally {
            safeClose(outStream);
            safeClose(inStream);
        }
    }

    private static void copyLibraryFromPath(NativeLibrary library, String sourcePath) throws Exception {
        String libName = library.getName();
        File sourceFile = new File(sourcePath, libName);
        if (!sourceFile.exists()) {
            return;
        }

        File targetFile = new File(tempDir.getCanonicalPath(), libName);
        if (sourceFile.getCanonicalPath().equals(targetFile.getCanonicalPath())) {
            return;
        }
        targetFile.deleteOnExit();
        Files.copy(sourceFile.toPath(), targetFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
    }

    private static void safeClose(Closeable is) {
        if (is != null) {
            try {
                is.close();
            } catch (IOException e) {
                // ignored.
            }
        }
    }

    private static class DeleteUnlockedTempFolders implements Runnable {

        @Override
        public void run() {

            // Define a filter that will return all existing temp folders with names that start with "speech-sdk-native-".
            // Note that we exclude the *.lock suffix, in order to only capture the folders, not their corresponding .lock file.
            FileFilter tmpDirFilter = new FileFilter() {
                public boolean accept(File pathname) {
                    return pathname.getName().startsWith(tempDirPrefix) && !pathname.getName().endsWith(lockFileExtension);
                }
            };

            // List all folders that match this filer
            File tmpDir = new File(System.getProperty("java.io.tmpdir"));
            File[] filteredDirList = tmpDir.listFiles(tmpDirFilter);

            // Delete all folders which do not have a corresponding .lock file
            for (int i = 0; i < filteredDirList.length; i++) {

                // The .lock file we expect to see next to this folder if the folder is still used by some other Java app
                File lockFile = new File(filteredDirList[i].getAbsolutePath() + lockFileExtension);

                if (!lockFile.exists()) {

                    // If the *.lock file DOES NOT exist, it's safe to delete this folder. But before
                    // we can delete the folder, we need to delete all the files in that folder
                    try {
                        Files.walkFileTree(Paths.get(filteredDirList[i].getAbsolutePath()),
                            new SimpleFileVisitor<Path>() {
                                @Override
                                public FileVisitResult visitFile(Path file, BasicFileAttributes attr) throws IOException {
                                    Files.delete(file);
                                    return FileVisitResult.CONTINUE;
                                }
                            }
                        );
                    }
                    catch (IOException e) {
                        // okay to ignore, this is a best-effort delete
                    }

                    // Now that the directory is empty, we can delete it
                    filteredDirList[i].delete();
                }
            }
        }
    }
}
