package com.microsoft.azure.documentdb.internal.query;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.microsoft.azure.documentdb.PartitionKeyDefinition;
import com.microsoft.azure.documentdb.SqlQuerySpec;
import com.microsoft.azure.documentdb.internal.ServiceJNIWrapper;
import com.microsoft.azure.documentdb.internal.Utils;
import com.microsoft.azure.documentdb.internal.routing.PartitionKeyInternal;
import com.microsoft.azure.documentdb.internal.routing.PartitionKeyInternalHelper;
import com.microsoft.azure.documentdb.internal.routing.Range;

/**
 * Used internally to provide functionality to get query execution information for the Azure Cosmos DB database service.
 */
public class QueryPartitionProvider {

    private static final List<Range<String>> singleFullRange = new ArrayList<Range<String>>() {{
        add(new Range<>(
                PartitionKeyInternalHelper.MinimumInclusiveEffectivePartitionKey,
                PartitionKeyInternalHelper.MaximumExclusiveEffectivePartitionKey,
                true,
                false));
    }};
    private String queryEngineConfiguration;
    private volatile long serviceProvider;

    public QueryPartitionProvider(String queryEngineConfiguration) {
        this.serviceProvider = 0;
        this.queryEngineConfiguration = queryEngineConfiguration;
    }

    public PartitionedQueryExecutionInfo getPartitionQueryExcecutionInfo(
            SqlQuerySpec querySpec, PartitionKeyDefinition partitionKeyDefinition) {
        if (querySpec == null
                || partitionKeyDefinition == null
                || partitionKeyDefinition.getPaths().size() == 0) {
            return new PartitionedQueryExecutionInfo(
                    new QueryInfo(),
                    QueryPartitionProvider.singleFullRange
            );
        }
        this.initializeServiceProvider();

        PartitionedQueryExecutionInfoInternal partitionedQueryExecutionInfoInternal = ServiceJNIWrapper.getPartitionKeyRangesFromQuery(
                this.serviceProvider,
                querySpec,
                partitionKeyDefinition
        );

        final List<Range<String>> effectiveRanges =
                new ArrayList<Range<String>>(partitionedQueryExecutionInfoInternal.getQueryRanges().size());
        for (Range<PartitionKeyInternal> internalRange : partitionedQueryExecutionInfoInternal.getQueryRanges()) {
            effectiveRanges.add(new Range<String>(
                    PartitionKeyInternalHelper.getEffectivePartitionKeyString(internalRange.getMin(),partitionKeyDefinition, false),
                    PartitionKeyInternalHelper.getEffectivePartitionKeyString(internalRange.getMax(),partitionKeyDefinition, false),
                    internalRange.isMinInclusive(),
                    internalRange.isMaxInclusive()
            ));
        }
        Collections.sort(effectiveRanges, new Range.MinComparator<String>());

        return new PartitionedQueryExecutionInfo(
                partitionedQueryExecutionInfoInternal.getQueryInfo(),
                effectiveRanges
        );
    }

    private void initializeServiceProvider() {
        if (this.serviceProvider == 0) {
            synchronized (this) {
                if (this.serviceProvider == 0) {
                    this.serviceProvider = ServiceJNIWrapper.createServiceProvider(this.queryEngineConfiguration);
                }
            }
        }
    }
}
