package com.microsoft.azure.documentdb.internal;

import java.net.URI;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.microsoft.azure.documentdb.DocumentClientException;

/**
 * A RetryPolicy implementation that ensures session reads succeed across
 * geo-replicated databases.
 */
final class SessionReadRetryPolicy implements RetryPolicy {
    private final static int maxRetryCount = 1;
    private final static int retryIntervalInMS = 0;
    private final static Logger LOGGER = LoggerFactory.getLogger(SessionReadRetryPolicy.class);
    private final DocumentServiceRequest currentRequest;
    private final EndpointManager globalEndpointManager;
    private final boolean endpointDiscoveryEnable;
    private boolean canUseMultipleWriteLocations;
    private volatile int sessionTokenRetryCount = 0;
    private URI locationEndpoint;
    
    /**
     * Creates a new instance of the SessionReadRetryPolicy class.
     *
     * @param globalEndpointManager the GlobalEndpointManager instance.
     * @param request               the current request to be retried.
     */
    public SessionReadRetryPolicy(EndpointManager globalEndpointManager,
                                  DocumentServiceRequest request,
                                  boolean endpointDiscoveryEnable) {
        this.globalEndpointManager = globalEndpointManager;
        this.currentRequest = request;
        this.endpointDiscoveryEnable = endpointDiscoveryEnable;
        
        this.canUseMultipleWriteLocations = this.globalEndpointManager.canUseMultipleWriteLocations(request);

        // clear previous location-based routing directive
        request.clearRouteToLocation();

        // Resolve the endpoint for the request and pin the resolution to the resolved endpoint
        // This enables marking the endpoint unavailability on endpoint failover/unreachability
        this.locationEndpoint = this.globalEndpointManager.resolveServiceEndpoint(request);
        request.routeToLocation(this.locationEndpoint);
    }

    /**
     * Gets the number of milliseconds to wait before retry the operation.
     * <p>
     * For session read failures, we retry immediately.
     *
     * @return the number of milliseconds to wait before retry the operation.
     */
    public long getRetryAfterInMilliseconds() {
        return retryIntervalInMS;
    }

    /**
     * Should the caller retry the operation.
     * <p>
     * This retry policy should only be invoked if HttpStatusCode is 404 (NotFound)
     * and 1002 (ReadSessionNotAvailable).
     *
     * @param exception the exception to check.
     * @return true if should retry.
     */
    public boolean shouldRetry(DocumentClientException exception) {
        this.sessionTokenRetryCount++;
        // clear previous location-based routing directive
        this.currentRequest.clearRouteToLocation();

        if (!this.endpointDiscoveryEnable) {
            // if endpoint discovery is disabled, the request cannot be retried anywhere else
            return false;
        } else {
            if (this.canUseMultipleWriteLocations) {
                List<String> endpoints = Utils.isReadOnlyOperation(this.currentRequest.getOperationType())
                        ? this.globalEndpointManager.getOrderedReadEndpoints():
                            this.globalEndpointManager.getOrderedWriteEndpoints();

                if (this.sessionTokenRetryCount > endpoints.size()) {
                    // When use multiple write locations is true and the request has been tried 
                    // on all locations, then don't retry the request
                    return false;
                } else {
                    // set location-based routing directive based on request retry context
                    this.currentRequest.routeToLocation(this.sessionTokenRetryCount - 1, this.sessionTokenRetryCount > maxRetryCount);
                    this.currentRequest.setShouldClearSessionTokenOnSessionReadFailure(this.sessionTokenRetryCount == endpoints.size()); // clear on last attempt
                    
                    // Resolve the endpoint for the request and pin the resolution to the resolved endpoint
                    // This enables marking the endpoint unavailability on endpoint failover/unreachability
                    this.locationEndpoint = this.globalEndpointManager.resolveServiceEndpoint(this.currentRequest);
                    this.currentRequest.routeToLocation(this.locationEndpoint);
                    return true;
                }
            }
            else {
                if (this.sessionTokenRetryCount > maxRetryCount) {
                    // When cannot use multiple write locations, then don't retry the request if 
                    // we have already tried this request on the write location
                    return false;
                } else {
                    // set location-based routing directive based on request retry context
                    this.currentRequest.routeToLocation(this.sessionTokenRetryCount - 1, false);
                    this.currentRequest.setShouldClearSessionTokenOnSessionReadFailure(true);
                    
                    // Resolve the endpoint for the request and pin the resolution to the resolved endpoint
                    // This enables marking the endpoint unavailability on endpoint failover/unreachability
                    this.locationEndpoint = this.globalEndpointManager.resolveServiceEndpoint(this.currentRequest);
                    this.currentRequest.routeToLocation(this.locationEndpoint);
                    
                    return true;
                }
            }
        }
    }
}
