/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.appcenter.data;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteQueryBuilder;
import android.support.annotation.NonNull;
import android.support.annotation.VisibleForTesting;
import android.support.annotation.WorkerThread;
import com.microsoft.appcenter.data.LocalDocumentStorageDatabaseListener;
import com.microsoft.appcenter.data.Utils;
import com.microsoft.appcenter.data.exception.DataException;
import com.microsoft.appcenter.data.models.DocumentWrapper;
import com.microsoft.appcenter.data.models.LocalDocument;
import com.microsoft.appcenter.data.models.ReadOptions;
import com.microsoft.appcenter.data.models.WriteOptions;
import com.microsoft.appcenter.utils.AppCenterLog;
import com.microsoft.appcenter.utils.storage.DatabaseManager;
import com.microsoft.appcenter.utils.storage.SQLiteUtils;
import java.util.ArrayList;
import java.util.List;

@WorkerThread
class LocalDocumentStorage {
    @VisibleForTesting
    static final String FAILED_TO_READ_FROM_CACHE = "Failed to read from cache.";
    @VisibleForTesting
    static final String PARTITION_COLUMN_NAME = "partition";
    @VisibleForTesting
    static final String DOCUMENT_ID_COLUMN_NAME = "document_id";
    private static final String DOCUMENT_COLUMN_NAME = "document";
    private static final String ETAG_COLUMN_NAME = "etag";
    private static final String EXPIRATION_TIME_COLUMN_NAME = "expiration_time";
    private static final String DOWNLOAD_TIME_COLUMN_NAME = "download_time";
    private static final String OPERATION_TIME_COLUMN_NAME = "operation_time";
    private static final String PENDING_OPERATION_COLUMN_NAME = "pending_operation";
    private static final int VERSION = 2;
    private static final String BY_PARTITION_AND_DOCUMENT_ID_WHERE_CLAUSE = String.format("%s = ? AND %s = ?", "partition", "document_id");
    private static final ContentValues SCHEMA = LocalDocumentStorage.getContentValues("", "", "", "", 0L, 0L, 0L, "");
    private final DatabaseManager mDatabaseManager;

    LocalDocumentStorage(Context context, String userTable) {
        this.mDatabaseManager = new DatabaseManager(context, "com.microsoft.appcenter.documents", "app_documents", 2, SCHEMA, (DatabaseManager.Listener)new LocalDocumentStorageDatabaseListener(userTable));
        if (userTable != null) {
            this.createTableIfDoesNotExist(userTable);
        }
    }

    void createTableIfDoesNotExist(String userTable) {
        this.mDatabaseManager.createTable(userTable, SCHEMA, new String[]{PARTITION_COLUMN_NAME, DOCUMENT_ID_COLUMN_NAME});
    }

    void resetDatabase() {
        this.mDatabaseManager.resetDatabase();
    }

    <T> void writeOffline(String table, DocumentWrapper<T> document, WriteOptions writeOptions) {
        this.write(table, document, writeOptions, "CREATE");
    }

    <T> void writeOnline(String table, DocumentWrapper<T> document, WriteOptions writeOptions) {
        this.write(table, document, writeOptions, null);
    }

    static boolean hasPendingOperation(@NonNull List<LocalDocument> localDocuments) {
        for (LocalDocument doc : localDocuments) {
            if (!doc.hasPendingOperation() || doc.getOperation().equals("DELETE")) continue;
            return true;
        }
        return false;
    }

    private static SQLiteQueryBuilder getPartitionAndDocumentIdQueryBuilder() {
        SQLiteQueryBuilder builder = SQLiteUtils.newSQLiteQueryBuilder();
        builder.appendWhere((CharSequence)BY_PARTITION_AND_DOCUMENT_ID_WHERE_CLAUSE);
        return builder;
    }

    private <T> long write(String table, DocumentWrapper<T> document, WriteOptions writeOptions, String pendingOperationValue) {
        if (writeOptions.getDeviceTimeToLive() == 0) {
            return 0L;
        }
        AppCenterLog.debug((String)"AppCenter", (String)String.format("Trying to replace %s:%s document to cache", document.getPartition(), document.getId()));
        long now = System.currentTimeMillis();
        ContentValues values = LocalDocumentStorage.getContentValues(document.getPartition(), document.getId(), document.getJsonValue(), document.getETag(), writeOptions.getDeviceTimeToLive() == -1 ? -1L : now + (long)writeOptions.getDeviceTimeToLive() * 1000L, document.getLastUpdatedDate().getTime(), document.getLastUpdatedDate().getTime(), pendingOperationValue);
        return this.mDatabaseManager.replace(table, values, new String[]{PARTITION_COLUMN_NAME, DOCUMENT_ID_COLUMN_NAME});
    }

    private <T> long createOffline(String table, DocumentWrapper<T> document, WriteOptions writeOptions) {
        return this.write(table, document, writeOptions, "CREATE");
    }

    private <T> long updateOffline(String table, DocumentWrapper<T> document, WriteOptions writeOptions) {
        return this.write(table, document, writeOptions, "REPLACE");
    }

    boolean deleteOffline(String table, DocumentWrapper<Void> documentWrapper, WriteOptions writeOptions) {
        documentWrapper.setFromCache(true);
        return this.deleteOffline(table, documentWrapper.getPartition(), documentWrapper.getId(), writeOptions);
    }

    boolean deleteOffline(String table, String partition, String documentId, WriteOptions writeOptions) {
        DocumentWrapper<Void> cachedDocument = this.read(table, partition, documentId, Void.class, null);
        DocumentWrapper<Object> writeDocument = new DocumentWrapper<Object>(null, partition, documentId, cachedDocument.getETag(), System.currentTimeMillis());
        return this.write(table, writeDocument, writeOptions, "DELETE") > 0L;
    }

    boolean deleteOnline(String table, String partition, String documentId) {
        AppCenterLog.debug((String)"AppCenter", (String)String.format("Trying to delete %s:%s document from cache", partition, documentId));
        try {
            return this.mDatabaseManager.delete(table, BY_PARTITION_AND_DOCUMENT_ID_WHERE_CLAUSE, new String[]{partition, documentId}) > 0;
        }
        catch (RuntimeException e) {
            AppCenterLog.error((String)"AppCenter", (String)"Failed to delete from cache: ", (Throwable)e);
            return false;
        }
    }

    <T> DocumentWrapper<T> createOrUpdateOffline(String table, String partition, String documentId, T document, Class<T> documentType, WriteOptions writeOptions) {
        long rowId;
        DocumentWrapper<T> cachedDocument = this.read(table, partition, documentId, documentType, null);
        cachedDocument.setFromCache(true);
        if (cachedDocument.getError() != null && cachedDocument.getError().getMessage().equals(FAILED_TO_READ_FROM_CACHE)) {
            return cachedDocument;
        }
        DocumentWrapper<T> writeDocument = new DocumentWrapper<T>(document, partition, documentId, cachedDocument.getETag(), System.currentTimeMillis() / 1000L);
        long l = rowId = cachedDocument.getError() != null ? this.createOffline(table, writeDocument, writeOptions) : this.updateOffline(table, writeDocument, writeOptions);
        if (rowId < 0L) {
            writeDocument = new DocumentWrapper(new DataException("Failed to write document into cache."));
        }
        writeDocument.setFromCache(true);
        return writeDocument;
    }

    List<LocalDocument> getPendingOperations(String table) {
        return this.queryLocalStorage(table, "pending_operation IS NOT NULL", null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<LocalDocument> queryLocalStorage(String table, String whereClause, String[] selectionArgs) {
        ArrayList<LocalDocument> result = new ArrayList<LocalDocument>();
        if (table == null) {
            return result;
        }
        SQLiteQueryBuilder builder = SQLiteUtils.newSQLiteQueryBuilder();
        builder.appendWhere((CharSequence)whereClause);
        try (Cursor cursor = this.mDatabaseManager.getCursor(table, builder, null, selectionArgs, null);){
            while (cursor.moveToNext()) {
                ContentValues values = this.mDatabaseManager.buildValues(cursor);
                String pendingOperation = values.getAsString(PENDING_OPERATION_COLUMN_NAME);
                Long expirationTime = values.getAsLong(EXPIRATION_TIME_COLUMN_NAME);
                result.add(new LocalDocument(table, pendingOperation, values.getAsString(PARTITION_COLUMN_NAME), values.getAsString(DOCUMENT_ID_COLUMN_NAME), values.getAsString(DOCUMENT_COLUMN_NAME), expirationTime, values.getAsLong(DOWNLOAD_TIME_COLUMN_NAME), values.getAsLong(OPERATION_TIME_COLUMN_NAME)));
            }
        }
        return result;
    }

    List<LocalDocument> getDocumentsByPartition(String table, String partition, ReadOptions readOptions) {
        ArrayList<LocalDocument> result = new ArrayList<LocalDocument>();
        if (table == null) {
            return result;
        }
        List<LocalDocument> localDocuments = this.queryLocalStorage(table, "partition = ?", new String[]{partition});
        for (LocalDocument localDocument : localDocuments) {
            boolean notDeleteOrNonpendingDocument;
            String pendingOperation = localDocument.getOperation();
            boolean isExpired = ReadOptions.isExpired(localDocument.getExpirationTime());
            boolean isNotPendingDeleteOperation = pendingOperation != null && !pendingOperation.equals("DELETE");
            boolean isDocumentWithoutPendingOperation = pendingOperation == null;
            boolean bl = notDeleteOrNonpendingDocument = isNotPendingDeleteOperation || isDocumentWithoutPendingOperation;
            if (isExpired && notDeleteOrNonpendingDocument || readOptions.getDeviceTimeToLive() == 0) {
                this.mDatabaseManager.delete(table, BY_PARTITION_AND_DOCUMENT_ID_WHERE_CLAUSE, new String[]{localDocument.getPartition(), localDocument.getDocumentId()});
            }
            if (isExpired || !notDeleteOrNonpendingDocument) continue;
            result.add(localDocument);
        }
        return result;
    }

    static boolean isValidPartitionName(String partition) {
        return "readonly".equals(partition) || "user".equals(partition);
    }

    private static ContentValues getContentValues(String partition, String documentId, String document, String eTag, long expirationTime, long downloadTime, long operationTime, String pendingOperation) {
        ContentValues values = new ContentValues();
        values.put(PARTITION_COLUMN_NAME, partition);
        values.put(DOCUMENT_ID_COLUMN_NAME, documentId);
        values.put(DOCUMENT_COLUMN_NAME, document);
        values.put(ETAG_COLUMN_NAME, eTag);
        values.put(EXPIRATION_TIME_COLUMN_NAME, Long.valueOf(expirationTime));
        values.put(DOWNLOAD_TIME_COLUMN_NAME, Long.valueOf(downloadTime));
        values.put(OPERATION_TIME_COLUMN_NAME, Long.valueOf(operationTime));
        values.put(PENDING_OPERATION_COLUMN_NAME, pendingOperation);
        return values;
    }

    void updatePendingOperation(LocalDocument operation) {
        ContentValues values = LocalDocumentStorage.getContentValues(operation.getPartition(), operation.getDocumentId(), operation.getDocument(), operation.getETag(), operation.getExpirationTime(), operation.getDownloadTime(), operation.getOperationTime(), operation.getOperation());
        this.mDatabaseManager.replace(operation.getTable(), values, new String[0]);
    }

    @NonNull
    <T> DocumentWrapper<T> read(String table, String partition, String documentId, Class<T> documentType, ReadOptions readOptions) {
        Cursor cursor;
        AppCenterLog.debug((String)"AppCenter", (String)String.format("Trying to read %s:%s document from cache", partition, documentId));
        try {
            cursor = this.mDatabaseManager.getCursor(table, LocalDocumentStorage.getPartitionAndDocumentIdQueryBuilder(), null, new String[]{partition, documentId}, "expiration_time DESC");
        }
        catch (RuntimeException e) {
            AppCenterLog.error((String)"AppCenter", (String)"Failed to read from cache: ", (Throwable)e);
            return new DocumentWrapper(FAILED_TO_READ_FROM_CACHE, e);
        }
        ContentValues values = this.mDatabaseManager.nextValues(cursor);
        cursor.close();
        if (values != null) {
            if (ReadOptions.isExpired(values.getAsLong(EXPIRATION_TIME_COLUMN_NAME))) {
                this.mDatabaseManager.delete(table, (Object)values.getAsLong("oid"));
                String errorMessage = "Document was found in the cache, but it was expired. The cached document has been invalidated.";
                AppCenterLog.debug((String)"AppCenter", (String)errorMessage);
                return new DocumentWrapper(new DataException(errorMessage));
            }
            String document = values.getAsString(DOCUMENT_COLUMN_NAME);
            String eTag = values.getAsString(ETAG_COLUMN_NAME);
            long operationTime = values.getAsLong(OPERATION_TIME_COLUMN_NAME);
            DocumentWrapper<T> documentWrapper = Utils.parseDocument(document, partition, documentId, eTag, operationTime / 1000L, documentType);
            documentWrapper.setFromCache(true);
            documentWrapper.setPendingOperation(values.getAsString(PENDING_OPERATION_COLUMN_NAME));
            if (readOptions != null) {
                if (readOptions.getDeviceTimeToLive() == 0) {
                    this.mDatabaseManager.delete(table, (Object)values.getAsLong("oid"));
                } else if (!documentWrapper.hasFailed()) {
                    this.write(table, documentWrapper, new WriteOptions(readOptions.getDeviceTimeToLive()), values.getAsString(PENDING_OPERATION_COLUMN_NAME));
                }
            }
            return documentWrapper;
        }
        AppCenterLog.debug((String)"AppCenter", (String)"Document was found in the cache, but it was expired. The cached document has been invalidated.");
        return new DocumentWrapper(new DataException("Document was not found in the cache."));
    }
}

