package com.mediatek.mcs.push;

import android.app.Activity;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.Bundle;
import android.support.v4.app.NotificationCompat;
import com.google.gson.Gson;
import com.mediatek.mcs.McsLog;
import com.mediatek.mcs.R;
import lombok.AllArgsConstructor;
import lombok.Getter;

public class McsPushBroadcastReceiver extends BroadcastReceiver {
  static final String ACTION_RECEIVE = "com.mediatek.mcs.push.intent.RECEIVE";
  static final String ACTION_OPEN = "com.mediatek.mcs.push.intent.OPEN";
  static final String ACTION_DISMISS = "com.mediatek.mcs.push.intent.DISMISS";

  static final int NOTIFICATION_ID = 1;
  static final String KEY_PUSH_DATA = "pushData";

  static final String PROPERTY_PUSH_ICON = "com.mediatek.mcs.push.notification_icon";

  @Override public void onReceive(Context context, Intent intent) {
    McsLog.d(context.getPackageName() + ": " + intent.getAction());

    if (ACTION_RECEIVE.equals(intent.getAction())) {
      onPushReceive(context, intent);
    } else if (ACTION_OPEN.equals(intent.getAction())) {
      onPushOpen(context, intent);
    } else if (ACTION_DISMISS.equals(intent.getAction())) {
      onPushDismiss(context, intent);
    }
  }

  protected void onPushReceive(Context context, Intent intent) {
    this.onMessageReceived(context, intent.getBundleExtra(McsPushBroadcastReceiver.KEY_PUSH_DATA));
  }

  protected void onPushOpen(Context context, Intent intent) {
    Class openActivity = getActivity(context, intent);
    if (openActivity != null) {
      Intent nextIntent = new Intent(context, openActivity);
      nextIntent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
      context.startActivity(nextIntent);
    }
  }

  protected void onPushDismiss(Context context, Intent intent) {
    McsLog.d("onPushDismiss. User dismiss the notif");
  }

  protected void onMessageReceived(Context context, Bundle data) {
    Device device = parseNotifDevice(data);
    String message = parseNotifMessage(data);

    String contentTitle = context.getString(context.getApplicationInfo().labelRes);
    NotificationCompat.BigTextStyle notifStyle = new NotificationCompat.BigTextStyle().setBigContentTitle(contentTitle)
        .setSummaryText(device.getName())
        .bigText(message);

    Notification notif =
        new NotificationCompat.Builder(context)
            .setAutoCancel(true)
            .setStyle(notifStyle)
            .setWhen(System.currentTimeMillis())
            .setDefaults(Notification.DEFAULT_ALL)
            .setSmallIcon(getSmallIcon(context))
            .setLargeIcon(getLargeIcon(context, device))
            .setContentTitle(contentTitle)
            .setContentText(message)
            .setTicker(message)
            .setContentIntent(PendingIntent.getBroadcast(context, 0, new Intent(ACTION_OPEN), 0))
            .setDeleteIntent(PendingIntent.getBroadcast(context, 0, new Intent(ACTION_DISMISS), 0))
            .setGroup(context.getString(context.getApplicationInfo().labelRes))
            .setGroupSummary(false)
        .build();

    int notifId = (int) System.currentTimeMillis();
    NotificationManager mNotificationManager = (NotificationManager)
        context.getSystemService(Context.NOTIFICATION_SERVICE);
    mNotificationManager.notify(notifId, notif);
  }

  protected Class<? extends Activity> getActivity(Context context, Intent intent) {
    return PushService.getDefaultPushCallback();
  }

  protected int getSmallIcon(Context context) {
    try {
      ApplicationInfo app = context.getPackageManager()
          .getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
      Bundle bundle = app.metaData;
      return bundle.getInt(PROPERTY_PUSH_ICON);
    } catch (PackageManager.NameNotFoundException | NullPointerException e) {
      return R.drawable.default_notif_icon;
    }
  }

  protected Bitmap getLargeIcon(Context context, Device device) {
    return BitmapFactory.decodeResource(context.getResources(), getSmallIcon(context));
  }

  protected String parseNotifMessage(Bundle data) {
    String msg = data.getString("alert");
    return (msg == null) ? "" : msg;
  }

  protected Device parseNotifDevice(Bundle data) {
    try {
      return new Gson().fromJson(data.get("device").toString(), Device.class);
    } catch (Exception e) {
      McsLog.e(e);
      return new Device("", "", "");
    }
  }

  @Getter @AllArgsConstructor public class Device {
    private String id;
    private String name;
    private String url;
  }
}
