package com.mediatek.mcs.net;

import com.android.volley.AuthFailureError;
import com.android.volley.NetworkResponse;
import com.android.volley.VolleyError;
import com.android.volley.toolbox.JsonObjectRequest;
import com.google.gson.Gson;
import com.mediatek.mcs.R;
import com.mediatek.mcs.Utils.UIUtils;
import com.mediatek.mcs.domain.McsResponse;
import com.mediatek.mcs.domain.McsSession;
import com.mediatek.mcs.entity.AuthEntity;
import java.net.HttpURLConnection;
import java.util.Map;
import org.json.JSONObject;

public class AuthedJsonRequest extends JsonObjectRequest {
  McsResponse.ErrorListener authTokenErrorListener = error -> {
    /**
     * token would be expired every 2 weeks (different from access_token)
     */
    if (isTokenExpired((VolleyError) error)) {
      UIUtils.toast(R.string.error_token_expired);
      logMeOut();
    } else {
      UIUtils.toast(RequestUtils.parseMcsNetworkError(error));
    }
  };
  private int method;
  private String url;
  private Map<String, String> headers;
  private String requestBody;
  private McsResponse.SuccessListener<JSONObject> listener;
  private McsResponse.ErrorListener errorListener;

  /**
   * Create a new request with some customized headers & requestBody.
   */
  public AuthedJsonRequest(int method, String url, Map<String, String> headers, String requestBody,
      McsResponse.SuccessListener<JSONObject> listener, McsResponse.ErrorListener errorListener) {
    super(method, url, requestBody, McsRequestHelper.transformListener(listener),
        McsRequestHelper.transformErrorListener(errorListener));
    this.headers = headers;
    initRequest(method, url, headers, requestBody, listener, errorListener);
  }

  /**
   * Get the customized headers for authenticated request.
   * AuthHeader includes Bearer Token and Content-Type application/json.
   * If the request carried headers, the AuthHeader would add upon it rather than
   * overwrite it, which means you only need to put additional headers.
   *
   * @return customized AuthHeader for authenticated request.
   * @throws AuthFailureError
   */
  @Override public Map<String, String> getHeaders() throws AuthFailureError {
    return RequestUtils.putAuthHeader(headers);
  }

  /**
   * @param method required.
   * @param url required.
   * @param headers optional. Default is as
   * @param requestBody optional. Default is null
   * @param listener required.
   * @param errorListener optional. Default is as {@link RequestErrorListener}
   * @see #getHeaders()
   */
  private void initRequest(int method, String url, Map<String, String> headers, String requestBody,
      McsResponse.SuccessListener<JSONObject> listener, McsResponse.ErrorListener errorListener) {
    this.method = method;
    this.url = url;
    this.headers = headers;
    this.requestBody = requestBody;
    this.listener = listener;
    this.errorListener = errorListener;
  }

  @Override public void deliverError(VolleyError error) {
    if (isTokenExpired(error)) {
      requestAccessToken();
    } else {
      super.deliverError(error);
    }
  }

  public boolean isTokenExpired(VolleyError error) {
    NetworkResponse networkResponse = error.networkResponse;
    return (networkResponse != null
        && networkResponse.statusCode == HttpURLConnection.HTTP_UNAUTHORIZED);
  }

  private void requestAccessToken() {
    RequestManager.sendInBackground(new SignInRequest(
        McsJsonRequest.Method.POST,
        RequestApi.POST_ACCESS_TOKEN,
        SignInRequest.getSignInParams(McsSession.getInstance().getCurrentUser().getToken()),
        response -> {

          McsSession.getInstance().saveTokenResultsToPref(
              new Gson().fromJson(response.toString(), AuthEntity.AccessToken.class)
                  .getResults());

          resendRequest();
        },
        authTokenErrorListener
    ));
  }

  private void resendRequest() {
    RequestManager.sendInBackground(new AuthedJsonRequest(
        this.method,
        this.url,
        this.headers,
        this.requestBody,
        this.listener,
        this.errorListener
    ));
  }

  private void logMeOut() {
    McsSession.getInstance().getCurrentUser().clearPref();
    android.os.Process.killProcess(android.os.Process.myPid());
  }
}
