/**
 Copyright (c) 2021 MarkLogic Corporation

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

'use strict';

import consts from "/data-hub/5/impl/consts.mjs";
import entityLib from "/data-hub/5/impl/entity-lib.mjs";
import entitySearchLib from "/data-hub/5/entities/entity-search-lib";

const sem = require("/MarkLogic/semantics.xqy");

const hubCentralConfig = cts.doc("/config/hubCentral.json");
const graphDebugTraceEnabled = xdmp.traceEnabled(consts.TRACE_GRAPH_DEBUG);
const graphTraceEnabled = xdmp.traceEnabled(consts.TRACE_GRAPH) || graphDebugTraceEnabled;
const graphTraceEvent = xdmp.traceEnabled(consts.TRACE_GRAPH) ? consts.TRACE_GRAPH : consts.TRACE_GRAPH_DEBUG;
const rdfTypeIri = sem.iri("http://www.w3.org/1999/02/22-rdf-syntax-ns#type");

// Alterations to rdf:type values that qualify as Concepts in Hub Central graph views go in this function.
function getRdfConceptTypes() {
  return [
    sem.curieExpand("rdf:Class"),
    sem.curieExpand("owl:Class"),
    sem.curieExpand("skos:Concept")
  ];
}

// Alterations to label predicates in order of priority
function getOrderedLabelPredicates() {
  return [
    sem.curieExpand("skos:prefLabel"),
    sem.curieExpand("skos:label"),
    sem.curieExpand("rdfs:label")
  ];
}

function getEntityNodesWithRelated(entityTypeIRIs, relatedEntityTypeIRIs, predicateConceptList, entitiesDifferentFromBaseAndRelated, conceptFacetList, ctsQueryCustom, limit = 100) {
  const startTime = xdmp.elapsedTime();
  const archivedCollections = getArchivedCollections();
  const archivedCollectionQuery = fn.exists(archivedCollections) ? cts.collectionQuery(archivedCollections) : null;
  const positiveQuery = cts.andQuery([
    ctsQueryCustom,
    cts.tripleRangeQuery(null, rdfTypeIri, entityTypeIRIs)
  ]);
  const docURIs = cts.uris(null, [`truncate=${limit}`, "concurrent", "document", "score-zero", "eager"],
    archivedCollectionQuery ? cts.andNotQuery(
      positiveQuery,
      archivedCollectionQuery) : positiveQuery
  ).toArray();
  const relatedLimit = Math.max(1, relatedEntityTypeIRIs.length) * limit;
  const conceptLimit = Math.max(1, predicateConceptList.length) * relatedLimit;

  const store = archivedCollectionQuery ? sem.store(null, cts.notQuery(archivedCollectionQuery)) : null;
  const results = sem.sparql(`PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
                   PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>
                   PREFIX xsd: <http://www.w3.org/2001/XMLSchema#>
                   SELECT * WHERE {
                   {
                     {
                       # The primary entity query
                       SELECT ?subjectIRI ?subjectIdentifier (MIN(?label) AS ?subjectLabel) ?predicateIRI (MIN(?anyPredicateLabel) AS ?predicateLabel) ?objectIRI ?objectIdentifier ?isInverse WHERE {
                          ?subjectIRI rdf:type $entityTypeIRIs;
                            rdfs:isDefinedBy ?subjectIdentifier.
                          FILTER (?subjectIdentifier = $docURIs)
                          OPTIONAL {
                            ?subjectIRI $labelIRI ?label.
                          }
                          # The primary first level connections on related entities
                          OPTIONAL {
                              {
                                  ?subjectIRI ?predicateIRI ?objectIRI.
                                  BIND("false"^^xsd:boolean AS ?isInverse)
                              } UNION {
                                  ?objectIRI ?predicateIRI ?subjectIRI.
                                  BIND("true"^^xsd:boolean AS ?isInverse)
                              }
                              ?objectIRI rdf:type $entityTypeOrConceptIRI;
                                  rdfs:isDefinedBy ?objectIdentifier.
                              OPTIONAL {
                                ?predicateIRI $labelIRI ?anyPredicateLabel.
                              }
                          }
                        }
                        GROUP BY ?subjectIRI ?subjectIdentifier ?predicateIRI ?objectIdentifier
                      }
                      BIND("entity graph" AS ?origin)
                    } UNION {
                         # The matching concepts
                       {
                        SELECT ?subjectIRI ?subjectIdentifier ?predicateIRI ?objectIdentifier ?conceptClassName (MIN(?anyConceptLabel) AS ?conceptLabel)  WHERE {
                                ?subjectIRI rdf:type $entityTypeIRIs;
                                  ?predicateIRI  ?objectIdentifier;
                                  rdfs:isDefinedBy ?subjectIdentifier.
                                FILTER (?subjectIdentifier = $docURIs && isIRI(?predicateIRI) && ?predicateIRI = $predicateConceptList)
                                 # Concept class
                                 OPTIONAL {
                                    $entityTypeIRIs <http://www.marklogic.com/data-hub#relatedConcept> ?conceptClassName.
                                    ?conceptClassName <http://www.marklogic.com/data-hub#conceptPredicate> ?predicateIRI.
                                    ?subjectIRI ?predicateIRI ?objectIdentifier.
                                    FILTER (isIRI(?predicateIRI) && ?predicateIRI = $predicateConceptList)
                                    OPTIONAL {
                                      ?objectIdentifier $labelIRI ?anyConceptLabel.
                                    }
                                 }
                                 ${conceptFacetList != null && conceptFacetList.length >= 1 ? "FILTER (isIRI(?objectIdentifier) && ?objectIdentifier = $conceptFacetList)" : ""}
                         }
                        GROUP BY ?subjectIdentifier ?predicateIRI ?objectIdentifier ?conceptClassName
                        LIMIT ${conceptLimit}
                       }
                       BIND("concept graph" AS ?origin)
                 }
             }
    `, {docURIs, conceptFacetList, entitiesDifferentFromBaseAndRelated, entityTypeIRIs, predicateConceptList, entityTypeOrConceptIRI: relatedEntityTypeIRIs.concat(entityTypeIRIs).concat(getRdfConceptTypes()), labelIRI: getOrderedLabelPredicates()}, null, store);
  if (graphTraceEnabled) {
    xdmp.trace(graphTraceEvent, `Graph search results: '${fn.count(results)}' in ${xdmp.elapsedTime().subtract(startTime)}`);
  }
  return results.toArray();
}

let _allEntityIds = null;

function getAllEntityIds() {
  if (!_allEntityIds) {
    _allEntityIds = fn.collection(entityLib.getModelCollection()).toArray().map(model => model.toObject().info.title);
  }
  return _allEntityIds;
}

function getEntityNodesByConcept(conceptIRI, limit) {
  if (graphTraceEnabled) {
    xdmp.trace(graphTraceEvent, `Creating plan for graph nodes and edges for concept '${conceptIRI}' with limit of ${limit}`);
  }
  const store = sem.store(null, cts.andNotQuery(cts.orQuery([cts.tripleRangeQuery(conceptIRI, null, null), cts.tripleRangeQuery(null, null, conceptIRI)]), cts.collectionQuery(getArchivedCollections())));
  const results = sem.sparql(`PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
                   PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>
                   PREFIX xsd: <http://www.w3.org/2001/XMLSchema#>
                   SELECT ?subjectIRI ?subjectIdentifier ?predicateIRI ?objectIRI ?objectIdentifier ?isInverse  WHERE {
                      {
                          ?objectIRI ?predicateIRI ?subjectIRI.
                          BIND("true"^^xsd:boolean AS ?isInverse)
                      } UNION {
                          ?subjectIRI ?predicateIRI ?objectIRI.
                          BIND("false"^^xsd:boolean AS ?isInverse)
                      }
                      FILTER (isIRI(?objectIRI) && isIRI(?subjectIRI) && ?subjectIRI = $objectConceptIRI)
                      OPTIONAL {
                          ?subjectIRI rdfs:isDefinedBy ?docURI.
                      }
                      BIND(IF(BOUND(?docURI), ?docURI, ?subjectIRI) AS ?subjectIdentifier)

                      OPTIONAL {
                          ?objectIRI rdfs:isDefinedBy ?objectURI.
                      }
                      BIND(IF(BOUND(?objectURI), ?objectURI, ?objectIRI) AS ?objectIdentifier)

                   }
                   LIMIT $limit
      `, {objectConceptIRI: conceptIRI, limit}, [], store);
  if (graphTraceEnabled) {
    xdmp.trace(graphTraceEvent, `Retrieved ${fn.count(results)} rows for concept '${conceptIRI}' with limit of ${limit}`);
  }
  if (graphDebugTraceEnabled) {
    xdmp.trace(graphTraceEvent, `Results for node expand '${xdmp.describe(results, Sequence.from([]), Sequence.from([]))}'`);
  }
  return results.toArray();

}

function getEntityNodesExpandingConcept(objectConceptIRI, limit) {
  if (graphTraceEnabled) {
    xdmp.trace(graphTraceEvent, `Creating plan for graph nodes and edges for concept '${objectConceptIRI}' with limit of ${limit}`);
  }
  const store = sem.store(null, cts.andNotQuery(cts.orQuery([cts.tripleRangeQuery(objectConceptIRI, null, null), cts.tripleRangeQuery(null, null, objectConceptIRI)]), cts.collectionQuery(getArchivedCollections())));
  const results =  sem.sparql(`PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
                 PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>
                 PREFIX xsd: <http://www.w3.org/2001/XMLSchema#>
                 SELECT ?subjectIRI  ?subjectIdentifier ?predicateIRI ?objectIRI ?objectIdentifier ?isInverse ?origin WHERE {
                    {
                      ?objectIRI ?predicateIRI ?subjectIRI.
                      BIND("true"^^xsd:boolean AS ?isInverse)
                    } UNION {
                      ?subjectIRI ?predicateIRI ?objectIRI.
                      BIND("false"^^xsd:boolean AS ?isInverse)
                    }
                    FILTER (isIRI(?objectIRI) && isIRI(?subjectIRI) && ?subjectIRI = $objectConceptIRI)
                    OPTIONAL {
                      ?subjectIRI rdfs:isDefinedBy ?docURI.
                    }
                    OPTIONAL {
                      ?objectIRI rdfs:isDefinedBy ?objectDocURI.
                    }
                    BIND(IF(BOUND(?docURI), ?docURI, ?subjectIRI) AS ?subjectIdentifier)
                    BIND(IF(BOUND(?objectDocURI), ?objectDocURI, ?subjectIRI) AS ?objectIdentifier)
                    BIND(IF(BOUND(?objectDocURI), "entity graph", "concept graph") AS ?origin)
                  }
                 LIMIT $limit
    `, {objectConceptIRI, limit}, [], store);
  if (graphTraceEnabled) {
    xdmp.trace(graphTraceEvent, `Retrieved ${fn.count(results)} rows for concept '${objectConceptIRI}' with limit of ${limit}`);
  }
  if (graphDebugTraceEnabled) {
    xdmp.trace(graphTraceEvent, `Results for node expand '${xdmp.describe(results, Sequence.from([]), Sequence.from([]))}'`);
  }
  return results.toArray();
}

function getEntityNodesByDocument(docURI, limit) {
  if (graphTraceEnabled) {
    xdmp.trace(graphTraceEvent, `Creating plan for graph nodes and edges for document '${docURI}' with limit of ${limit}`);
  }
  // Using separate sem.sparql calls, instead of an Optic join to avoid a seg fault with ML 10.0-7
  const bindings = {parentDocURI: docURI, labelIRI: getOrderedLabelPredicates(), allPredicates: getAllPredicates(), limit};
  const collectionQuery = cts.andNotQuery(cts.collectionQuery(getAllEntityIds()), cts.collectionQuery(getArchivedCollections()));
  const subjectResults = sem.sparql(`
        PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
        PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>
        PREFIX xsd: <http://www.w3.org/2001/XMLSchema#>
        SELECT * WHERE {
          {
            SELECT ?subjectIRI ?subjectIdentifier ?predicateIRI (MIN(?anyPredicateLabel) AS ?predicateLabel) ?objectIRI ?objectIdentifier ?isInverse ?origin WHERE {
                ?subjectIRI rdfs:isDefinedBy ?docURI.
                {
                    ?subjectIRI ?predicateIRI ?objectIRI.
                    BIND("false"^^xsd:boolean AS ?isInverse)
                } UNION {
                    ?objectIRI ?predicateIRI ?subjectIRI.
                    BIND("true"^^xsd:boolean AS ?isInverse)
                }
                OPTIONAL {
                    ?objectIRI rdfs:isDefinedBy ?objectDocURI.
                }
                OPTIONAL {
                    ?subjectIRI rdfs:isDefinedBy ?subjectDocURI.
                }
                BIND(IF(BOUND(?objectDocURI), ?objectDocURI, ?objectIRI) AS ?objectIdentifier)
                BIND(IF(BOUND(?subjectDocURI), ?subjectDocURI, ?subjectIRI) AS ?subjectIdentifier)
                BIND(IF(BOUND(?objectDocURI), "entity graph", "concept graph") AS ?origin)
                OPTIONAL {
                  ?predicateIRI $labelIRI ?anyPredicateLabel.
                }
                FILTER (isLiteral(?docURI) && ?docURI = $parentDocURI && ?predicateIRI = $allPredicates)
            }
            GROUP BY ?subjectIRI ?subjectIdentifier ?predicateIRI ?objectIRI ?objectIdentifier
          }
          OPTIONAL {
            ?subjectIRI rdf:type ?entityType.
            ?entityType <http://www.marklogic.com/data-hub#relatedConcept> ?conceptClassName.
            ?conceptClassName <http://www.marklogic.com/data-hub#conceptPredicate> ?predicateIRI.
          }
        }
    `, bindings, [], collectionQuery).toArray();
  if (graphDebugTraceEnabled) {
    xdmp.trace(graphTraceEvent, `Subject (${docURI}) results for node expand '${xdmp.describe(subjectResults, Sequence.from([]), Sequence.from([]))}'`);
  }
  return subjectResults;
}

function getNodeLabel(entityType, defaultLabel, objectUri, document) {
  let label = "";
  let configurationLabel = getLabelFromHubConfigByEntityType(entityType);
  if (document && configurationLabel.length > 0) {
    //getting the value of the configuration property
    label = fn.string(getValueFromProperty(configurationLabel, document, entityType));
  }

  if (label.length === 0) {
    if (fn.exists(objectUri)) {
      label = decodeURI(objectUri);
    } else {
      label = defaultLabel;
    }
  }
  return label;
}

function getNodeStyles(type, isConcept = false) {
  if (!hubCentralConfig) {
    return {};
  }
  const config = hubCentralConfig.toObject();
  if (isConcept) {
    const typesInfo = (config && config.modeling) ? config.modeling["concepts"] : {};
    let conceptStyle= {};
    if (typesInfo != null && typesInfo != undefined) {
      Object.keys(typesInfo).forEach(key => {
        if (typesInfo[key].hasOwnProperty("semanticConcepts")) {
          Object.keys(typesInfo[key]["semanticConcepts"]).forEach(sem => {
            if (sem === type) {
              conceptStyle = JSON.parse(JSON.stringify(typesInfo[key]));
              delete conceptStyle["semanticConcepts"];
            }
          });
        }
      });
    }
    return conceptStyle;
  }

  const typesInfo = (config && config.modeling) ? config.modeling["entities"] : {};
  return (typesInfo && typesInfo[type]) ? typesInfo[type]: {};
}

function cleanIriForLabel(iri) {
  const lastIndex = Math.max(iri.lastIndexOf("/"), iri.lastIndexOf("#"));
  return (lastIndex >= 0) ? iri.substring(lastIndex + 1): iri;
}

function graphResultsToNodesAndEdges(result) {
  const startTime = xdmp.elapsedTime();
  const nodesByID = {};
  const edgesByID = {};
  const distinctIdentifiers = new Set();
  const distinctIRIs = new Set();
  for (const item of result) {
    const subjectIdentifier = fn.string(item.subjectIdentifier);
    distinctIdentifiers.add(subjectIdentifier);
    distinctIRIs.add(fn.head(item.subjectIRI));
    const objectIRI = fn.head(item.objectIRI);
    if (objectIRI) {
      distinctIRIs.add(objectIRI);
    }
    const objectIdentifier = fn.string(item.objectIdentifier);
    if (objectIdentifier) {
      if (!objectIRI) {
        distinctIRIs.add(fn.head(item.objectIdentifier));
      }
      distinctIdentifiers.add(objectIdentifier);
    }
  }
  const hasRelationshipsMap = getHasRelationshipsMap([...distinctIRIs], [...distinctIdentifiers]);
  distinctIRIs.clear();
  for (const item of result) {
    const subjectIRI = fn.string(item.subjectIRI);
    const subjectIdentifier = fn.string(item.subjectIdentifier);
    const predicateIRI = fn.string(item.predicateIRI);
    const predicateLabel = fn.string(item.predicateLabel) || cleanIriForLabel(predicateIRI);
    const objectIRI = fn.string(item.objectIRI);
    const objectIdentifier = fn.string(item.objectIdentifier);
    [{iri: subjectIRI || subjectIdentifier, id: subjectIdentifier || subjectIRI}, {iri: objectIRI || objectIdentifier, id: objectIdentifier || objectIRI}].forEach((nodeInfo) => {
      if (nodeInfo.id && !nodesByID[nodeInfo.id]) {
        const id = nodeInfo.id;
        const lastIndexOfSeparator = Math.max(nodeInfo.iri.lastIndexOf("/"), nodeInfo.iri.lastIndexOf("#"));
        const hasSeparator = lastIndexOfSeparator >= 0;
        const defaultLabel = cleanIriForLabel(nodeInfo.id);
        const isObject = id === objectIdentifier;
        const isConcept = isObject && fn.string(item.origin) === "concept graph";
        const iriGroup = (hasSeparator && !isConcept) ? nodeInfo.iri.substring(0, lastIndexOfSeparator): nodeInfo.iri;
        const iriGroupParts = hasSeparator ? iriGroup.split("/"): [nodeInfo.iri];
        const entityType = iriGroupParts[iriGroupParts.length - 1];
        const styles = getNodeStyles(entityType, isConcept);
        // if the node is an entity, get all the IRIs associated with the document
        const hasRelationships = hasRelationshipsMap.has(id) ? hasRelationshipsMap.get(id): false;
        nodesByID[id] = {
          entityType,
          id,
          label: {text: defaultLabel},
          data: {
            entityType: isConcept ? undefined: entityType,
            iri: nodeInfo.iri,
            iriGroup: iriGroup,
            isConcept,
            conceptClassName: isConcept && isObject ? fn.string(item.conceptClassName): undefined,
            hasRelationships
          },
          fontIcon: styles.icon ? {text: styles.icon}: undefined,
          color: styles.color
        };
      }
    });
    if (objectIdentifier) {
      nodesByID[objectIdentifier].data.group = `${subjectIdentifier}:${predicateLabel}`;
      const sortedIdentifiers = [subjectIdentifier, objectIdentifier].sort();
      const isInverse = fn.head(item.isInverse);
      const from = isInverse ? objectIdentifier: subjectIdentifier;
      const to = isInverse ? subjectIdentifier: objectIdentifier;
      const edgeId = `${sortedIdentifiers[0]}:${predicateIRI}:${sortedIdentifiers[1]}`;
      if (!edgesByID[edgeId]) {
        edgesByID[edgeId] = {
          id1: from,
          id2: to,
          label: {text: predicateLabel}
        };
      }
    }
  }
  hasRelationshipsMap.clear();
  const count = distinctIdentifiers.size;
  const documents = cts.search(cts.andNotQuery(cts.documentQuery([...distinctIdentifiers]), cts.collectionQuery(getArchivedCollections())));
  distinctIdentifiers.clear();
  for (const document of documents) {
    const id = fn.string(xdmp.nodeUri(document));
    const node = nodesByID[id];
    node.data.propertiesOnHover = entityLib.getValuesPropertiesOnHover(document, node.entityType, hubCentralConfig);
    node.label.text = getNodeLabel(node.entityType, node.label.text, node.id, document);
    const unmergeDetails = entitySearchLib.fetchUnmergeDetails(document, node.entityType);
    node.data.unmerge = unmergeDetails["unmerge"];
    node.data.unmergeUris = unmergeDetails["unmergeUris"];
    node.data.unmergeDisabled = unmergeDetails["unmergeDisabled"];
    node.data.matchStepName = unmergeDetails["matchStepName"] ? unmergeDetails["matchStepName"] : undefined;
  }
  if (graphTraceEnabled) {
    xdmp.trace(graphTraceEvent, `graphResultsToNodesAndEdges completed in ${xdmp.elapsedTime().subtract(startTime)}`);
  }
  return {
    count,
    data: Object.assign({}, nodesByID, edgesByID)
  };
}

const labelsByEntityType = {};

function getLabelFromHubConfigByEntityType(entityType) {
  if (labelsByEntityType[entityType] === undefined) {
    const labelXPath = `/modeling/entities/${entityType}/label`;
    if (hubCentralConfig != null && cts.validExtractPath(labelXPath)) {
      labelsByEntityType[entityType] = fn.string(fn.head(hubCentralConfig.xpath(labelXPath)));
    } else {
      labelsByEntityType[entityType] = "";
    }
  }
  return labelsByEntityType[entityType];
}

function getValueFromProperty(propertyName, document, entityType) {
  if (fn.exists(document) && entityType && propertyName) {
    const property = document.xpath(`*:envelope/*:instance/*:${entityType}/*:${propertyName}`);
    if (fn.exists(property)) {
      return fn.data(fn.head(property));
    }
  }
  return "";
}

function getRelatedEntitiesCounting(allRelatedPredicateList, ctsQueryCustom) {
  if (allRelatedPredicateList.length === 0) {
    return 0;
  }
  /* TODO: Investigate why mapping bindings for predicates weren't giving accurate docUri count */
  return sem.sparql(`PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
  PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>
  SELECT (COUNT(?docUri) as ?total) WHERE {
      ?s ${allRelatedPredicateList.map(pred => `<${fn.string(pred)}>`).join("|")} ?o.
      ?o rdfs:isDefinedBy ?docUri.
  } `, {}, [], ctsQueryCustom);
}

function getEntityTypeIRIsCounting(entityTypeIRIs, ctsQueryCustom) {
  return sem.sparql(`PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>  PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>
  SELECT (COUNT(DISTINCT(?docUri)) AS ?total)  WHERE {
  ?subjectIRI rdf:type $entityTypeIRIs;
      rdfs:isDefinedBy ?docUri.
  } `, {entityTypeIRIs}, [], ctsQueryCustom);
}

function getConceptCounting(entityTypeIRIs, predicateConceptList, ctsQueryCustom) {
  const totalConcepts = sem.sparql(`PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
                   PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>
                   SELECT (COUNT(DISTINCT(?objectIRI)) AS ?total) WHERE {
                          ?subjectIRI rdf:type $entityTypeIRIs;
                          ?predicateIRI  ?objectIRI;
                          FILTER (isIRI(?predicateIRI) && ?predicateIRI = $predicateConceptList)
                          }`, {entityTypeIRIs, predicateConceptList}, [], ctsQueryCustom);
  return totalConcepts;
}

function getHasRelationshipsMap(allIris, allIds) {
  if (!allIds || allIds.length === 0) {
    return new Map();
  }
  const startTime = xdmp.elapsedTime();
  const predicates = getAllPredicates();
  const results = [];
  const batchSize = 500;
  for (let i = 0; i < allIds.length; i += batchSize) {
    const ids = allIds.slice(i, i + batchSize);
    results.push(
      ...sem.sparql(`
      PREFIX xsd: <http://www.w3.org/2001/XMLSchema#>
      PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>
      SELECT (xsd:string(?identifier) as ?id) (true as ?hasRelationships)  {
        ?iri rdfs:isDefinedBy ?identifier.
        # Match IRIs in the batch and either $predicates or is a concept (doesn't have an id)
        FILTER (?identifier = $ids)
        {
          ?iri ?predicate ?linkedIRI.
        } UNION {
          ?linkedIRI ?predicate ?iri.
        }
        FILTER (?predicate = $predicates)
        # Minus the triples that have IRIs already in the search results
        MINUS {
          {
            ?iri ?predicate ?linkedIRI.
          } UNION {
            ?linkedIRI ?predicate ?iri.
          }
          FILTER (isIRI(?linkedIRI) && ?linkedIRI = $allIris)
        }
        # Either $predicates match or is a concept (doesn't have an id)
        OPTIONAL {
          ?linkedIRI rdfs:isDefinedBy ?linkedIdentifier.
          BIND(xsd:string(?linkedIdentifier) AS ?linkedId)
        }
      }
      GROUP BY ?identifier
    `, {predicates, ids, allIris, allIds}, ["array"], sem.store(["document"]))
    );
  }
  // We also need to check for IRIs that are not defined by any document (i.e., concepts)
  for (let i = 0; i < allIris.length; i += batchSize) {
    const iris = allIris.slice(i, i + batchSize);
    results.push(
      ...sem.sparql(`
      PREFIX xsd: <http://www.w3.org/2001/XMLSchema#>
      PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>
      SELECT (xsd:string(?iri) as ?id) (true as ?hasRelationships)  {
        {
          ?iri ?predicate ?linkedIRI.
        } UNION {
          ?linkedIRI ?predicate ?iri.
        }
        # Match IRIs in the batch
        FILTER (?iri = $iris && !(?linkedIRI = $allIris))
        FILTER NOT EXISTS {
          ?iri rdfs:isDefinedBy ?identifier.
        }
        OPTIONAL {
          ?linkedIRI rdfs:isDefinedBy ?linkedIdentifier.
        }
        FILTER (!(BOUND(?linkedIdentifier) && ?linkedIdentifier = $allIds))
      }
      GROUP BY ?iri
    `, {iris, allIris, allIds}, ["array"], sem.store(["document"]))
    );
  }
  if (graphTraceEnabled) {
    xdmp.trace(graphTraceEvent, `getHasRelationshipsMap returns ${results.length} in ${xdmp.elapsedTime().subtract(startTime)}`);
  }
  return new Map(results);
}

let predicatesMap = null;

function getPredicatesMap() {
  if (!predicatesMap) {
    predicatesMap = new Map();
    for (const modelNode of fn.collection(entityLib.getModelCollection())) {
      const model = modelNode.toObject();
      const entityName = model.info.title;
      predicatesMap.set(entityName, entityLib.getPredicatesByModel(model, true));
    }
  }
  return predicatesMap;
}

function getAllPredicates() {
  const predicatesMap = getPredicatesMap();
  let allPredicates = [];
  predicatesMap.forEach(val => {
    allPredicates = allPredicates.concat(val);
  });
  return allPredicates;
}


function getEntityWithConcepts(entityTypeIRIs, predicateConceptList) {
  const subjectPlanConcept = sem.sparql(`PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
                   PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>
                   PREFIX mlDH: <http://www.marklogic.com/data-hub#>
                   PREFIX es: <http://marklogic.com/entity-services#>

                   SELECT DISTINCT ?objectIRI ?conceptClassName ?entityTypeIRI WHERE {
                          ?entityTypeIRI rdf:type es:EntityType;
                                mlDH:relatedConcept ?conceptClassName.
                          ?conceptClassName mlDH:conceptPredicate ?conceptPredicate.
                          ?subjectIRI ?conceptPredicate ?objectIRI.
                          FILTER (isIRI(?entityTypeIRI) && ?entityTypeIRI = $entityTypeIRIs && isIRI(?conceptPredicate) && ?conceptPredicate = $predicateConceptList)
                   }`, {entityTypeIRIs, predicateConceptList});

  return subjectPlanConcept.toArray();
}

function getRelatedEntityInstancesCount(semanticConceptIRI, ctsQuery) {
  const relatedEntityInstancesCount = sem.sparql(`PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
      PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>
      SELECT (COUNT(DISTINCT(?subjectIRI)) AS ?total) ?entityTypeIRI  WHERE {
          ?entityTypeIRI rdf:type <http://marklogic.com/entity-services#EntityType>.
          ?subjectIRI ?p $semanticConceptIRI;
              rdf:type ?entityTypeIRI.
      }
      GROUP BY ?entityTypeIRI`, {semanticConceptIRI}, [], ctsQuery ? cts.orQuery([ctsQuery, cts.collectionQuery(consts.ENTITY_MODEL_COLLECTION)]): null
  );
  return relatedEntityInstancesCount.toObject();
}

function describeIRI(semanticConceptIRI) {
  const description = {};
  const describeTriples = sem.sparql(`DESCRIBE @semanticConceptIRI`, {semanticConceptIRI});
  for (const triple of describeTriples) {
    description[fn.string(sem.triplePredicate(triple))] = fn.string(sem.tripleObject(triple));
  }
  return description;
}

function getArchivedCollections() {
  return cts.valueMatch(cts.collectionReference(), "sm-*-archived");
}

const returnFlags = `<return-aggregates xmlns="http://marklogic.com/appservices/search">false</return-aggregates>
  <return-constraints xmlns="http://marklogic.com/appservices/search">false</return-constraints>
  <return-facets xmlns="http://marklogic.com/appservices/search">false</return-facets>
  <return-frequencies xmlns="http://marklogic.com/appservices/search">false</return-frequencies>
  <return-metrics xmlns="http://marklogic.com/appservices/search">false</return-metrics>
  <return-plan xmlns="http://marklogic.com/appservices/search">false</return-plan>
  <return-qtext xmlns="http://marklogic.com/appservices/search">false</return-qtext>
  <return-results xmlns="http://marklogic.com/appservices/search">false</return-results>
  <return-similar xmlns="http://marklogic.com/appservices/search">false</return-similar>
  <return-values xmlns="http://marklogic.com/appservices/search">false</return-values>
  <return-query xmlns="http://marklogic.com/appservices/search">true</return-query>`;

const stylesheet = fn.head(xdmp.unquote(`<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform" version="1.0">
   <xsl:template match="node()|@*">
      <xsl:copy>
         <xsl:apply-templates select="node()|@*" />
      </xsl:copy>
   </xsl:template>
   <xsl:template match="*:return-aggregates|*:return-constraints|*:return-facets|*:return-frequencies|*:return-metrics|*:return-plan|*:return-qtext|*:return-results|*:return-similar|*:return-values|*:return-query" />
   <xsl:template match="*:options">
      <xsl:copy>
         <xsl:apply-templates select="node()|@*" />
         ${returnFlags}
      </xsl:copy>
   </xsl:template>
</xsl:stylesheet>`));

function buildCtsQuery(structuredQuery, searchText, queryOptions) {
  const search = require('/MarkLogic/appservices/search/search.xqy');
  queryOptions = queryOptions ? fn.head(xdmp.unquote(queryOptions)).root: null;
  const cleanOptions = queryOptions ? fn.head(xdmp.xsltEval(stylesheet, queryOptions)).root: null;
  let qrySearch;
  if (structuredQuery !== undefined && structuredQuery.toString().length > 0) {
    structuredQuery = fn.head(xdmp.unquote(structuredQuery)).root;
    const searchResponse = fn.head(search.resolve(structuredQuery, cleanOptions));
    qrySearch = cts.query(searchResponse.xpath('./*/*'));
  }
  let ctsQuery = cts.trueQuery();
  if (searchText !== undefined && searchText.toString().length > 0) {
    const searchTxtResponse = fn.head(search.parse(searchText, cleanOptions));
    ctsQuery = cts.query(searchTxtResponse);
    if (qrySearch !== undefined) {
      ctsQuery = cts.andQuery([qrySearch, ctsQuery]);
    }
  } else {
    // if doesn't has search text, but could has facetSelects
    if (qrySearch !== undefined) {
      ctsQuery = qrySearch;
    }
  }
  return ctsQuery;
}

export default {
  describeIRI,
  getAllEntityIds,
  getAllPredicates,
  getEntityNodesWithRelated,
  getEntityNodesByDocument,
  getEntityNodesByConcept,
  getEntityTypeIRIsCounting,
  getEntityNodesExpandingConcept,
  getRelatedEntitiesCounting,
  getConceptCounting,
  getRelatedEntityInstancesCount,
  getEntityWithConcepts,
  graphResultsToNodesAndEdges,
  buildCtsQuery
};
