package com.mapbox.android.accounts.navigation.sku.v1;

import android.annotation.SuppressLint;
import android.content.SharedPreferences;
import android.text.TextUtils;

import androidx.annotation.Keep;
import androidx.annotation.NonNull;

import com.mapbox.android.accounts.v1.MapboxAccounts;

import static com.mapbox.android.accounts.v1.AccountsConstants.KEY_PREFERENCE_MAPS_SKU_TOKEN;

/**
 * The class is responsible to request @link[MapboxAccounts.java] to generate the MAU token
 * and rotate the SKU token based on the refresh time
 */
public class MauSku implements SkuGenerator {

    private static final String KEY_PREFERENCE_NAV_MAU_SKU = "com.mapbox.navigation.accounts.mau.skutoken";
    private static final String KEY_PREFERENCE_NAV_USER_ID = "com.mapbox.navigation.accounts.mau.userid";
    private static final String KEY_PREFERENCE_NAV_MAU_TIMESTAMP = "com.mapbox.navigation.accounts.mau.timestamp";
    private static final long DEFAULT_TOKEN_TIMER = 0L;

    private SharedPreferences preferences;
    private long timerExpireAfter;
    private String applicationId;

    @Keep
    public MauSku(
            @NonNull SharedPreferences preferences,
            long timerExpireAfter,
            @NonNull String applicationId
    ) {
        this.preferences = preferences;
        this.applicationId = applicationId;
        this.timerExpireAfter = timerExpireAfter;
    }

    /**
     * Generates a SKU token. It also sets the route request count to default and timer to current time
     */
    @Keep
    @Override
    public void initializeSKU() {
        setTimerExpiry(getNow());
        persistMauSkuToken();
    }

    /**
     * Generates the SKU token
     * @return the SKU token
     */
    @Keep
    @Override
    public String generateToken() {
        refreshSkuToken();
        return retrieveNavMauSkuToken();
    }

    @Keep
    @Override
    public void onNavigationStart() {
    }

    /**
     * Sets the last time a request was made to DirectionsAPI to the default value
     */
    @Keep
    @Override
    public void onNavigationEnd() {
        setTimerExpiry(DEFAULT_TOKEN_TIMER);
    }

    private void refreshSkuToken() {
        if (!shouldRefreshSku()) {
            return;
        }
        setTimerExpiry(getNow());
        persistMauSkuToken();
    }

    private boolean shouldRefreshSku() {
        return validateTimerExpiry();
    }

    private boolean validateTimerExpiry() {
        long skuTokenTimeStamp = getTimerExpiry();
        return isOneHoursExpired(skuTokenTimeStamp);
    }

    @SuppressLint("ApplySharedPref")
    private void setTimerExpiry(long then) {
        preferences.edit().putLong(applicationId + "." + KEY_PREFERENCE_NAV_MAU_TIMESTAMP, then).apply();
    }

    private long getTimerExpiry() {
        return preferences.getLong(applicationId + "." + KEY_PREFERENCE_NAV_MAU_TIMESTAMP, DEFAULT_TOKEN_TIMER);
    }

    @SuppressLint("ApplySharedPref")
    private void persistMauSkuToken() {
        String token = generateNavMauSkuToken();
        preferences.edit().putString(applicationId + "." + KEY_PREFERENCE_NAV_MAU_SKU, token).apply();
        preferences.edit().putString(KEY_PREFERENCE_MAPS_SKU_TOKEN, token).apply();
    }

    private String retrieveNavMauSkuToken() {
        return preferences.getString(applicationId + "." + KEY_PREFERENCE_NAV_MAU_SKU, "");
    }

    @SuppressLint("ApplySharedPref")
    private void persistUserId(String userId) {
        preferences.edit().putString(applicationId + "." + KEY_PREFERENCE_NAV_USER_ID, userId).apply();
    }

    private String retrieveUserId() {
        return preferences.getString(applicationId + "." + KEY_PREFERENCE_NAV_USER_ID, "");
    }

    private String getUserId() {
        String userId = retrieveUserId();
        if (TextUtils.isEmpty(userId)) {
            userId = generateUserId();
            persistUserId(userId);
        }
        return userId;
    }

    private String generateUserId() {
        return MapboxAccounts.obtainEndUserId();
    }

    private String generateNavMauSkuToken() {
        return MapboxAccounts.obtainNavigationSkuUserToken(getUserId());
    }

    private boolean isOneHoursExpired(long then) {
        return isExpired(getNow(), then);
    }

    private long getNow() {
        return System.currentTimeMillis();
    }

    private boolean isExpired(long now, long then) {
        return (now - then) / 1000 > timerExpireAfter;
    }
}
