/*
 * Decompiled with CFR 0.152.
 */
package com.mapbox.services.android.telemetry;

import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.SharedPreferences;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.ServiceInfo;
import android.location.Location;
import android.os.Build;
import android.os.Handler;
import android.os.Parcelable;
import android.support.annotation.IntRange;
import android.support.annotation.NonNull;
import android.support.v4.content.LocalBroadcastManager;
import android.text.TextUtils;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.WindowManager;
import com.mapbox.services.android.core.location.LocationEngine;
import com.mapbox.services.android.core.location.LocationEngineListener;
import com.mapbox.services.android.core.location.LocationEngineProvider;
import com.mapbox.services.android.telemetry.Experimental;
import com.mapbox.services.android.telemetry.TelemetryException;
import com.mapbox.services.android.telemetry.TelemetryListener;
import com.mapbox.services.android.telemetry.backoff.ExponentialBackoff;
import com.mapbox.services.android.telemetry.connectivity.ConnectivityReceiver;
import com.mapbox.services.android.telemetry.constants.TelemetryConstants;
import com.mapbox.services.android.telemetry.http.TelemetryClient;
import com.mapbox.services.android.telemetry.permissions.PermissionsManager;
import com.mapbox.services.android.telemetry.service.TelemetryService;
import com.mapbox.services.android.telemetry.utils.TelemetryUtils;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.Vector;
import java.util.concurrent.CopyOnWriteArrayList;
import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Response;
import okhttp3.internal.Util;
import org.json.JSONObject;

public class MapboxTelemetry
implements Callback,
LocationEngineListener {
    private static final String LOG_TAG = "MapboxTelemetry";
    private static final int UNAVAILABLE_BATTERY_LEVEL = 100;
    private static MapboxTelemetry instance;
    private boolean initialized = false;
    private Context context = null;
    private String accessToken = null;
    private String userAgent = null;
    private String mapboxSessionId = null;
    private long mapboxSessionIdLastSet = 0L;
    private String mapboxVendorId = null;
    private DisplayMetrics displayMetrics = null;
    private Intent batteryStatus = null;
    private TelemetryClient client = null;
    private Vector<Hashtable<String, Object>> events = new Vector();
    private Timer timer = null;
    private LocationEngine locationEngine = null;
    private boolean withShutDown = false;
    private Boolean telemetryEnabled = null;
    protected CopyOnWriteArrayList<TelemetryListener> telemetryListeners;
    private Hashtable<String, Object> customTurnstileEvent = null;
    private int sessionIdRotationTime = 24;
    private boolean debugLoggingEnabled = false;
    private String sdkIdentifier = "";
    private String sdkVersion = "";
    private static final List<String> VALID_USER_AGENTS;

    private MapboxTelemetry() {
    }

    public static synchronized MapboxTelemetry getInstance() {
        if (instance == null) {
            instance = new MapboxTelemetry();
        }
        return instance;
    }

    public void initialize(@NonNull Context context, @NonNull String accessToken, @NonNull String userAgent, @NonNull LocationEngine locationEngine) {
        this.locationEngine = locationEngine;
        this.initialize(context, accessToken, userAgent);
    }

    public void initialize(@NonNull Context context, @NonNull String accessToken, @NonNull String userAgent, @NonNull String sdkIdentifier, @NonNull String sdkVersion) {
        this.sdkIdentifier = sdkIdentifier;
        this.sdkVersion = sdkVersion;
        this.initialize(context, accessToken, userAgent);
    }

    public void initialize(@NonNull Context context, @NonNull String accessToken, @NonNull String userAgent) {
        if (this.initialized) {
            return;
        }
        Log.v((String)LOG_TAG, (String)"Initializing telemetry.");
        this.context = context.getApplicationContext();
        this.accessToken = accessToken;
        this.userAgent = userAgent;
        if (this.context == null || TextUtils.isEmpty((CharSequence)this.accessToken) || TextUtils.isEmpty((CharSequence)this.userAgent) || this.sdkIdentifier == null || this.sdkVersion == null) {
            throw new TelemetryException("Please, make sure you provide a valid context, access token, user agent, sdkIdentifier and sdkVersion. For more information, please visit https://www.mapbox.com/android-sdk.");
        }
        this.telemetryListeners = new CopyOnWriteArrayList();
        this.validateTelemetryServiceConfigured();
        this.setupHttpClient();
        this.checkStagingServerInformation();
        this.setUserAgent();
        this.rotateSessionId();
        this.readDisplayMetrics();
        this.registerBatteryUpdates();
        this.loadUserPreferences();
        this.initialized = true;
    }

    public void addTelemetryListener(TelemetryListener listener) {
        if (!this.telemetryListeners.contains(listener)) {
            this.telemetryListeners.add(listener);
        }
    }

    public boolean removeTelemetryListener(TelemetryListener listener) {
        return this.telemetryListeners.remove(listener);
    }

    public Hashtable<String, Object> getCustomTurnstileEvent() {
        return this.customTurnstileEvent;
    }

    public void setCustomTurnstileEvent(Hashtable<String, Object> customTurnstileEvent) {
        this.customTurnstileEvent = customTurnstileEvent;
    }

    @Experimental
    public void setSessionIdRotationTime(@IntRange(from=1L, to=24L) int sessionIdRotationTime) {
        this.sessionIdRotationTime = sessionIdRotationTime;
    }

    @Experimental
    public boolean isDebugLoggingEnabled() {
        return this.debugLoggingEnabled;
    }

    @Experimental
    public void setDebugLoggingEnabled(boolean debugLoggingEnabled) {
        this.debugLoggingEnabled = debugLoggingEnabled;
        if (this.client != null) {
            this.client.setDebugLoggingEnabled(debugLoggingEnabled);
        }
    }

    @Experimental
    public void newUserAgent(String userAgent) {
        if (this.isUserAgentValid(userAgent)) {
            this.userAgent = userAgent;
            this.setUserAgent();
        }
    }

    private void validateTelemetryServiceConfigured() {
        try {
            PackageInfo packageInfo = this.context.getPackageManager().getPackageInfo(this.context.getPackageName(), 4);
            if (packageInfo.services != null) {
                for (ServiceInfo service : packageInfo.services) {
                    if (!TextUtils.equals((CharSequence)"com.mapbox.services.android.telemetry.service.TelemetryService", (CharSequence)service.name)) continue;
                    return;
                }
            }
        }
        catch (Exception exception) {
            throw new TelemetryException(String.format(TelemetryConstants.DEFAULT_LOCALE, "Failed to find the Telemetry service in your `AndroidManifest.xml` file (%s). For more information, please visit https://www.mapbox.com/android-sdk.", exception.getMessage()));
        }
        throw new TelemetryException("Please, make sure you add the Telemetry service (`com.mapbox.services.android.telemetry.service.TelemetryService`) to your `AndroidManifest.xml` file. For more information, please visit https://www.mapbox.com/android-sdk.");
    }

    private void setupHttpClient() {
        this.client = new TelemetryClient(this.accessToken);
    }

    private void checkStagingServerInformation() {
        try {
            String stagingURL = null;
            String stagingAccessToken = null;
            boolean cnServer = false;
            ApplicationInfo appInfo = this.context.getPackageManager().getApplicationInfo(this.context.getPackageName(), 128);
            if (appInfo != null && appInfo.metaData != null) {
                stagingURL = appInfo.metaData.getString("com.mapbox.TestEventsServer");
                stagingAccessToken = appInfo.metaData.getString("com.mapbox.TestEventsAccessToken");
                cnServer = appInfo.metaData.getBoolean("com.mapbox.CnEventsServer");
            }
            if (cnServer) {
                this.client.setEnableCnEndpoint();
            } else {
                if (TextUtils.isEmpty(stagingURL) || TextUtils.isEmpty(stagingAccessToken)) {
                    SharedPreferences prefs = TelemetryUtils.getSharedPreferences(this.context);
                    stagingURL = prefs.getString("mapboxTelemetryStagingUrl", null);
                    stagingAccessToken = prefs.getString("mapboxTelemetryStagingAccessToken", null);
                }
                if (!TextUtils.isEmpty((CharSequence)stagingURL) && !TextUtils.isEmpty((CharSequence)stagingAccessToken)) {
                    Log.w((String)LOG_TAG, (String)String.format("Using staging server '%s' with access token '%s'.", stagingURL, stagingAccessToken));
                    this.client.setEventsEndpoint(stagingURL);
                    this.client.setAccessToken(stagingAccessToken);
                    this.client.setStagingEnvironment(true);
                }
            }
        }
        catch (Exception exception) {
            Log.e((String)LOG_TAG, (String)String.format("Failed to check for staging credentials: %s", exception.getMessage()));
        }
    }

    private void setUserAgent() {
        String appIdentifier = TelemetryUtils.getApplicationIdentifier(this.context);
        String fullUserAgent = TextUtils.isEmpty((CharSequence)appIdentifier) ? this.userAgent : Util.toHumanReadableAscii((String)String.format(TelemetryConstants.DEFAULT_LOCALE, "%s %s", appIdentifier, this.userAgent));
        this.client.setUserAgent(fullUserAgent);
    }

    private boolean isUserAgentValid(String userAgent) {
        if (!TextUtils.isEmpty((CharSequence)userAgent)) {
            for (String userAgentPrefix : VALID_USER_AGENTS) {
                if (!userAgent.startsWith(userAgentPrefix)) continue;
                return true;
            }
        }
        return false;
    }

    private void rotateSessionId() {
        long timeSinceLastSet = System.currentTimeMillis() - this.mapboxSessionIdLastSet;
        if (TextUtils.isEmpty((CharSequence)this.mapboxSessionId) || timeSinceLastSet > (long)(this.sessionIdRotationTime * 3600000)) {
            this.mapboxSessionId = TelemetryUtils.buildUUID();
            this.mapboxSessionIdLastSet = System.currentTimeMillis();
        }
    }

    private void readDisplayMetrics() {
        this.displayMetrics = new DisplayMetrics();
        ((WindowManager)this.context.getSystemService("window")).getDefaultDisplay().getMetrics(this.displayMetrics);
    }

    private void registerBatteryUpdates() {
        IntentFilter filter = new IntentFilter("android.intent.action.BATTERY_CHANGED");
        this.batteryStatus = this.context.registerReceiver(null, filter);
    }

    private int getBatteryLevel() {
        if (this.batteryStatus != null) {
            int level = this.batteryStatus.getIntExtra("level", -1);
            int scale = this.batteryStatus.getIntExtra("scale", -1);
            return Math.round((float)level / (float)scale * 100.0f);
        }
        return 100;
    }

    private boolean isPluggedIn() {
        int chargePlug;
        return this.batteryStatus != null && ((chargePlug = this.batteryStatus.getIntExtra("plugged", -1)) == 2 || chargePlug == 1);
    }

    private void loadUserPreferences() {
        SharedPreferences prefs = TelemetryUtils.getSharedPreferences(this.context);
        if (prefs.contains("mapboxVendorId")) {
            this.mapboxVendorId = prefs.getString("mapboxVendorId", "");
        }
        if (TextUtils.isEmpty((CharSequence)this.mapboxVendorId)) {
            this.mapboxVendorId = TelemetryUtils.buildUUID();
            SharedPreferences.Editor editor = prefs.edit();
            editor.putString("mapboxVendorId", this.mapboxVendorId);
            editor.apply();
        }
        this.setTelemetryEnabled(this.isTelemetryEnabled());
    }

    public boolean isTelemetryEnabled() {
        if (this.telemetryEnabled == null) {
            this.telemetryEnabled = TelemetryUtils.getSharedPreferences(this.context).getBoolean("mapboxTelemetryEnabled", true);
        }
        return this.telemetryEnabled;
    }

    public void setTelemetryEnabled(boolean telemetryEnabled) {
        if (this.initialized && this.isTelemetryEnabled() == telemetryEnabled) {
            Log.v((String)LOG_TAG, (String)String.format("Telemetry was already initialized on that state (enabled: %b).", telemetryEnabled));
            return;
        }
        if (telemetryEnabled) {
            Log.v((String)LOG_TAG, (String)"Enabling telemetry.");
            this.context.startService(new Intent(this.context, TelemetryService.class));
            if (PermissionsManager.areLocationPermissionsGranted(this.context)) {
                this.registerLocationUpdates();
            } else {
                this.startPermissionsTimer();
            }
            this.timer = new Timer();
            this.timer.schedule(new TimerTask(){

                @Override
                public void run() {
                    MapboxTelemetry.this.flushEventsQueueImmediately(false);
                }
            }, 10000L, 180000L);
        } else {
            Log.v((String)LOG_TAG, (String)"Disabling telemetry.");
            this.withShutDown = true;
            this.pushTurnstileEvent();
        }
        this.telemetryEnabled = telemetryEnabled;
        SharedPreferences prefs = TelemetryUtils.getSharedPreferences(this.context);
        SharedPreferences.Editor editor = prefs.edit();
        editor.putBoolean("mapboxTelemetryEnabled", telemetryEnabled);
        editor.apply();
    }

    private void startPermissionsTimer() {
        final Handler handler = new Handler();
        final ExponentialBackoff counter = new ExponentialBackoff();
        Runnable runnable = new Runnable(){

            @Override
            public void run() {
                if (PermissionsManager.areLocationPermissionsGranted(MapboxTelemetry.this.context)) {
                    MapboxTelemetry.this.registerLocationUpdates();
                } else {
                    long nextWaitTime = counter.nextBackOffMillis();
                    handler.postDelayed((Runnable)this, nextWaitTime);
                }
            }
        };
        long nextWaitTime = counter.nextBackOffMillis();
        handler.postDelayed(runnable, nextWaitTime);
    }

    private void registerLocationUpdates() {
        LocationEngineProvider locationEngineProvider = new LocationEngineProvider(this.context);
        this.locationEngine = locationEngineProvider.obtainBestLocationEngineAvailable();
        this.locationEngine.addLocationEngineListener((LocationEngineListener)this);
        this.locationEngine.activate();
    }

    public void onConnected() {
        this.locationEngine.requestLocationUpdates();
    }

    public void onLocationChanged(Location location) {
        Intent intent = new Intent("com.mapbox.services.android.telemetry.location.TelemetryLocationReceiver");
        intent.putExtra("location", (Parcelable)location);
        LocalBroadcastManager.getInstance((Context)this.context.getApplicationContext()).sendBroadcast(intent);
    }

    private boolean isInitializedAndEnabled() {
        return this.initialized && this.isTelemetryEnabled();
    }

    private void putEventOnQueue(@NonNull Hashtable<String, Object> event) {
        this.events.add(event);
        if (this.events.size() >= 180) {
            this.flushEventsQueueImmediately(false);
        }
    }

    protected void addLocationEvent(Location location) {
        if (!this.isInitializedAndEnabled()) {
            return;
        }
        if (Double.isNaN(location.getLatitude()) || Double.isNaN(location.getLongitude()) || Double.isNaN(location.getAltitude()) || Float.isNaN(location.getAccuracy())) {
            return;
        }
        if (Double.isInfinite(location.getLatitude()) || Double.isInfinite(location.getLongitude()) || Double.isInfinite(location.getAltitude()) || Float.isInfinite(location.getAccuracy())) {
            return;
        }
        double latitudeScaled = new BigDecimal(location.getLatitude()).setScale(7, 1).doubleValue();
        double longitudeScaled = new BigDecimal(location.getLongitude()).setScale(7, 1).doubleValue();
        Hashtable<String, Object> event = new Hashtable<String, Object>();
        event.put("event", "location");
        event.put("created", TelemetryUtils.generateCreateDate(location));
        event.put("source", "mapbox");
        event.put("sessionId", this.mapboxSessionId);
        event.put("lat", latitudeScaled);
        event.put("lng", longitudeScaled);
        event.put("altitude", Math.round(location.getAltitude()));
        event.put("horizontalAccuracy", Math.round(location.getAccuracy()));
        event.put("operatingSystem", TelemetryConstants.OPERATING_SYSTEM);
        event.put("applicationState", TelemetryUtils.getApplicationState(this.context));
        this.putEventOnQueue(event);
        this.rotateSessionId();
    }

    public void pushEvent(Hashtable<String, Object> eventWithAttributes) {
        if (!this.isInitializedAndEnabled()) {
            return;
        }
        if (eventWithAttributes == null) {
            return;
        }
        String eventType = (String)eventWithAttributes.get("event");
        if (TextUtils.isEmpty((CharSequence)eventType)) {
            return;
        }
        if (eventType.equalsIgnoreCase("map.load")) {
            eventWithAttributes.put("userId", this.mapboxVendorId);
            eventWithAttributes.put("model", Build.MODEL);
            eventWithAttributes.put("operatingSystem", TelemetryConstants.OPERATING_SYSTEM);
            eventWithAttributes.put("resolution", Float.valueOf(this.displayMetrics.density));
            eventWithAttributes.put("accessibilityFontScale", Float.valueOf(TelemetryUtils.getAccesibilityFontScaleSize(this.context)));
            eventWithAttributes.put("orientation", TelemetryUtils.getOrientation(this.context));
            eventWithAttributes.put("batteryLevel", this.getBatteryLevel());
            eventWithAttributes.put("pluggedIn", this.isPluggedIn());
            eventWithAttributes.put("carrier", TelemetryUtils.getCellularCarrier(this.context));
            eventWithAttributes.put("cellularNetworkType", TelemetryUtils.getCellularNetworkType(this.context));
            eventWithAttributes.put("wifi", TelemetryUtils.getConnectedToWifi(this.context));
            this.putEventOnQueue(eventWithAttributes);
            this.pushTurnstileEvent();
        } else if (eventType.equalsIgnoreCase("map.click")) {
            eventWithAttributes.put("orientation", TelemetryUtils.getOrientation(this.context));
            eventWithAttributes.put("batteryLevel", this.getBatteryLevel());
            eventWithAttributes.put("pluggedIn", this.isPluggedIn());
            eventWithAttributes.put("carrier", TelemetryUtils.getCellularCarrier(this.context));
            eventWithAttributes.put("cellularNetworkType", TelemetryUtils.getCellularNetworkType(this.context));
            eventWithAttributes.put("wifi", TelemetryUtils.getConnectedToWifi(this.context));
            this.putEventOnQueue(eventWithAttributes);
        } else if (eventType.equalsIgnoreCase("map.dragend")) {
            eventWithAttributes.put("orientation", TelemetryUtils.getOrientation(this.context));
            eventWithAttributes.put("batteryLevel", this.getBatteryLevel());
            eventWithAttributes.put("pluggedIn", this.isPluggedIn());
            eventWithAttributes.put("carrier", TelemetryUtils.getCellularCarrier(this.context));
            eventWithAttributes.put("cellularNetworkType", TelemetryUtils.getCellularNetworkType(this.context));
            eventWithAttributes.put("wifi", TelemetryUtils.getConnectedToWifi(this.context));
            this.putEventOnQueue(eventWithAttributes);
        } else if (this.isANavigationEvent(eventType)) {
            this.addGeneralNavigationMetadataTo(eventWithAttributes);
            this.putEventOnQueue(eventWithAttributes);
        } else {
            Log.w((String)LOG_TAG, (String)String.format("Unknown event type provided: %s.", eventType));
        }
    }

    private boolean isANavigationEvent(String eventType) {
        boolean isDepart = eventType.equalsIgnoreCase("navigation.depart");
        boolean isFeedback = eventType.equalsIgnoreCase("navigation.feedback");
        boolean isArrived = eventType.equalsIgnoreCase("navigation.arrive");
        boolean isCanceled = eventType.equalsIgnoreCase("navigation.cancel");
        boolean isReroute = eventType.equalsIgnoreCase("navigation.reroute");
        boolean isANavigationEvent = isDepart || isFeedback || isArrived || isCanceled || isReroute;
        return isANavigationEvent;
    }

    private void addGeneralNavigationMetadataTo(Hashtable<String, Object> eventWithAttributes) {
        eventWithAttributes.put("device", Build.MODEL);
        eventWithAttributes.put("volumeLevel", TelemetryUtils.getVolumeLevel(this.context));
        eventWithAttributes.put("audioType", TelemetryUtils.obtainAudioType(this.context));
        eventWithAttributes.put("screenBrightness", TelemetryUtils.getScreenBrightness(this.context));
        eventWithAttributes.put("applicationState", TelemetryUtils.getApplicationState(this.context));
        eventWithAttributes.put("batteryPluggedIn", this.isPluggedIn());
        eventWithAttributes.put("batteryLevel", this.getBatteryLevel());
        eventWithAttributes.put("connectivity", TelemetryUtils.getCellularNetworkType(this.context));
    }

    public void flushEventsQueueImmediately(boolean hasTurnstileEvent) {
        if (this.initialized && this.events.size() > 0) {
            if (ConnectivityReceiver.isConnected(this.context)) {
                if (hasTurnstileEvent || this.isTelemetryEnabled()) {
                    this.client.sendEvents(this.events, this);
                    for (TelemetryListener listener : this.telemetryListeners) {
                        listener.onSendEvents(this.events.size());
                    }
                } else {
                    this.events.removeAllElements();
                }
            } else {
                this.events.removeAllElements();
            }
        }
        if (this.withShutDown) {
            this.shutdownTelemetry();
        }
    }

    private void pushTurnstileEvent() {
        Hashtable<String, Object> event = this.getCustomTurnstileEvent();
        if (event == null) {
            event = new Hashtable();
            event.put("event", "appUserTurnstile");
        }
        event.put("created", TelemetryUtils.generateCreateDate(null));
        event.put("userId", this.mapboxVendorId);
        event.put("enabled.telemetry", this.isTelemetryEnabled());
        event.put("sdkIdentifier", this.addSdkIdentifier());
        event.put("sdkVersion", this.addSdkVersion());
        this.events.add(event);
        this.flushEventsQueueImmediately(true);
    }

    public void onFailure(Call call, IOException e) {
        for (TelemetryListener listener : this.telemetryListeners) {
            listener.onHttpFailure(e.getMessage());
        }
        this.events.removeAllElements();
        if (this.withShutDown) {
            this.shutdownTelemetry();
        }
    }

    public void onResponse(Call call, Response response) throws IOException {
        for (TelemetryListener listener : this.telemetryListeners) {
            listener.onHttpResponse(response.isSuccessful(), response.code());
        }
        this.events.removeAllElements();
        if (response != null && response.body() != null) {
            response.body().close();
        }
        if (this.withShutDown) {
            this.shutdownTelemetry();
        }
    }

    private void shutdownTelemetry() {
        Log.d((String)LOG_TAG, (String)"Shutting down telemetry service.");
        this.withShutDown = false;
        this.events.removeAllElements();
        this.context.stopService(new Intent(this.context, TelemetryService.class));
        if (this.locationEngine == null) {
            Log.e((String)LOG_TAG, (String)String.format("Shutdown error: Location Engine instance wasn't set up (initialized: %b).", this.initialized));
        } else {
            this.locationEngine.removeLocationEngineListener((LocationEngineListener)this);
            this.locationEngine.removeLocationUpdates();
        }
        if (this.timer != null) {
            this.timer.cancel();
            this.timer = null;
        }
    }

    @Experimental
    public void setAccessToken(@NonNull String accessToken) {
        if (this.client == null || TextUtils.isEmpty((CharSequence)accessToken)) {
            throw new TelemetryException("Please, make sure you have initialized MapboxTelemetry before resetting the access token and it's a valid one. For more information, please visit https://www.mapbox.com/android-sdk.");
        }
        this.accessToken = accessToken;
        this.client.setAccessToken(accessToken);
    }

    private Object addSdkIdentifier() {
        if (this.sdkIdentifier.isEmpty()) {
            return JSONObject.NULL;
        }
        return this.sdkIdentifier;
    }

    private Object addSdkVersion() {
        if (this.sdkVersion.isEmpty()) {
            return JSONObject.NULL;
        }
        return this.sdkVersion;
    }

    @Experimental
    public void addLocationEngineName(@NonNull String locationEngineName, Hashtable<String, Object> eventWithAttributes) {
        eventWithAttributes.put("locationEngine", locationEngineName);
    }

    @Experimental
    public void addAbsoluteDistanceToDestination(int absoluteDistanceToDestination, Hashtable<String, Object> eventWithAttributes) {
        eventWithAttributes.put("absoluteDistanceToDestination", absoluteDistanceToDestination);
    }

    @Experimental
    public void addPercentTimeInForeground(@IntRange(from=0L, to=100L) int percentTimeInForeground, Hashtable<String, Object> eventWithAttributes) {
        eventWithAttributes.put("percentTimeInForeground", percentTimeInForeground);
    }

    @Experimental
    public void addPercentTimeInPortrait(@IntRange(from=0L, to=100L) int percentTimeInPortrait, Hashtable<String, Object> eventWithAttributes) {
        eventWithAttributes.put("percentTimeInPortrait", percentTimeInPortrait);
    }

    @Experimental
    public void addRatingToCancel(@IntRange(from=-1L, to=100L) int rating, Hashtable<String, Object> eventWithAttributes) {
        eventWithAttributes.put("rating", rating);
    }

    @Experimental
    public void addCommentToCancel(String comment, Hashtable<String, Object> eventWithAttributes) {
        eventWithAttributes.put("comment", comment);
    }

    static {
        VALID_USER_AGENTS = new ArrayList<String>(){
            {
                this.add("MapboxEventsAndroid/");
                this.add("MapboxTelemetryAndroid/");
                this.add("MapboxEventsUnityAndroid/");
                this.add("mapbox-navigation-android/");
                this.add("mapbox-navigation-ui-android/");
            }
        };
    }
}

