package com.mapbox.mapboxsdk.plugins.china.shift;

import android.util.Log;

import com.mapbox.geojson.Point;
import com.mapbox.geojson.shifter.CoordinateShifter;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

/**
 * A class created by the Mapbox team. This class uses the shifting algorithm to shift longitude and latitude
 * coordinates so that the coordinates are complaint with China's GCJ-O2 coordinate system.
 *
 * @since 1.2.0
 */
public class ChinaCoordinateShifter implements CoordinateShifter {

  private ShiftForChina shiftForChina = new ShiftForChina();

  public ChinaCoordinateShifter() {
  }

  /**
   * Shifts the longitude and latitude coordinates of a Point object.
   *
   * @param lon The unshifted longitude coordinate.
   * @param lat The unshifted longitude coordinate.
   * @return A list of doubles. The list has the shifted longitude and shifted latitude coordinates.
   *
   * @since 1.2.0
   */
  @Override
  public List<Double> shiftLonLat(double lon, double lat) {
    return shift(lon, lat);
  }

  /**
   * Shifts the longitude and latitude coordinates of a Point object. Also adds the Point object's altitude value
   * to the list.
   *
   * @param lon      The unshifted longitude coordinate.
   * @param lat      The unshifted longitude coordinate.
   * @param altitude The altitude value for the Point object.
   * @return A list of doubles. The list has the shifted longitude, the shifted latitude coordinates, and the altitude.
   *
   * @since 1.2.0
   */
  @Override
  public List<Double> shiftLonLatAlt(double lon, double lat, double altitude) {
    List<Double> shiftedLongLat = shift(lon, lat);
    shiftedLongLat.add(altitude);
    return shiftedLongLat;
  }

  /**
   * "Unshifts" a Point object which has GCJ-02 coordinates.
   *
   * @param shiftedPoint The Point object with coordinates that are shifted for the GCJ-O2 system
   * @return A list of doubles. Mapbox does not provide logic to convert coordinates from the
   * GCJ-02 coordinate system to the WGS 84 system.
   *
   * @since 1.2.0
   */
  @Override
  public List<Double> unshiftPoint(Point shiftedPoint) {
    List<Double> shiftedCoordinates = new ArrayList(2);
    shiftedCoordinates.add(shiftedPoint.longitude());
    shiftedCoordinates.add(shiftedPoint.latitude());
    return shiftedCoordinates;
  }

  /**
   * "Unshifts" a list of Double objects which has GCJ-02 coordinates.
   *
   * @param shiftedCoordinates A list of doubles. The list has shifted longitude coordinate and shifted latitude
   * that abide by China's GCJ-O2 coordinate system.
   * @return A list of doubles. The list has the unshifted coordinates that are on the WGS system.
   *
   * @since 1.2.0
   */
  @Override
  public List<Double> unshiftPoint(List<Double> shiftedCoordinates) {
    // Left empty on purpose
    return shiftedCoordinates;
  }

  /**
   * Internal shifting method that handles shifting within this class. This method is created so that shifting
   * doesn't have to be duplicated in this class' shiftLonLat() and shiftLonLatAlt() methods.
   *
   * @param lon The unshifted longitude coordinate.
   * @param lat The unshifted longitude coordinate.
   * @return A list of doubles. The list has the shifted longitude coordinate and the shifted latitude coordinate.
   *
   * @since 1.2.0
   */
  private List<Double> shift(double lon, double lat) {
    String toJson = shiftForChina.shift(lon, lat);
    List<Double> shiftedCoordinates = new ArrayList(3);
    try {
      JSONObject jsonObject = new JSONObject(toJson);
      double shiftedLon = jsonObject.getDouble("lon");
      double shiftedLat = jsonObject.getDouble("lat");
      shiftedCoordinates.add(shiftedLon);
      shiftedCoordinates.add(shiftedLat);
    } catch (JSONException jsonException) {
      shiftedCoordinates.add(lon);
      shiftedCoordinates.add(lat);
      jsonException.printStackTrace();
      Log.d("ChinaCoordinateShifter","Can't convert shifted coordinate String to a JSONObject");
    }
    return shiftedCoordinates;
  }
}
