/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */

package com.liferay.mobile.push;

import android.content.Context;

import androidx.annotation.NonNull;

import com.google.android.gms.tasks.OnCompleteListener;
import com.google.android.gms.tasks.Task;
import com.google.firebase.iid.FirebaseInstanceId;
import com.google.firebase.iid.InstanceIdResult;

import com.liferay.mobile.android.callback.typed.JSONObjectCallback;
import com.liferay.mobile.android.service.Session;
import com.liferay.mobile.android.service.SessionImpl;
import com.liferay.mobile.android.util.PortalVersion;

import java.util.ArrayList;
import java.util.List;

import org.json.JSONArray;
import org.json.JSONObject;

/**
 * @author Bruno Farache
 */
public class Push {

	public static final String FIREBASE = "firebase";

	public static Push with(Session session) {
		return new Push(session);
	}

	public Push onFailure(OnFailure onFailure) {
		this.onFailure = onFailure;

		return this;
	}

	public Push onPushNotification(OnPushNotification onPushNotification) {
		_subscriber.subscribe();

		this.onPushNotification = onPushNotification;

		return this;
	}

	public Push onSuccess(OnSuccess onSuccess) {
		this.onSuccess = onSuccess;

		return this;
	}

	public void register() {
		FirebaseInstanceId.getInstance().getInstanceId()
				.addOnCompleteListener(new OnCompleteListener<InstanceIdResult>() {
					@Override
					public void onComplete(@NonNull Task<InstanceIdResult> task) {
						if (!task.isSuccessful()) {
							return;
						}

						String token = task.getResult().getToken();

						try {
							register(token);
						} catch (Exception e) {
							if (Push.this.onFailure != null) {
								Push.this.onFailure.onFailure(e);
							}
						}
					}
				});
	}

	@Deprecated
	public void register(Context context, String senderId) throws Exception {
		register();
	}

	public void register(String registrationId) throws Exception {
		getService().addPushNotificationsDevice(registrationId, FIREBASE);
	}

	public void send(List<Long> toUserIds, JSONObject pushNotification) throws Exception {

		JSONArray toUserIdsJSONArray = new JSONArray();

		for (long toUserId : toUserIds) {
			toUserIdsJSONArray.put(toUserId);
		}

		getService().sendPushNotification(toUserIdsJSONArray, pushNotification.toString());
	}

	public void send(long toUserId, JSONObject pushNotification) throws Exception {

		List<Long> toUserIds = new ArrayList<Long>();
		toUserIds.add(toUserId);

		send(toUserIds, pushNotification);
	}

	public void unregister(String registrationId) throws Exception {
		getService().deletePushNotificationsDevice(registrationId);
	}

	public void unsubscribe() {
		_subscriber.unsubscribe();
	}

	public Push withPortalVersion(int portalVersion) {
		_portalVersion = portalVersion;
		return this;
	}

	public interface OnFailure {

		void onFailure(Exception e);
	}

	public interface OnPushNotification {

		void onPushNotification(JSONObject pushNotification);
	}

	public interface OnSuccess {

		void onSuccess(JSONObject jsonObject);
	}

	protected Push(Session session) {
		_session = new SessionImpl(session);
		_session.setHeaders(session.getHeaders());
		_subscriber = new PushSubscriber(this);
		_session.setCallback(new JSONObjectCallback() {

			@Override
			public void onFailure(Exception e) {
				if (onFailure != null) {
					onFailure.onFailure(e);
				}
			}

			@Override
			public void onSuccess(JSONObject jsonObject) {
				if (onSuccess != null) {
					onSuccess.onSuccess(jsonObject);
				}
			}
		});
	}

	protected PushNotificationsDeviceServiceWrapper getService() {
		return new PushNotificationsDeviceServiceWrapper(_session, _portalVersion);
	}

	protected OnFailure onFailure;
	protected OnPushNotification onPushNotification;
	protected OnSuccess onSuccess;

	private int _portalVersion = PortalVersion.V_6_2;
	private Session _session;
	private PushSubscriber _subscriber;
}