/**
 * Copyright (c) 2000-2019 Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */
package com.liferay.faces.lsv_485.patch.renderkit;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.faces.context.ExternalContext;
import javax.portlet.ClientDataRequest;
import javax.portlet.PortletContext;
import javax.portlet.filter.PortletRequestWrapper;
import javax.servlet.AsyncContext;
import javax.servlet.DispatcherType;
import javax.servlet.RequestDispatcher;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletInputStream;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.servlet.http.Part;

import com.liferay.faces.util.logging.Logger;
import com.liferay.faces.util.logging.LoggerFactory;
import java.util.HashMap;


/**
 * @author  Kyle Stiemann
 */
public final class HttpServletRequestFileUploadAdapter extends PortletRequestWrapper implements HttpServletRequest,
	ClientDataRequest {

	// Logger
	private static final Logger logger = LoggerFactory.getLogger(HttpServletRequestFileUploadAdapter.class);

	// Private Constants
	private static final String CONTENT_LENGTH = "Content-Length";
	private static final String CONTENT_TYPE = "Content-Type";

	// Private Final Data Members
	private final ExternalContext externalContext;
	private final Map<String, List> uploadedFileMap;
	private final ServletContext servletContext;

	// Private Data Members
	private String queryString;
	private Collection<Part> parts;

	public HttpServletRequestFileUploadAdapter(ClientDataRequest clientDataRequest, Map<String, List> uploadedFileMap,
		ExternalContext externalContext) {

		super(clientDataRequest);
		this.uploadedFileMap = uploadedFileMap;

		PortletContext portletContext = (PortletContext) externalContext.getContext();
		this.servletContext = new ServletContextFileUploadAdapterImpl(portletContext);
		this.externalContext = externalContext;
	}

	private static Part newPart(Object uploadedFile, String partName) {

		Part part;

		if (uploadedFile instanceof com.liferay.faces.util.model.UploadedFile) {
			part = new PartFileUploadAdapterUtilImpl((com.liferay.faces.util.model.UploadedFile) uploadedFile,
					partName);
		}
		else if (uploadedFile instanceof com.liferay.faces.bridge.model.UploadedFile) {
			part = new PartFileUploadAdapterBridgeImpl((com.liferay.faces.bridge.model.UploadedFile) uploadedFile,
					partName);
		}
		else {
			throw new UnsupportedOperationException();
		}

		return part;
	}

	@Override
	public boolean authenticate(HttpServletResponse response) throws IOException, ServletException {
		throw new UnsupportedOperationException();
	}

	@Override
	public AsyncContext getAsyncContext() {
		throw new UnsupportedOperationException();
	}

	@Override
	public String getCharacterEncoding() {
		return getClientDataRequest().getCharacterEncoding();
	}

	@Override
	public int getContentLength() {
		return getClientDataRequest().getContentLength();
	}

	public long getContentLengthLong() {
		return getContentLength();
	}

	@Override
	public String getContentType() {
		return getClientDataRequest().getContentType();
	}

	@Override
	public long getDateHeader(String name) {
		throw new IllegalArgumentException();
	}

	@Override
	public DispatcherType getDispatcherType() {
		throw new UnsupportedOperationException();
	}

	private static Map getRequestHeaderMap(ExternalContext externalContext, boolean arrayValue) {

		Map originalRequestHeaderMap;

		if (arrayValue) {
			originalRequestHeaderMap = externalContext.getRequestHeaderValuesMap();
		}
		else {
			originalRequestHeaderMap = externalContext.getRequestHeaderMap();
		}

		boolean containsContentLength = originalRequestHeaderMap.containsKey(CONTENT_LENGTH);
		boolean containsContentType = originalRequestHeaderMap.containsKey(CONTENT_TYPE);

		if (containsContentLength && containsContentType) {
			return originalRequestHeaderMap;
		}

		Map requestHeaderMap = new HashMap(originalRequestHeaderMap);

		if (!containsContentLength) {

			Object requestContentLengthValue = Integer.toString(externalContext.getRequestContentLength());
			addHeader(requestHeaderMap, CONTENT_LENGTH, requestContentLengthValue, arrayValue);
		}

		if (!containsContentType) {

			Object requestContentTypeValue = externalContext.getRequestContentType();
			addHeader(requestHeaderMap, CONTENT_TYPE, requestContentTypeValue, arrayValue);
		}

		return Collections.unmodifiableMap(requestHeaderMap);
	}

	private static void addHeader(Map requestHeaderMap, String headerName, Object headerValue, boolean arrayValue) {

		if (arrayValue) {
			headerValue = new String[] { (String) headerValue };
		}

		requestHeaderMap.put(headerName, headerValue);
	}

	@Override
	public String getHeader(String name) {

		Map<String, String> requestHeaderMap = getRequestHeaderMap(externalContext, false);

		return requestHeaderMap.get(name);
	}

	@Override
	public Enumeration<String> getHeaderNames() {

		Map<String, String[]> requestHeaderValuesMap = getRequestHeaderMap(externalContext, true);

		return Collections.enumeration(requestHeaderValuesMap.keySet());
	}

	@Override
	public Enumeration<String> getHeaders(String name) {

		Map<String, String[]> requestHeaderValuesMap = getRequestHeaderMap(externalContext, true);
		String[] headersArray = requestHeaderValuesMap.get(name);
		List<String> headers;

		if (headersArray != null) {
			headers = Arrays.asList(headersArray);
		}
		else {
			headers = Collections.emptyList();
		}

		return Collections.enumeration(headers);
	}

	@Override
	public ServletInputStream getInputStream() throws IOException {
		throw new UnsupportedOperationException();
	}

	@Override
	public int getIntHeader(String name) {

		int intHeader = -1;
		String header = getHeader(name);

		if (header != null) {
			intHeader = Integer.parseInt(header);
		}

		return intHeader;
	}

	@Override
	public String getLocalAddr() {
		throw new UnsupportedOperationException();
	}

	@Override
	public String getLocalName() {
		throw new UnsupportedOperationException();
	}

	@Override
	public int getLocalPort() {
		throw new UnsupportedOperationException();
	}

	@Override
	public String getMethod() {
		return getClientDataRequest().getMethod();
	}

	@Override
	public Part getPart(String name) throws IOException, ServletException {

		Part part = null;
		List uploadedFiles = uploadedFileMap.get(name);

		if (uploadedFiles != null && !uploadedFiles.isEmpty()) {
			part = newPart(uploadedFiles.get(0), name);
		}

		return part;
	}

	@Override
	public Collection<Part> getParts() throws IOException, ServletException {

		if (parts == null) {

			List<Part> parts = new ArrayList<Part>();
			Set<Map.Entry<String, List>> uploadedFileMapEntries = uploadedFileMap.entrySet();

			for (Map.Entry<String, List> uploadedFileMapEntry : uploadedFileMapEntries) {

				List uploadedFilesForName = uploadedFileMapEntry.getValue();

				if ((uploadedFilesForName != null) && !uploadedFilesForName.isEmpty()) {

					String partName = uploadedFileMapEntry.getKey();

					for (Object uploadedFile : uploadedFilesForName) {
						parts.add(newPart(uploadedFile, partName));
					}
				}
			}

			this.parts = Collections.unmodifiableList(parts);
		}

		return parts;
	}

	@Override
	public String getPathInfo() {
		return null;
	}

	@Override
	public String getPathTranslated() {
		return null;
	}

	@Override
	public InputStream getPortletInputStream() throws IOException {
		return getClientDataRequest().getPortletInputStream();
	}

	@Override
	public String getProtocol() {
		throw new UnsupportedOperationException();
	}

	@Override
	public String getQueryString() {

		if (queryString == null) {

			StringBuilder stringBuilder = new StringBuilder();
			Map<String, String[]> parameterMap = getParameterMap();

			for (Map.Entry<String, String[]> parameter : parameterMap.entrySet()) {

				String[] values = parameter.getValue();
				String name = parameter.getKey();

				if (values.length > 0) {

					for (String value : values) {

						if (stringBuilder.length() > 0) {
							stringBuilder.append("&");
						}

						try {

							stringBuilder.append(URLEncoder.encode(name, "UTF-8"));
							stringBuilder.append("=");
							stringBuilder.append(URLEncoder.encode(value, "UTF-8"));
						}
						catch (UnsupportedEncodingException e) {

							logger.error("Unable to url encode name=[{0}], value=[{1}] pair with UTF-8 encoding:", name,
								value);
							logger.error(e);
						}
					}
				}
				else {

					if (stringBuilder.length() > 0) {
						stringBuilder.append("&");
					}

					stringBuilder.append(name);
				}
			}

			queryString = stringBuilder.toString();
		}

		return queryString;
	}

	@Override
	public BufferedReader getReader() throws IOException {
		throw new UnsupportedOperationException();
	}

	@Override
	public String getRealPath(String path) {
		throw new UnsupportedOperationException();
	}

	@Override
	public String getRemoteAddr() {
		throw new UnsupportedOperationException();
	}

	@Override
	public String getRemoteHost() {
		throw new UnsupportedOperationException();
	}

	@Override
	public int getRemotePort() {
		throw new UnsupportedOperationException();
	}

	@Override
	public RequestDispatcher getRequestDispatcher(String path) {
		throw new UnsupportedOperationException();
	}

	@Override
	public String getRequestURI() {
		return getContextPath();
	}

	@Override
	public StringBuffer getRequestURL() {
		throw new UnsupportedOperationException();
	}

	@Override
	public ServletContext getServletContext() {
		return servletContext;
	}

	@Override
	public String getServletPath() {
		return "";
	}

	@Override
	public HttpSession getSession() {
		throw new UnsupportedOperationException();
	}

	@Override
	public HttpSession getSession(boolean create) {
		throw new UnsupportedOperationException();
	}

	@Override
	public boolean isAsyncStarted() {
		throw new UnsupportedOperationException();
	}

	@Override
	public boolean isAsyncSupported() {
		throw new UnsupportedOperationException();
	}

	@Override
	public boolean isRequestedSessionIdFromCookie() {
		throw new UnsupportedOperationException();
	}

	@Override
	public boolean isRequestedSessionIdFromURL() {
		throw new UnsupportedOperationException();
	}

	@Override
	public boolean isRequestedSessionIdFromUrl() {
		throw new UnsupportedOperationException();
	}

	@Override
	public void login(String username, String password) throws ServletException {
		throw new UnsupportedOperationException();
	}

	@Override
	public void logout() throws ServletException {
		throw new UnsupportedOperationException();
	}

	@Override
	public void setCharacterEncoding(String characterEncoding) throws UnsupportedEncodingException {
		getClientDataRequest().setCharacterEncoding(characterEncoding);
	}

	@Override
	public AsyncContext startAsync() throws IllegalStateException {
		throw new UnsupportedOperationException();
	}

	@Override
	public AsyncContext startAsync(ServletRequest servletRequest, ServletResponse servletResponse)
		throws IllegalStateException {
		throw new UnsupportedOperationException();
	}

	private ClientDataRequest getClientDataRequest() {
		return (ClientDataRequest) getRequest();
	}
}
