/**
 * Copyright (c) 2000-2019 Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */
package com.liferay.faces.lsv_485.patch.renderkit;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.List;
import java.util.Map;

import javax.faces.component.UIComponent;
import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.faces.render.Renderer;
import javax.portlet.ClientDataRequest;
import javax.portlet.PortletRequest;

import com.liferay.faces.bridge.BridgeFactoryFinder;
import com.liferay.faces.bridge.context.BridgeContext;
import com.liferay.faces.bridge.context.map.internal.ContextMapFactory;


/**
 * @author  Kyle Stiemann
 */
/* package-private */ final class FileUploadRendererPortletImpl extends RendererWrapper {

	// Private Final Data Members
	private final Renderer wrappedRenderer;
	private final boolean liferayFacesLegacy;
	private final Class<?> contextMapFactoryClass;

	public FileUploadRendererPortletImpl(Renderer renderer, Class<?> contextMapFactoryClass,
		boolean liferayFacesLegacy) {

		this.wrappedRenderer = renderer;
		this.contextMapFactoryClass = contextMapFactoryClass;
		this.liferayFacesLegacy = liferayFacesLegacy;
	}

	@Override
	public void decode(FacesContext facesContext, UIComponent uiComponent) {

		ExternalContext externalContext = facesContext.getExternalContext();
		ClientDataRequest clientDataRequest = (ClientDataRequest) externalContext.getRequest();
		Map<String, List> uploadedFileMap = (Map<String, List>) getUploadedFileMap(clientDataRequest);

		if ((uploadedFileMap != null) && !uploadedFileMap.isEmpty()) {

			try {

				externalContext.setRequest(new HttpServletRequestFileUploadAdapter(clientDataRequest, uploadedFileMap,
						externalContext));
				super.decode(facesContext, uiComponent);
			}
			finally {
				externalContext.setRequest(clientDataRequest);
			}
		}
	}

	@Override
	public Renderer getWrapped() {
		return wrappedRenderer;
	}

	private Map getUploadedFileMap(ClientDataRequest clientDataRequest) {

		UploadedFileMapGetter uploadedFileMapGetter;

		if (liferayFacesLegacy) {
			uploadedFileMapGetter = new UploadedFileMapGetterLegacyImpl();
		}
		else {
			uploadedFileMapGetter = new UploadedFileMapGetterImpl();
		}

		return uploadedFileMapGetter.getUploadedFileMap(contextMapFactoryClass, clientDataRequest);
	}

	private static interface UploadedFileMapGetter {
		public Map getUploadedFileMap(Class<?> contextMapFactoryClass, ClientDataRequest clientDataRequest);
	}

	private static final class UploadedFileMapGetterImpl implements UploadedFileMapGetter {

		@Override
		public Map getUploadedFileMap(Class<?> contextMapFactoryClass, ClientDataRequest clientDataRequest) {

			Object contextMapFactory = BridgeFactoryFinder.getFactory(contextMapFactoryClass);

			try {

				Method method = contextMapFactoryClass.getMethod("getUploadedFileMap", PortletRequest.class);

				return (Map) method.invoke(contextMapFactory, clientDataRequest);
			}
			catch (IllegalAccessException e) {
				throw new RuntimeException(e);
			}
			catch (InvocationTargetException e) {
				throw new RuntimeException(e);
			}
			catch (NoSuchMethodException e) {
				throw new RuntimeException(e);
			}
		}
	}

	private static final class UploadedFileMapGetterLegacyImpl implements UploadedFileMapGetter {

		@Override
		public Map getUploadedFileMap(Class<?> contextMapFactoryClass, ClientDataRequest clientDataRequest) {

			ContextMapFactory contextMapFactory = (ContextMapFactory) BridgeFactoryFinder.getFactory(
					contextMapFactoryClass);
			BridgeContext bridgeContext = BridgeContext.getCurrentInstance();

			return contextMapFactory.getUploadedFileMap(bridgeContext);
		}
	}
}
