/**
 * Copyright (c) 2000-2019 Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */
package com.liferay.faces.alloy.component.selectrating.internal;

import java.io.IOException;

import javax.faces.component.UIComponent;
import javax.faces.component.ValueHolder;
import javax.faces.context.FacesContext;
import javax.faces.context.ResponseWriter;

import com.liferay.faces.alloy.component.selectrating.SelectRating;
import com.liferay.faces.util.component.ClientComponent;
import com.liferay.faces.util.component.Styleable;
import com.liferay.faces.util.render.RendererUtil;


/**
 * @author  Vernon Singleton
 */
public class SelectRatingRenderer extends SelectRatingRendererBase {

	// Private Constants
	private static final String FACES_RUNTIME_ONCLICK = "facesRuntimeOnClick";
	private static final String SELECTED_INDEX = "selectedIndex";
	private static final String DOUBLE_BACKSLASH = "\\\\";

	@Override
	public void encodeChildren(FacesContext facesContext, UIComponent uiComponent) throws IOException {
		// NO-OP: prevent rendering children since they are rendered in the encodeMarkupBegin() method when
		// super.encodeAll() is called.
	}

	@Override
	public void encodeJavaScriptCustom(FacesContext facesContext, UIComponent uiComponent) throws IOException {

		ResponseWriter responseWriter = facesContext.getResponseWriter();

		ClientComponent clientComponent = (ClientComponent) uiComponent;
		String clientVarName = getClientVarName(facesContext, clientComponent);
		String clientKey = clientComponent.getClientKey();

		if (clientKey == null) {
			clientKey = clientVarName;
		}

		encodeLiferayComponentVar(responseWriter, clientVarName, clientKey);

		// The above should render something like this: var _1_WAR_showcaseportlet__j_idt6_j_idt19_j_idt22_j_idt23 =
		// Liferay.component('myClientKey');

		//J-
			// What should we consider when we initialize the value of this component?
			//
			// Contributor			   What value do they want?				What will we do with this value?
			// ----------------------  -----------------------------------	---------------------------------------------
			// 1. AlloyUI			   -1									Ignore.
			//
			// 2. JSF				   ""									Use as default.
			//
			// 3. Developer Attribute  Liferay.component(liferayKey).		If defined, use this, but remember it is the
			//							.get('defaultSelected')				number of stars, not the value.
			//
			// 4. Developer EL		   rating.getValue()					If defined, use this.
			//
			// 5. User Event												Since no interaction from the user yet, use
			//																the value chosen above.
		//J+

		// 2. JSF
		String hiddenInputValue = "";

		//J-
		// 3. Developer attribute
		// no developer attribute is allowed for this component, even though alloyui allows for an attribute called defaultSelected
		// that attribute has not been exposed here in favor of using normal JSF application techniques for establishing defaults
		// in the model.
		//J+

		// 4. and 5. If the developer EL or user input is specified, then the value of the hidden input field should be
		// set to this value of the rating component
		ValueHolder valueHolder = (ValueHolder) uiComponent;
		Object value = valueHolder.getValue();

		if (value != null) {
			hiddenInputValue = value.toString();
		}

		// Initialize the hidden input generated by AlloyUI to the hiddenInputValue, instead of its default of -1.
		responseWriter.write("document.getElementsByName('");

		String clientId = uiComponent.getClientId(facesContext);
		String escapedClientId = RendererUtil.escapeJavaScript(clientId);
		responseWriter.write(escapedClientId);
		responseWriter.write("')[0].value='");

		String escapedHiddenInputValue = RendererUtil.escapeJavaScript(hiddenInputValue);
		responseWriter.write(escapedHiddenInputValue);
		responseWriter.write("';");

		// Make sure that the rendered rating is correct (i.e. how many stars are filled). The only way that the
		// client-side LiferayComponent would have a selectedIndex at this point is if the defaultSelected attribute
		// is set. Carefully decide if we need to clear the selected rating or if we need to select the user's selected
		// rating.
		Long selectedIndex = (Long) facesContext.getAttributes().remove(SELECTED_INDEX);

		// If the selectedIndex is -1, then that means one of two things ... 1. the user selected no value (or cleared
		// the stars), which must have been in a postback. 2. no selected index was found because this is the initial
		// render of the page, and no valid default value was set.
		if (selectedIndex.intValue() == SelectRatingResponseWriter.NO_SELECTION_INDEX) {

			// This is case 1.  the user selected to clear the value.
			if (facesContext.isPostback()) {
				responseWriter.write(clientVarName);
				responseWriter.write(".clearSelection();");
			}

			// Otherwise, this is case 2. Since there has been no user interaction yet, then we do not need to clear
			// anything, or select anything, since the Alloy component will simply be able to render itself correctly.
			else {
				// no operation needed.
			}
		}

		//J-
		// Otherwise, JSF says that this component has a value, because JSF rendered one of the input type="radio" as
		// "checked" and its selectedIndex is not -1.  There are two possible reasons for this:
		// 1. The model has a default value coming through on initial render
		// 2. The user clicked on a star, this is a postback, and we need the component to have that many stars filled.
		//
		// This is the case where an alloySelectedIndex has been established (or Alloy renders stars already
		// filled to the defaultSelected input), but the user has selected a different number of stars than the
		// alloySelectedIndex, so we need to "select" the correct number of stars to be filled. If the alloySelectedIndex is
		// the same as the indexSelectedByJSF, we do not want to select it again, since alloy has already filled the
		// correct number of stars. If we did select it again, it would clear the stars ... not showing what the user
		// selected.
		//J+
		else {

			String indexSelectedByJSF = selectedIndex.toString();

			responseWriter.write("var alloySelectedIndex=");
			responseWriter.write(clientVarName);
			responseWriter.write(".get('selectedIndex');if(");

			responseWriter.write(indexSelectedByJSF);
			responseWriter.write("!=alloySelectedIndex){");
			responseWriter.write(clientVarName);
			responseWriter.write(".select(");
			responseWriter.write(indexSelectedByJSF);
			responseWriter.write(");}");
		}

		//J-
			// What should we consider when a user clicks on the component?
			//
			// Contributor			   What value do they want?				What will we do with this value?
			// ----------------------  -----------------------------------	---------------------------------------------
			// 1. AlloyUI			   event.target.get('value')			Use this as the value unless it is -1 which is
			//																the cancel value.
			//
			// 2. JSF				   ''									Use this value instead of -1 or the cancel
			//																value.
			//
			// 3. Developer Attribute  Liferay.component('liferayKey')		Ignore because user input is overriding this.
			//							 .get('defaultSelected')
			//
			// 4. Developer EL		   rating.getValue()					Ignore because user input is overriding this.
			//
			// 5. User Event		   event.target.get('value')			Use this unless it is the cancel or reset
			//																value.
		//J+

		// Start encoding the onclick event.
		responseWriter.write(clientVarName);
		responseWriter.write(".on('click',function(event){");

		// Within the onclick event, establish the newValue of the hiddenInput.
		responseWriter.write("var newValue=(event.target.get('value')=='-1')?'':event.target.get('value');");

		// Within the onclick event, set the newValue of the hidden input.
		responseWriter.write("document.getElementsByName('");
		responseWriter.write(escapedClientId);
		responseWriter.write("')[0].value=newValue;");

		// Within the onclick event, write out the jsf onclick event to call, that was captured by the ResponseWriter,
		// if any.
		String onClick = (String) facesContext.getAttributes().remove(FACES_RUNTIME_ONCLICK);

		if (onClick != null) {

			// Escape the escapedClientId's backslashes since it is passed as a argument to the JavaScript
			// replaceFirst function below, which expects a regex.
			String backslashEscapedClientId = escapedClientId.replace("\\", DOUBLE_BACKSLASH);
			String hiddenInputNodeJs = "document.getElementsByName('" + backslashEscapedClientId + "')[0]";
			responseWriter.write(onClick.replaceFirst("this", hiddenInputNodeJs));
		}

		// Finish encoding the onclick event.
		responseWriter.write("});");
	}

	@Override
	public void encodeMarkupBegin(FacesContext facesContext, UIComponent uiComponent) throws IOException {

		ResponseWriter responseWriter = facesContext.getResponseWriter();

		// Start the encoding of the outermost <span> element.
		responseWriter.startElement("span", uiComponent);
		responseWriter.writeAttribute("id", uiComponent.getClientId(facesContext), "id");
		RendererUtil.encodeStyleable(responseWriter, (Styleable) uiComponent);

		// Encode the child radio inputs by delegating to the renderer from the JSF runtime using our own
		// SelectRatingResponseWriter to control the output.
		SelectRatingResponseWriter selectRatingResponseWriter = new SelectRatingResponseWriter(responseWriter);
		super.encodeAll(facesContext, uiComponent, selectRatingResponseWriter);

		// Save the onclick for later use in the JavaScript.
		String onClick = selectRatingResponseWriter.getOnClick();
		facesContext.getAttributes().put(FACES_RUNTIME_ONCLICK, onClick);

		// Save the selectedIndex for later use in the JavaScript.
		Long selectedIndex = selectRatingResponseWriter.getSelectedIndex();
		facesContext.getAttributes().put(SELECTED_INDEX, selectedIndex);

	}

	@Override
	public void encodeMarkupEnd(FacesContext facesContext, UIComponent uiComponent) throws IOException {

		// Finish the encoding of the outermost </span> element.
		ResponseWriter responseWriter = facesContext.getResponseWriter();
		responseWriter.endElement("span");
	}

	@Override
	public boolean getRendersChildren() {
		return true;
	}

	@Override
	protected void encodeHiddenAttributes(FacesContext facesContext, ResponseWriter responseWriter,
		SelectRating selectRating, boolean first) throws IOException {

		// boundingBox
		String clientId = selectRating.getClientId(facesContext);
		encodeClientId(responseWriter, BOUNDING_BOX, clientId, first);

		first = false;

		// render : true
		encodeWidgetRender(responseWriter, first);

	}
}
