/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.headless.commerce.admin.account.internal.resource.v1_0;

import com.liferay.commerce.account.exception.NoSuchAccountException;
import com.liferay.commerce.account.exception.NoSuchAccountGroupException;
import com.liferay.commerce.account.model.CommerceAccount;
import com.liferay.commerce.account.model.CommerceAccountGroup;
import com.liferay.commerce.account.service.CommerceAccountGroupService;
import com.liferay.commerce.account.service.CommerceAccountService;
import com.liferay.headless.commerce.admin.account.dto.v1_0.AccountGroup;
import com.liferay.headless.commerce.admin.account.internal.dto.v1_0.converter.AccountGroupDTOConverter;
import com.liferay.headless.commerce.admin.account.internal.odata.entity.v1_0.AccountGroupEntityModel;
import com.liferay.headless.commerce.admin.account.resource.v1_0.AccountGroupResource;
import com.liferay.headless.commerce.core.util.ExpandoUtil;
import com.liferay.headless.commerce.core.util.ServiceContextHelper;
import com.liferay.petra.function.transform.TransformUtil;
import com.liferay.petra.string.StringPool;
import com.liferay.portal.kernel.search.Field;
import com.liferay.portal.kernel.search.Sort;
import com.liferay.portal.kernel.search.filter.Filter;
import com.liferay.portal.kernel.util.GetterUtil;
import com.liferay.portal.kernel.util.Validator;
import com.liferay.portal.odata.entity.EntityModel;
import com.liferay.portal.vulcan.dto.converter.DefaultDTOConverterContext;
import com.liferay.portal.vulcan.pagination.Page;
import com.liferay.portal.vulcan.pagination.Pagination;
import com.liferay.portal.vulcan.util.SearchUtil;

import java.util.Collections;
import java.util.Map;

import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.Response;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ServiceScope;

/**
 * @author Alessio Antonio Rendina
 */
@Component(
	properties = "OSGI-INF/liferay/rest/v1_0/account-group.properties",
	scope = ServiceScope.PROTOTYPE, service = AccountGroupResource.class
)
public class AccountGroupResourceImpl extends BaseAccountGroupResourceImpl {

	@Override
	public Response deleteAccountGroup(Long id) throws Exception {
		_commerceAccountGroupService.deleteCommerceAccountGroup(id);

		Response.ResponseBuilder responseBuilder = Response.ok();

		return responseBuilder.build();
	}

	@Override
	public Response deleteAccountGroupByExternalReferenceCode(
			String externalReferenceCode)
		throws Exception {

		CommerceAccountGroup commerceAccountGroup =
			_commerceAccountGroupService.fetchByExternalReferenceCode(
				contextCompany.getCompanyId(), externalReferenceCode);

		if (commerceAccountGroup == null) {
			throw new NoSuchAccountGroupException(
				"Unable to find account group with external reference code " +
					externalReferenceCode);
		}

		_commerceAccountGroupService.deleteCommerceAccountGroup(
			commerceAccountGroup.getCommerceAccountGroupId());

		Response.ResponseBuilder responseBuilder = Response.ok();

		return responseBuilder.build();
	}

	@Override
	public Page<AccountGroup>
			getAccountByExternalReferenceCodeAccountGroupsPage(
				String externalReferenceCode, Pagination pagination)
		throws Exception {

		CommerceAccount commerceAccount =
			_commerceAccountService.fetchByExternalReferenceCode(
				contextCompany.getCompanyId(), externalReferenceCode);

		if (commerceAccount == null) {
			throw new NoSuchAccountException(
				"Unable to find account with external reference code " +
					externalReferenceCode);
		}

		return _getAccountAccountGroups(
			commerceAccount.getCommerceAccountId(), pagination);
	}

	@Override
	public AccountGroup getAccountGroup(Long id) throws Exception {
		return _accountGroupDTOConverter.toDTO(
			new DefaultDTOConverterContext(
				GetterUtil.getLong(id),
				contextAcceptLanguage.getPreferredLocale()));
	}

	@Override
	public AccountGroup getAccountGroupByExternalReferenceCode(
			String externalReferenceCode)
		throws Exception {

		CommerceAccountGroup commerceAccountGroup =
			_commerceAccountGroupService.fetchByExternalReferenceCode(
				contextCompany.getCompanyId(), externalReferenceCode);

		if (commerceAccountGroup == null) {
			throw new NoSuchAccountGroupException(
				"Unable to find account group with external reference code " +
					externalReferenceCode);
		}

		return _accountGroupDTOConverter.toDTO(
			new DefaultDTOConverterContext(
				commerceAccountGroup.getCommerceAccountGroupId(),
				contextAcceptLanguage.getPreferredLocale()));
	}

	@Override
	public Page<AccountGroup> getAccountGroupsPage(
			Filter filter, Pagination pagination, Sort[] sorts)
		throws Exception {

		return SearchUtil.search(
			Collections.emptyMap(),
			booleanQuery -> booleanQuery.getPreBooleanFilter(), filter,
			com.liferay.account.model.AccountGroup.class.getName(),
			StringPool.BLANK, pagination,
			queryConfig -> queryConfig.setSelectedFieldNames(
				Field.ENTRY_CLASS_PK),
			searchContext -> searchContext.setCompanyId(
				contextCompany.getCompanyId()),
			sorts,
			document -> _toAccountGroup(
				_commerceAccountGroupService.getCommerceAccountGroup(
					GetterUtil.getLong(document.get(Field.ENTRY_CLASS_PK)))));
	}

	@Override
	public Page<AccountGroup> getAccountIdAccountGroupsPage(
			Long id, Pagination pagination)
		throws Exception {

		return _getAccountAccountGroups(id, pagination);
	}

	@Override
	public EntityModel getEntityModel(MultivaluedMap multivaluedMap)
		throws Exception {

		return _entityModel;
	}

	@Override
	public Response patchAccountGroup(Long id, AccountGroup accountGroup)
		throws Exception {

		_commerceAccountGroupService.updateCommerceAccountGroup(
			id, accountGroup.getName(),
			_serviceContextHelper.getServiceContext());

		Response.ResponseBuilder responseBuilder = Response.ok();

		return responseBuilder.build();
	}

	@Override
	public Response patchAccountGroupByExternalReferenceCode(
			String externalReferenceCode, AccountGroup accountGroup)
		throws Exception {

		CommerceAccountGroup commerceAccountGroup =
			_commerceAccountGroupService.fetchByExternalReferenceCode(
				contextCompany.getCompanyId(), externalReferenceCode);

		if (commerceAccountGroup == null) {
			throw new NoSuchAccountGroupException(
				"Unable to find account group with external reference code " +
					externalReferenceCode);
		}

		_commerceAccountGroupService.updateCommerceAccountGroup(
			commerceAccountGroup.getCommerceAccountGroupId(),
			accountGroup.getName(), _serviceContextHelper.getServiceContext());

		// Expando

		Map<String, ?> customFields = accountGroup.getCustomFields();

		if ((customFields != null) && !customFields.isEmpty()) {
			ExpandoUtil.updateExpando(
				contextCompany.getCompanyId(),
				com.liferay.account.model.AccountGroup.class,
				commerceAccountGroup.getPrimaryKey(), customFields);
		}

		Response.ResponseBuilder responseBuilder = Response.ok();

		return responseBuilder.build();
	}

	@Override
	public AccountGroup postAccountGroup(AccountGroup accountGroup)
		throws Exception {

		CommerceAccountGroup commerceAccountGroup = null;

		if (Validator.isNotNull(accountGroup.getExternalReferenceCode())) {
			commerceAccountGroup =
				_commerceAccountGroupService.fetchByExternalReferenceCode(
					contextCompany.getCompanyId(),
					accountGroup.getExternalReferenceCode());
		}

		if (commerceAccountGroup == null) {
			commerceAccountGroup =
				_commerceAccountGroupService.addCommerceAccountGroup(
					contextCompany.getCompanyId(), accountGroup.getName(), 0,
					accountGroup.getExternalReferenceCode(),
					_serviceContextHelper.getServiceContext());
		}
		else {
			commerceAccountGroup =
				_commerceAccountGroupService.updateCommerceAccountGroup(
					commerceAccountGroup.getCommerceAccountGroupId(),
					accountGroup.getName(),
					_serviceContextHelper.getServiceContext());
		}

		// Expando

		Map<String, ?> customFields = accountGroup.getCustomFields();

		if ((customFields != null) && !customFields.isEmpty()) {
			ExpandoUtil.updateExpando(
				contextCompany.getCompanyId(),
				com.liferay.account.model.AccountGroup.class,
				commerceAccountGroup.getPrimaryKey(), customFields);
		}

		return _accountGroupDTOConverter.toDTO(
			new DefaultDTOConverterContext(
				commerceAccountGroup.getCommerceAccountGroupId(),
				contextAcceptLanguage.getPreferredLocale()));
	}

	private Page<AccountGroup> _getAccountAccountGroups(
			long commerceAccountId, Pagination pagination)
		throws Exception {

		return Page.of(
			TransformUtil.transform(
				_commerceAccountGroupService.
					getCommerceAccountGroupsByCommerceAccountId(
						commerceAccountId, pagination.getStartPosition(),
						pagination.getEndPosition()),
				commerceAccountGroup -> _toAccountGroup(commerceAccountGroup)),
			pagination,
			_commerceAccountGroupService.
				getCommerceAccountGroupsByCommerceAccountIdCount(
					commerceAccountId));
	}

	private AccountGroup _toAccountGroup(
			CommerceAccountGroup commerceAccountGroup)
		throws Exception {

		return _accountGroupDTOConverter.toDTO(
			new DefaultDTOConverterContext(
				commerceAccountGroup.getCommerceAccountGroupId(),
				contextAcceptLanguage.getPreferredLocale()));
	}

	@Reference
	private AccountGroupDTOConverter _accountGroupDTOConverter;

	@Reference
	private CommerceAccountGroupService _commerceAccountGroupService;

	@Reference
	private CommerceAccountService _commerceAccountService;

	private final EntityModel _entityModel = new AccountGroupEntityModel();

	@Reference
	private ServiceContextHelper _serviceContextHelper;

}