/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.commerce.internal.permission;

import com.liferay.commerce.model.CommerceOrderType;
import com.liferay.commerce.permission.CommerceOrderTypePermission;
import com.liferay.commerce.service.CommerceOrderTypeLocalService;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.security.auth.PrincipalException;
import com.liferay.portal.kernel.security.permission.PermissionChecker;
import com.liferay.portal.kernel.util.ArrayUtil;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * @author Riccardo Alberti
 */
@Component(
	enabled = false, immediate = true,
	service = CommerceOrderTypePermission.class
)
public class CommerceOrderTypePermissionImpl
	implements CommerceOrderTypePermission {

	@Override
	public void check(
			PermissionChecker permissionChecker,
			CommerceOrderType commerceOrderType, String actionId)
		throws PortalException {

		if (!contains(permissionChecker, commerceOrderType, actionId)) {
			throw new PrincipalException.MustHavePermission(
				permissionChecker, CommerceOrderType.class.getName(),
				commerceOrderType.getCommerceOrderTypeId(), actionId);
		}
	}

	@Override
	public void check(
			PermissionChecker permissionChecker, long commerceOrderTypeId,
			String actionId)
		throws PortalException {

		if (!contains(permissionChecker, commerceOrderTypeId, actionId)) {
			throw new PrincipalException.MustHavePermission(
				permissionChecker, CommerceOrderType.class.getName(),
				commerceOrderTypeId, actionId);
		}
	}

	@Override
	public boolean contains(
			PermissionChecker permissionChecker,
			CommerceOrderType commerceOrderType, String actionId)
		throws PortalException {

		if (contains(
				permissionChecker, commerceOrderType.getCommerceOrderTypeId(),
				actionId)) {

			return true;
		}

		return false;
	}

	@Override
	public boolean contains(
			PermissionChecker permissionChecker, long commerceOrderTypeId,
			String actionId)
		throws PortalException {

		CommerceOrderType commerceOrderType =
			_commerceOrderTypeLocalService.fetchCommerceOrderType(
				commerceOrderTypeId);

		if (commerceOrderType == null) {
			return false;
		}

		return _contains(permissionChecker, commerceOrderType, actionId);
	}

	@Override
	public boolean contains(
			PermissionChecker permissionChecker, long[] commerceOrderTypeIds,
			String actionId)
		throws PortalException {

		if (ArrayUtil.isEmpty(commerceOrderTypeIds)) {
			return false;
		}

		for (long commerceOrderTypeId : commerceOrderTypeIds) {
			if (!contains(permissionChecker, commerceOrderTypeId, actionId)) {
				return false;
			}
		}

		return true;
	}

	private boolean _contains(
			PermissionChecker permissionChecker,
			CommerceOrderType commerceOrderType, String actionId)
		throws PortalException {

		if (permissionChecker.isCompanyAdmin(
				commerceOrderType.getCompanyId()) ||
			permissionChecker.isOmniadmin()) {

			return true;
		}

		if (permissionChecker.hasOwnerPermission(
				permissionChecker.getCompanyId(),
				CommerceOrderType.class.getName(),
				commerceOrderType.getCommerceOrderTypeId(),
				permissionChecker.getUserId(), actionId) &&
			(commerceOrderType.getUserId() == permissionChecker.getUserId())) {

			return true;
		}

		return permissionChecker.hasPermission(
			null, CommerceOrderType.class.getName(),
			commerceOrderType.getCommerceOrderTypeId(), actionId);
	}

	@Reference
	private CommerceOrderTypeLocalService _commerceOrderTypeLocalService;

}