/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */

package com.liferay.commerce.inventory.web.internal.display.context;

import static com.liferay.portal.kernel.security.permission.PermissionThreadLocal.getPermissionChecker;

import com.liferay.commerce.frontend.ClayCreationMenu;
import com.liferay.commerce.frontend.ClayCreationMenuActionItem;
import com.liferay.commerce.frontend.model.HeaderActionModel;
import com.liferay.commerce.inventory.constants.CommerceInventoryActionKeys;
import com.liferay.commerce.inventory.model.CommerceInventoryReplenishmentItem;
import com.liferay.commerce.inventory.model.CommerceInventoryWarehouse;
import com.liferay.commerce.inventory.model.CommerceInventoryWarehouseItem;
import com.liferay.commerce.inventory.service.CommerceInventoryReplenishmentItemService;
import com.liferay.commerce.inventory.service.CommerceInventoryWarehouseItemService;
import com.liferay.commerce.inventory.service.CommerceInventoryWarehouseService;
import com.liferay.commerce.product.display.context.util.CPRequestHelper;
import com.liferay.portal.kernel.dao.orm.QueryUtil;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.language.LanguageUtil;
import com.liferay.portal.kernel.portlet.LiferayPortletResponse;
import com.liferay.portal.kernel.portlet.LiferayWindowState;
import com.liferay.portal.kernel.security.auth.PrincipalException;
import com.liferay.portal.kernel.service.permission.PortalPermissionUtil;
import com.liferay.portal.kernel.util.Constants;
import com.liferay.portal.kernel.util.ParamUtil;
import com.liferay.portal.kernel.util.Validator;

import java.util.ArrayList;
import java.util.List;

import javax.portlet.ActionRequest;
import javax.portlet.PortletURL;
import javax.portlet.RenderResponse;
import javax.portlet.RenderURL;

import javax.servlet.http.HttpServletRequest;

/**
 * @author Luca Pellizzon
 * @author Alessio Antonio Rendina
 */
public class CommerceInventoryDisplayContext {

	public CommerceInventoryDisplayContext(
		CommerceInventoryReplenishmentItemService
			commerceInventoryReplenishmentItemService,
		CommerceInventoryWarehouseService commerceInventoryWarehouseService,
		CommerceInventoryWarehouseItemService
			commerceInventoryWarehouseItemService,
		HttpServletRequest httpServletRequest) {

		_commerceInventoryReplenishmentItemService =
			commerceInventoryReplenishmentItemService;
		_commerceInventoryWarehouseService = commerceInventoryWarehouseService;
		_commerceInventoryWarehouseItemService =
			commerceInventoryWarehouseItemService;

		_cpRequestHelper = new CPRequestHelper(httpServletRequest);

		_sku = ParamUtil.getString(httpServletRequest, "sku");
	}

	public String getAddQuantityActionURL() throws Exception {
		LiferayPortletResponse liferayPortletResponse =
			_cpRequestHelper.getLiferayPortletResponse();

		PortletURL portletURL = liferayPortletResponse.createRenderURL();

		portletURL.setParameter(
			"mvcRenderCommandName", "editCommerceInventoryWarehouse");
		portletURL.setParameter("sku", _sku);

		portletURL.setWindowState(LiferayWindowState.POP_UP);

		return portletURL.toString();
	}

	public CommerceInventoryReplenishmentItem
			getCommerceInventoryReplenishmentItem()
		throws PortalException {

		long commerceInventoryReplenishmentItemId = ParamUtil.getLong(
			_cpRequestHelper.getRequest(),
			"commerceInventoryReplenishmentItemId");

		if (commerceInventoryReplenishmentItemId > 0) {
			return _commerceInventoryReplenishmentItemService.
				getCommerceInventoryReplenishmentItem(
					commerceInventoryReplenishmentItemId);
		}

		return null;
	}

	public long getCommerceInventoryReplenishmentItemId()
		throws PortalException {

		CommerceInventoryReplenishmentItem commerceInventoryReplenishmentItem =
			getCommerceInventoryReplenishmentItem();

		if (commerceInventoryReplenishmentItem == null) {
			return 0;
		}

		return commerceInventoryReplenishmentItem.
			getCommerceInventoryReplenishmentItemId();
	}

	public CommerceInventoryWarehouseItem getCommerceInventoryWarehouseItem()
		throws PortalException {

		long commerceInventoryWarehouseItemId = ParamUtil.getLong(
			_cpRequestHelper.getRequest(), "commerceInventoryWarehouseItemId");

		if (commerceInventoryWarehouseItemId > 0) {
			return _commerceInventoryWarehouseItemService.
				getCommerceInventoryWarehouseItem(
					commerceInventoryWarehouseItemId);
		}

		return null;
	}

	public long getCommerceInventoryWarehouseItemId() throws PortalException {
		CommerceInventoryWarehouseItem commerceInventoryWarehouseItem =
			getCommerceInventoryWarehouseItem();

		if (commerceInventoryWarehouseItem == null) {
			return 0;
		}

		return commerceInventoryWarehouseItem.
			getCommerceInventoryWarehouseItemId();
	}

	public List<CommerceInventoryWarehouse> getCommerceInventoryWarehouses()
		throws PrincipalException {

		long companyId = _cpRequestHelper.getCompanyId();

		return _commerceInventoryWarehouseService.
			getCommerceInventoryWarehouses(
				companyId, true, QueryUtil.ALL_POS, QueryUtil.ALL_POS, null);
	}

	public String getCreateInventoryItemActionURL() throws Exception {
		LiferayPortletResponse liferayPortletResponse =
			_cpRequestHelper.getLiferayPortletResponse();

		PortletURL portletURL = liferayPortletResponse.createRenderURL();

		portletURL.setParameter(
			"mvcRenderCommandName", "addCommerceInventoryWarehouseItem");

		portletURL.setWindowState(LiferayWindowState.POP_UP);

		return portletURL.toString();
	}

	public String getCreateReplenishmentActionURL() throws Exception {
		LiferayPortletResponse liferayPortletResponse =
			_cpRequestHelper.getLiferayPortletResponse();

		PortletURL portletURL = liferayPortletResponse.createRenderURL();

		portletURL.setParameter(
			"mvcRenderCommandName", "editCommerceInventoryReplenishmentItem");
		portletURL.setParameter("sku", _sku);

		portletURL.setWindowState(LiferayWindowState.POP_UP);

		return portletURL.toString();
	}

	public List<HeaderActionModel> getHeaderActionModels() {
		List<HeaderActionModel> headerActionModels = new ArrayList<>();

		if (_sku == null) {
			return headerActionModels;
		}

		if (PortalPermissionUtil.contains(
				getPermissionChecker(),
				CommerceInventoryActionKeys.MANAGE_INVENTORY)) {

			RenderResponse renderResponse =
				_cpRequestHelper.getRenderResponse();

			RenderURL cancelURL = renderResponse.createRenderURL();

			headerActionModels.add(
				new HeaderActionModel(
					null, cancelURL.toString(), null, "cancel"));
		}

		return headerActionModels;
	}

	public ClayCreationMenu getInventoryItemClayCreationMenu()
		throws Exception {

		ClayCreationMenu clayCreationMenu = new ClayCreationMenu();

		if (PortalPermissionUtil.contains(
				getPermissionChecker(),
				CommerceInventoryActionKeys.MANAGE_INVENTORY)) {

			clayCreationMenu.addClayCreationMenuActionItem(
				new ClayCreationMenuActionItem(
					getCreateInventoryItemActionURL(),
					LanguageUtil.get(
						_cpRequestHelper.getRequest(), "add-inventory-item"),
					ClayCreationMenuActionItem.
						CLAY_MENU_ACTION_ITEM_TARGET_MODAL));
		}

		return clayCreationMenu;
	}

	public PortletURL getPortletURL() {
		LiferayPortletResponse liferayPortletResponse =
			_cpRequestHelper.getLiferayPortletResponse();

		PortletURL portletURL = liferayPortletResponse.createRenderURL();

		String redirect = ParamUtil.getString(
			_cpRequestHelper.getRequest(), "redirect");

		if (Validator.isNotNull(redirect)) {
			portletURL.setParameter("redirect", redirect);
		}

		if (_sku != null) {
			portletURL.setParameter("sku", _sku);
		}

		return portletURL;
	}

	public ClayCreationMenu getReplenishmentClayCreationMenu()
		throws Exception {

		ClayCreationMenu clayCreationMenu = new ClayCreationMenu();

		if (PortalPermissionUtil.contains(
				getPermissionChecker(),
				CommerceInventoryActionKeys.MANAGE_INVENTORY)) {

			clayCreationMenu.addClayCreationMenuActionItem(
				new ClayCreationMenuActionItem(
					getCreateReplenishmentActionURL(),
					LanguageUtil.get(
						_cpRequestHelper.getRequest(), "add-income"),
					ClayCreationMenuActionItem.
						CLAY_MENU_ACTION_ITEM_TARGET_MODAL));
		}

		return clayCreationMenu;
	}

	public String getSku() {
		return _sku;
	}

	public String getTransferQuantitiesActionURL() throws Exception {
		LiferayPortletResponse liferayPortletResponse =
			_cpRequestHelper.getLiferayPortletResponse();

		PortletURL portletURL = liferayPortletResponse.createRenderURL();

		portletURL.setParameter("mvcRenderCommandName", "transferQuantities");
		portletURL.setParameter("sku", _sku);

		portletURL.setWindowState(LiferayWindowState.POP_UP);

		return portletURL.toString();
	}

	public PortletURL getTransitionInventoryPortletURL() {
		LiferayPortletResponse liferayPortletResponse =
			_cpRequestHelper.getLiferayPortletResponse();

		PortletURL portletURL = liferayPortletResponse.createActionURL();

		portletURL.setParameter(
			ActionRequest.ACTION_NAME, "editCommerceInventoryItem");
		portletURL.setParameter(Constants.CMD, "transition");
		portletURL.setParameter("sku", _sku);
		portletURL.setParameter("redirect", _cpRequestHelper.getCurrentURL());

		return portletURL;
	}

	public ClayCreationMenu getWarehousesClayCreationMenu() throws Exception {
		ClayCreationMenu clayCreationMenu = new ClayCreationMenu();

		if (PortalPermissionUtil.contains(
				getPermissionChecker(),
				CommerceInventoryActionKeys.MANAGE_INVENTORY)) {

			clayCreationMenu.addClayCreationMenuActionItem(
				new ClayCreationMenuActionItem(
					getAddQuantityActionURL(),
					LanguageUtil.get(
						_cpRequestHelper.getRequest(), "add-quantity"),
					ClayCreationMenuActionItem.
						CLAY_MENU_ACTION_ITEM_TARGET_MODAL));

			clayCreationMenu.addClayCreationMenuActionItem(
				new ClayCreationMenuActionItem(
					getTransferQuantitiesActionURL(),
					LanguageUtil.get(
						_cpRequestHelper.getRequest(), "transfer-quantity"),
					ClayCreationMenuActionItem.
						CLAY_MENU_ACTION_ITEM_TARGET_MODAL));
		}

		return clayCreationMenu;
	}

	private final CommerceInventoryReplenishmentItemService
		_commerceInventoryReplenishmentItemService;
	private final CommerceInventoryWarehouseItemService
		_commerceInventoryWarehouseItemService;
	private final CommerceInventoryWarehouseService
		_commerceInventoryWarehouseService;
	private final CPRequestHelper _cpRequestHelper;
	private String _sku;

}